# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Mobius Forensic Toolkit
# Copyright (C) 2008 Eduardo Aguiar
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
import os
import gtk
import mobius
import mobius.config
import mobius.ui.case_treeview
import mobius.ui.add_item_dialog

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Case properties dialog
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class CasePropertiesDialog (gtk.Dialog):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Build widget
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def __init__ (self, case):
    gtk.Dialog.__init__ (self, mobius.config.APP_TITLE, None, gtk.DIALOG_MODAL,
       (gtk.STOCK_OK, gtk.RESPONSE_OK, gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL))
    self.set_position (gtk.WIN_POS_CENTER)
    self.set_default_size (580, 480)
    self.set_type_hint (gtk.gdk.WINDOW_TYPE_HINT_DIALOG)
    self.set_border_width (10)

    table = gtk.Table (5, 2)
    table.set_border_width (10)
    table.set_col_spacings (5)
    table.set_row_spacings (5)
    table.show ()
    self.vbox.pack_start (table)

    icon = mobius.mediator.call ('ui.render-icon', case.icon_data, 32, 32)
    image = gtk.Image ()
    image.set_from_pixbuf (icon)
    image.show ()
    table.attach (image, 0, 1, 0, 1, 0, 0)

    label = gtk.Label ()
    label.set_markup ('<b>Case Properties</b>')
    label.set_alignment (0, -1)
    label.show ()
    table.attach (label, 1, 2, 0, 1, gtk.FILL | gtk.EXPAND, 0)

    label = gtk.Label ('ID')
    label.set_alignment (1, -1)
    label.show ()
    table.attach (label, 0, 1, 1, 2, 0, 0)

    self.case_id_entry = gtk.Entry ()
    self.case_id_entry.set_text (case.get_attribute ('id'))
    self.case_id_entry.show ()
    table.attach (self.case_id_entry, 1, 2, 1, 2, gtk.FILL | gtk.EXPAND, 0)

    label = gtk.Label ('Name')
    label.set_alignment (1, -1)
    label.show ()
    table.attach (label, 0, 1, 2, 3, 0, 0)

    self.case_name_entry = gtk.Entry ()
    self.case_name_entry.set_text (case.get_attribute ('name'))
    self.case_name_entry.show ()
    table.attach (self.case_name_entry, 1, 2, 2, 3, gtk.FILL | gtk.EXPAND, 0)

    label = gtk.Label ('Base dir')
    label.set_alignment (1, -1)
    label.show ()
    table.attach (label, 0, 1, 3, 4, 0, 0)

    hbox = gtk.HBox ()
    hbox.set_spacing (5)
    hbox.show ()
    table.attach (hbox, 1, 2, 3, 4, gtk.FILL | gtk.EXPAND, 0)

    self.case_rootdir_entry = gtk.Entry ()
    self.case_rootdir_entry.set_text (case.get_rootdir () or os.getcwd ())
    self.case_rootdir_entry.show ()
    hbox.pack_start (self.case_rootdir_entry)

    button = gtk.Button (stock=gtk.STOCK_OPEN)
    button.connect ('clicked', self.on_folder_choose)
    button.show ()
    hbox.pack_end (button, False, False)

    label = gtk.Label ('Description')
    label.set_alignment (1, -1)
    label.show ()
    table.attach (label, 0, 1, 4, 5, 0, 0)

    frame = gtk.Frame ()
    frame.show ()
    table.attach (frame, 1, 2, 4, 5)

    sw = gtk.ScrolledWindow ()
    sw.set_policy (gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
    sw.show ()
    frame.add (sw)

    self.case_description_textview = gtk.TextView ()
    self.case_description_textview.set_wrap_mode (gtk.WRAP_WORD)
    self.case_description_textview.show ()
    sw.add (self.case_description_textview)

    buffer = self.case_description_textview.get_buffer ()
    buffer.set_text (case.get_attribute ('description') or '')
    self.case = case

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Run dialog
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def run (self):
    rc = gtk.Dialog.run (self)

    # set new values, if OK
    if rc == gtk.RESPONSE_OK:

      # change id
      id = self.case.get_attribute ('id')
      new_id = self.case_id_entry.get_text ().decode ('utf-8')

      if new_id != id:
        self.case.set_attribute ('id', new_id)
        mobius.mediator.emit ('case.attribute-modified', self.case, 'id', id, new_id)

      # change name
      name = self.case.get_attribute ('name')
      new_name = self.case_name_entry.get_text ().decode ('utf-8')

      if new_name != name:
        self.case.set_attribute ('name', new_name)
        mobius.mediator.emit ('case.attribute-modified', self.case, 'name', name, new_name)

      # change description
      description = self.case.get_attribute ('description')
      buffer = self.case_description_textview.get_buffer ()
      start, end = buffer.get_bounds ()
      new_description = buffer.get_text (start, end).decode ('utf-8').strip ()

      if new_description != description:
        self.case.set_attribute ('description', new_description)
        mobius.mediator.emit ('case.attribute-modified', self.case, 'description', description, new_description)

      # change rootdir
      rootdir = self.case.get_rootdir ()
      new_rootdir = self.case_rootdir_entry.get_text ().decode ('utf-8')

      if new_rootdir != rootdir:
        self.case.set_rootdir (new_rootdir)
        mobius.mediator.emit ('case.attribute-modified', self.case, 'rootdir', rootdir, new_rootdir)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief On folder choose
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_folder_choose (self, widget, *args):
    dialog = gtk.FileChooserDialog ('Select case rootdir', self,
         gtk.FILE_CHOOSER_ACTION_CREATE_FOLDER,
        (gtk.STOCK_OK, gtk.RESPONSE_OK, gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL))
    dialog.set_current_folder (self.case_rootdir_entry.get_text ())

    rc = dialog.run ()

    if rc == gtk.RESPONSE_OK:
      self.case_rootdir_entry.set_text (dialog.get_filename ())

    dialog.destroy ()

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Case window
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Window (gtk.Window):

  def __init__ (self, *args):
    self.case = None

    gtk.Window.__init__ (self, *args)
    self.connect ('delete-event', self.on_file_close)
    self.set_default_size (600, 400)
    self.set_title ('Mobius v%s' % mobius.config.APP_VERSION)

    # accel_group
    accel_group = gtk.AccelGroup ()
    self.add_accel_group (accel_group)

    # vbox
    vbox = gtk.VBox (False, 1)
    vbox.set_border_width (1)
    self.add (vbox)
    vbox.show ()

    # menubar
    menubar = gtk.MenuBar ()
    menubar.show ()
    vbox.pack_start (menubar, False, False)

    item = gtk.MenuItem ('_File')
    item.show ()
    menubar.append (item)

    menu = gtk.Menu ()
    menu.show ()
    item.set_submenu (menu)

    self.save_menuitem = gtk.ImageMenuItem (gtk.STOCK_SAVE, accel_group)
    self.save_menuitem.connect ("activate", self.on_file_save)
    self.save_menuitem.set_sensitive (False)
    self.save_menuitem.show ()
    menu.append (self.save_menuitem)

    item = gtk.ImageMenuItem (gtk.STOCK_SAVE_AS, accel_group)
    item.connect ("activate", self.on_file_save_as)
    item.show ()
    menu.append (item)

    item = gtk.ImageMenuItem (gtk.STOCK_CLOSE, accel_group)
    item.connect ("activate", self.on_file_close)
    item.show ()
    menu.append (item)

    item = gtk.SeparatorMenuItem ()
    item.show ()
    menu.append (item)

    item = gtk.ImageMenuItem (gtk.STOCK_PROPERTIES, accel_group)
    item.connect ("activate", self.on_file_properties)
    item.show ()
    menu.append (item)

    # toolbar
    self.tooltips = gtk.Tooltips ()

    toolbar = gtk.Toolbar ()
    toolbar.set_style (gtk.TOOLBAR_ICONS)
    toolbar.set_tooltips (True)
    toolbar.show ()
    vbox.pack_start (toolbar, False, False)

    self.save_toolitem = gtk.ToolButton (gtk.STOCK_SAVE)
    self.save_toolitem.set_sensitive (False)
    self.save_toolitem.connect ("clicked", self.on_file_save)
    self.save_toolitem.show ()
    self.save_toolitem.set_tooltip (self.tooltips, "Save current case")
    toolbar.insert (self.save_toolitem, -1)

    item = gtk.ToolButton (gtk.STOCK_SAVE_AS)
    item.connect ("clicked", self.on_file_save_as)
    item.show ()
    item.set_tooltip (self.tooltips, "Save case as")
    toolbar.insert (item, -1)

    toolitem = gtk.SeparatorToolItem ()
    toolitem.show ()
    toolbar.insert (toolitem, -1)

    self.add_toolitem = gtk.ToolButton (gtk.STOCK_ADD)
    self.add_toolitem.set_sensitive (False)
    self.add_toolitem.connect ("clicked", self.on_add_item)
    self.add_toolitem.show ()
    self.add_toolitem.set_tooltip (self.tooltips, "Add item to case")
    toolbar.insert (self.add_toolitem, -1)

    self.remove_toolitem = gtk.ToolButton (gtk.STOCK_REMOVE)
    self.remove_toolitem.set_sensitive (False)
    self.remove_toolitem.connect ("clicked", self.on_remove_item)
    self.remove_toolitem.show ()
    self.remove_toolitem.set_tooltip (self.tooltips, "Remove item from case")
    toolbar.insert (self.remove_toolitem, -1)

    toolitem = gtk.ToolButton (gtk.STOCK_PROPERTIES)
    toolitem.connect ("clicked", self.on_file_properties)
    toolitem.show ()
    toolitem.set_tooltip (self.tooltips, "Case properties")
    toolbar.insert (toolitem, -1)

    # case treeview
    frame = gtk.Frame ()
    frame.show ()
    vbox.pack_start (frame)

    sw = gtk.ScrolledWindow ()
    sw.set_policy (gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
    sw.show ()
    frame.add (sw)

    self.case_treeview = mobius.ui.case_treeview.CaseTreeView ()
    self.case_treeview.show ()
    sw.add (self.case_treeview)

    # status bar
    frame = gtk.Frame ()
    frame.set_shadow_type (gtk.SHADOW_IN)
    frame.show ()
    vbox.pack_end (frame, False, False)

    self.status_label = gtk.Label ()
    self.status_label.set_alignment (0, -1)
    self.status_label.show ()
    frame.add (self.status_label)

    self.mediator = mobius.mediator.new_client_mediator ()
    self.mediator.connect ('case-modified', self.on_case_modified)
    self.mediator.connect ('case.attribute-modified', self.on_case_attribute_modified)
    self.mediator.connect ('case-selected', self.on_case_selected)
    self.mediator.connect ('item-selected', self.on_item_selected)
    self.mediator.connect ('item.attribute-modified', self.on_attribute_modified)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief update title
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def update_title (self):
    title = '%s v%s - Case manager' % (mobius.config.APP_NAME, mobius.config.APP_VERSION)
    if self.case:
      title += ' [%s]' % self.case.get_attribute ('name')
      if self.case.is_modified ():
        title += '*'

    self.set_title (title)
    self.save_menuitem.set_sensitive (self.case.is_modified ())
    self.save_toolitem.set_sensitive (self.case.is_modified ())

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief case attribute modified
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_case_attribute_modified (self, case, attr, old_value, new_value):
    case.set_modified (True)
    self.update_title ()

    if attr == 'name' and case.uid == self.case.uid:
      self.case_treeview.set_case_name (new_value)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief case modified
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_case_modified (self, case):
    case.set_modified (True)
    self.update_title ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle item selection
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_item_selected (self, item):
    self.add_toolitem.set_sensitive (True)
    self.remove_toolitem.set_sensitive (True)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle attribute modified
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_attribute_modified (self, item, id, old_text, new_text):
    item.case.set_modified (True)
    self.update_title ()

    name = item.get_attribute ('name')
    self.case_treeview.set_selected_item_name (name)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle case selection
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_case_selected (self, case):
    self.add_toolitem.set_sensitive (True)
    self.remove_toolitem.set_sensitive (False)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief close case
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_file_close (self, widget, *args):

    # show 'save changes' dialog if necessary
    if self.case.is_modified ():
      dialog = gtk.MessageDialog (self,
                      gtk.DIALOG_MODAL,
                      gtk.MESSAGE_QUESTION,
                      gtk.BUTTONS_YES_NO,
                      "Save changes to '%s'?" % self.case.filename)
      dialog.add_button (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL)
      rc = dialog.run ()
      dialog.destroy ()

      if rc == gtk.RESPONSE_CANCEL:
        return True

      elif rc == gtk.RESPONSE_YES:
        mobius.mediator.call ('case.save', self.case)

    mobius.mediator.call ('case.close', self.case)

    # close window
    self.mediator.clear ()
    self.manager.remove_window (self)
    self.destroy ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief close case
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_file_save (self, widget, *args):

    if self.case.is_new ():
      self.on_file_save_as (widget, *args)
    else:
      mobius.mediator.call ('case.save', self.case)
      self.update_title ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief close case
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_file_save_as (self, widget, *args):

    # choose file
    fs = gtk.FileChooserDialog ('Save Case', parent=self, action=gtk.FILE_CHOOSER_ACTION_SAVE)
    fs.add_button (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL)
    fs.add_button (gtk.STOCK_OK, gtk.RESPONSE_OK)
    fs.set_do_overwrite_confirmation (True)

    filter = gtk.FileFilter ()
    filter.set_name ('Mobius case')
    filter.add_pattern ('*.case')
    fs.set_filter (filter)

    rc = fs.run ()
    filename = fs.get_filename ()
    fs.destroy ()

    if rc != gtk.RESPONSE_OK:
      return

    # save file
    root, ext = os.path.splitext (filename)
    if ext != '.case':
      filename += '.case'
    self.case.filename = filename

    mobius.mediator.call ('case.save', self.case)
    self.update_title ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief case properties
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_file_properties (self, widget, *args):
    dialog = CasePropertiesDialog (self.case)
    dialog.run ()
    dialog.destroy ()

    self.update_title ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief call add item dialog
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_add_item (self, widget, *args):
    dialog = mobius.ui.add_item_dialog.Dialog ()
    rc = dialog.run ()

    amount = dialog.get_amount ()
    category = dialog.get_category ()
    dialog.destroy ()

    if rc != gtk.RESPONSE_OK:
      return

    for i in range (amount):
      item = self.case.create_item (category)
      self.case_treeview.add_child (item)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief call remove item dialog
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_remove_item (self, widget, *args):
    item = self.case_treeview.get_selected_item ()

    title = 'You are about to remove an item'
    if item.has_child ():
      title += ' and its sub-items'
    title += '. Are you sure?'
    
    dialog = gtk.MessageDialog (self,
                    gtk.DIALOG_MODAL,
                    gtk.MESSAGE_QUESTION,
                    gtk.BUTTONS_YES_NO,
                    title)
    rc = dialog.run ()
    dialog.destroy ()

    if rc != gtk.RESPONSE_YES:
      return

    self.case_treeview.remove_selected_item ()
    item.parent.remove_child (item)
    self.remove_toolitem.set_sensitive (False)
