// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief  C++ API imagefile module wrapper
//!\author Eduardo Aguiar
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include <Python.h>
#include <datetime.h>
#include <mobius/imagefile/imagefile.h>
#include <mobius/io/reader.h>
#include <mobius/io/writer.h>
#include <python/mobius/io/io.h>

static mobius_io_capi *mobius_io_api = nullptr;

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief imagefile module methods
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyMethodDef imagefile_methods[] =
{
  {NULL, NULL, 0, NULL} // sentinel
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief imagefile.imagefile: data structure
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
typedef struct
{
  PyObject_HEAD
  mobius::imagefile::imagefile *obj;
} imagefile_imagefile_o;

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief imagefile.imagefile: prototypes
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static void imagefile_imagefile_tp_dealloc (imagefile_imagefile_o *);
static PyObject *imagefile_imagefile_tp_new (PyTypeObject *, PyObject *, PyObject *);
static PyObject *imagefile_imagefile_get_is_available (imagefile_imagefile_o *);
static PyObject *imagefile_imagefile_get_is_valid (imagefile_imagefile_o *);
static PyObject *imagefile_imagefile_get_url (imagefile_imagefile_o *);
static PyObject *imagefile_imagefile_get_type (imagefile_imagefile_o *);
static PyObject *imagefile_imagefile_get_size (imagefile_imagefile_o *);
static int imagefile_imagefile_set_size (imagefile_imagefile_o *, PyObject *, void *);
static PyObject *imagefile_imagefile_get_sectors (imagefile_imagefile_o *);
static int imagefile_imagefile_set_sectors (imagefile_imagefile_o *, PyObject *, void *);
static PyObject *imagefile_imagefile_get_sector_size (imagefile_imagefile_o *);
static int imagefile_imagefile_set_sector_size (imagefile_imagefile_o *, PyObject *, void *);
static PyObject *imagefile_imagefile_get_segments (imagefile_imagefile_o *);
static int imagefile_imagefile_set_segments (imagefile_imagefile_o *, PyObject *, void *);
static PyObject *imagefile_imagefile_get_segment_size (imagefile_imagefile_o *);
static int imagefile_imagefile_set_segment_size (imagefile_imagefile_o *, PyObject *, void *);
static PyObject *imagefile_imagefile_get_drive_vendor (imagefile_imagefile_o *);
static int imagefile_imagefile_set_drive_vendor (imagefile_imagefile_o *, PyObject *, void *);
static PyObject *imagefile_imagefile_get_drive_model (imagefile_imagefile_o *);
static int imagefile_imagefile_set_drive_model (imagefile_imagefile_o *, PyObject *, void *);
static PyObject *imagefile_imagefile_get_drive_serial_number (imagefile_imagefile_o *);
static int imagefile_imagefile_set_drive_serial_number (imagefile_imagefile_o *, PyObject *, void *);
static PyObject *imagefile_imagefile_get_acquisition_user (imagefile_imagefile_o *);
static int imagefile_imagefile_set_acquisition_user (imagefile_imagefile_o *, PyObject *, void *);
static PyObject *imagefile_imagefile_get_acquisition_time (imagefile_imagefile_o *);
static int imagefile_imagefile_set_acquisition_time (imagefile_imagefile_o *, PyObject *, void *);
static PyObject *imagefile_imagefile_get_acquisition_tool (imagefile_imagefile_o *);
static int imagefile_imagefile_set_acquisition_tool (imagefile_imagefile_o *, PyObject *, void *);
static PyObject *imagefile_imagefile_get_acquisition_platform (imagefile_imagefile_o *);
static int imagefile_imagefile_set_acquisition_platform (imagefile_imagefile_o *, PyObject *, void *);
static PyObject *imagefile_imagefile_get_hash_md5 (imagefile_imagefile_o *);
static int imagefile_imagefile_set_hash_md5 (imagefile_imagefile_o *, PyObject *, void *);
static PyObject *imagefile_imagefile_f_new_reader (imagefile_imagefile_o *, PyObject *);
static PyObject *imagefile_imagefile_f_new_writer (imagefile_imagefile_o *, PyObject *);

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief imagefile.imagefile: getters and setters structure
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyGetSetDef imagefile_imagefile_getsetters[] =
{
  {
    (char *) "is_available",
    (getter) imagefile_imagefile_get_is_available,
    (setter) 0,
    (char *) "check if imagefile is available", NULL
  },
  {
    (char *) "is_valid",
    (getter) imagefile_imagefile_get_is_valid,
    (setter) 0,
    (char *) "check if imagefile is valid", NULL
  },
  {
    (char *) "url",
    (getter) imagefile_imagefile_get_url,
    (setter) 0,
    (char *) "imagefile URL", NULL
  },
  {
    (char *) "type",
    (getter) imagefile_imagefile_get_type,
    (setter) 0,
    (char *) "imagefile type", NULL
  },
  {
    (char *) "size",
    (getter) imagefile_imagefile_get_size,
    (setter) imagefile_imagefile_set_size,
    (char *) "imagefile size in bytes", NULL
  },
  {
    (char *) "sectors",
    (getter) imagefile_imagefile_get_sectors,
    (setter) imagefile_imagefile_set_sectors,
    (char *) "number of sectors", NULL
  },
  {
    (char *) "sector_size",
    (getter) imagefile_imagefile_get_sector_size,
    (setter) imagefile_imagefile_set_sector_size,
    (char *) "sector size in bytes", NULL
  },
  {
    (char *) "segments",
    (getter) imagefile_imagefile_get_segments,
    (setter) imagefile_imagefile_set_segments,
    (char *) "number of file segments", NULL
  },
  {
    (char *) "segment_size",
    (getter) imagefile_imagefile_get_segment_size,
    (setter) imagefile_imagefile_set_segment_size,
    (char *) "segment size in bytes", NULL
  },
  {
    (char *) "drive_vendor",
    (getter) imagefile_imagefile_get_drive_vendor,
    (setter) imagefile_imagefile_set_drive_vendor,
    (char *) "drive vendor name", NULL
  },
  {
    (char *) "drive_model",
    (getter) imagefile_imagefile_get_drive_model,
    (setter) imagefile_imagefile_set_drive_model,
    (char *) "drive model", NULL
  },
  {
    (char *) "drive_serial_number",
    (getter) imagefile_imagefile_get_drive_serial_number,
    (setter) imagefile_imagefile_set_drive_serial_number,
    (char *) "drive serial number", NULL
  },
  {
    (char *) "acquisition_user",
    (getter) imagefile_imagefile_get_acquisition_user,
    (setter) imagefile_imagefile_set_acquisition_user,
    (char *) "acquisition user name", NULL
  },
  {
    (char *) "acquisition_time",
    (getter) imagefile_imagefile_get_acquisition_time,
    (setter) imagefile_imagefile_set_acquisition_time,
    (char *) "acquisition date and time", NULL
  },
  {
    (char *) "acquisition_tool",
    (getter) imagefile_imagefile_get_acquisition_tool,
    (setter) imagefile_imagefile_set_acquisition_tool,
    (char *) "acquisition tool", NULL
  },
  {
    (char *) "acquisition_platform",
    (getter) imagefile_imagefile_get_acquisition_platform,
    (setter) imagefile_imagefile_set_acquisition_platform,
    (char *) "acquisition platform", NULL
  },
  {
    (char *) "hash_md5",
    (getter) imagefile_imagefile_get_hash_md5,
    (setter) imagefile_imagefile_set_hash_md5,
    (char *) "MD5 hash", NULL
  },
  {NULL} // sentinel
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief imagefile.imagefile: methods structure
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyMethodDef imagefile_imagefile_methods[] =
{
  {
    (char *) "new_reader",
    (PyCFunction) imagefile_imagefile_f_new_reader,
    METH_VARARGS,
    "create new reader"
  },
  {
    (char *) "new_writer",
    (PyCFunction) imagefile_imagefile_f_new_writer,
    METH_VARARGS,
    "create new writer"
  },
  {NULL} // sentinel
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief imagefile.imagefile: type structure
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyTypeObject imagefile_imagefile_t =
{
  PyObject_HEAD_INIT (0)
  0,                                          		// ob_size
  "imagefile.imagefile",                      		// tp_name
  sizeof (imagefile_imagefile_o),             		// tp_basicsize
  0,                                          		// tp_itemsize
  (destructor) imagefile_imagefile_tp_dealloc,		// tp_dealloc
  0,                                          		// tp_print
  0,                                          		// tp_getattr
  0,                                          		// tp_setattr
  0,                                          		// tp_compare
  0,                                          		// tp_repr
  0,                                          		// tp_as_number
  0,                                          		// tp_as_sequence
  0,                                          		// tp_as_mapping
  0,                                          		// tp_hash
  0,                                          		// tp_call
  0,                                          		// tp_str
  0,                                          		// tp_getattro
  0,                                          		// tp_setattro
  0,                                          		// tp_as_buffer
  Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE,   		// tp_flags
  "imagefile class",                          		// tp_doc
  0,                                          		// tp_traverse
  0,                                          		// tp_clear
  0,                                          		// tp_richcompare
  0,                                          		// tp_weaklistoffset
  0,                                          		// tp_iter
  0,                                          		// tp_iternext
  imagefile_imagefile_methods,                		// tp_methods
  0,                                          		// tp_members
  imagefile_imagefile_getsetters,             		// tp_getset
  0,                                          		// tp_base
  0,                                          		// tp_dict
  0,                                          		// tp_descr_get
  0,                                          		// tp_descr_set
  0,                                          		// tp_dictoffset
  0,                                          		// tp_init
  0,                                          		// tp_alloc
  imagefile_imagefile_tp_new                  		// tp_new
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief imagefile.imagefile: tp_new
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
imagefile_imagefile_tp_new (PyTypeObject *type, PyObject *args, PyObject *kwds)
{
  const char *arg_url = nullptr;
  const char *arg_type = "autodetect";

  if (!PyArg_ParseTuple (args, "|ss", &arg_url, &arg_type))
    return nullptr;

  imagefile_imagefile_o *self = (imagefile_imagefile_o *) type->tp_alloc (type, 0);
  if (self != nullptr)
    {
      try
        {
          if (arg_url)
            self->obj = new mobius::imagefile::imagefile (arg_url, arg_type);
          else
            self->obj = new mobius::imagefile::imagefile ();
        }
      catch (const std::runtime_error& e)
        {
          Py_DECREF (self);
          PyErr_SetString (PyExc_IOError, e.what ());
          self = nullptr;
        }
    }

  return (PyObject *) self;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief imagefile.imagefile: tp_dealloc
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static void
imagefile_imagefile_tp_dealloc (imagefile_imagefile_o *self)
{
  delete self->obj;
  self->ob_type->tp_free ((PyObject*) self);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief imagefile.imagefile: is_available getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
imagefile_imagefile_get_is_available (imagefile_imagefile_o *self)
{
  return PyBool_FromLong (self->obj->is_available ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief imagefile.imagefile: is_valid getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
imagefile_imagefile_get_is_valid (imagefile_imagefile_o *self)
{
  return PyBool_FromLong (self->obj->operator bool ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief imagefile.imagefile: url getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
imagefile_imagefile_get_url (imagefile_imagefile_o *self)
{
  return PyString_FromString (self->obj->get_url ().c_str ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief imagefile.imagefile: type getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
imagefile_imagefile_get_type (imagefile_imagefile_o *self)
{
  return PyString_FromString (self->obj->get_type ().c_str ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief imagefile.imagefile: size getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
imagefile_imagefile_get_size (imagefile_imagefile_o *self)
{
  return PyInt_FromLong (self->obj->get_size ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief imagefile.imagefile: size setter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static int
imagefile_imagefile_set_size (imagefile_imagefile_o *self, PyObject *value, void *closure)
{
  if (value == nullptr)
    {
      PyErr_SetString (PyExc_TypeError, "cannot delete 'size' attribute");
      return -1;
    }

  if (!PyInt_Check (value) && !PyLong_Check (value))
    {
      PyErr_SetString (PyExc_TypeError, "invalid type for 'size' attribute");
      return -1;
    }

  // set size
  std::uint64_t arg_size = PyInt_AsUnsignedLongLongMask (value);
  self->obj->set_size (arg_size);

  return 0;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief imagefile.imagefile: sectors getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
imagefile_imagefile_get_sectors (imagefile_imagefile_o *self)
{
  return PyInt_FromLong (self->obj->get_sectors ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief imagefile.imagefile: sectors setter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static int
imagefile_imagefile_set_sectors (imagefile_imagefile_o *self, PyObject *value, void *closure)
{
  if (value == nullptr)
    {
      PyErr_SetString (PyExc_TypeError, "cannot delete 'sectors' attribute");
      return -1;
    }

  if (!PyInt_Check (value) && !PyLong_Check (value))
    {
      PyErr_SetString (PyExc_TypeError, "invalid type for 'sectors' attribute");
      return -1;
    }

  // set sectors
  std::uint64_t arg_sectors = PyInt_AsUnsignedLongLongMask (value);
  self->obj->set_sectors (arg_sectors);

  return 0;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief imagefile.imagefile: sector_size getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
imagefile_imagefile_get_sector_size (imagefile_imagefile_o *self)
{
  return PyInt_FromLong (self->obj->get_sector_size ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief imagefile.imagefile: sector_size setter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static int
imagefile_imagefile_set_sector_size (imagefile_imagefile_o *self, PyObject *value, void *closure)
{
  if (value == nullptr)
    {
      PyErr_SetString (PyExc_TypeError, "cannot delete 'sector_size' attribute");
      return -1;
    }

  if (!PyInt_Check (value) && !PyLong_Check (value))
    {
      PyErr_SetString (PyExc_TypeError, "invalid type for 'sector_size' attribute");
      return -1;
    }

  // set sector_size
  std::uint64_t arg_sector_size = PyInt_AsUnsignedLongLongMask (value);
  self->obj->set_sector_size (arg_sector_size);

  return 0;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief imagefile.imagefile: segments getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
imagefile_imagefile_get_segments (imagefile_imagefile_o *self)
{
  return PyInt_FromLong (self->obj->get_segments ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief imagefile.imagefile: segments setter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static int
imagefile_imagefile_set_segments (imagefile_imagefile_o *self, PyObject *value, void *closure)
{
  if (value == nullptr)
    {
      PyErr_SetString (PyExc_TypeError, "cannot delete 'segments' attribute");
      return -1;
    }

  if (!PyInt_Check (value) && !PyLong_Check (value))
    {
      PyErr_SetString (PyExc_TypeError, "invalid type for 'segments' attribute");
      return -1;
    }

  // set segments
  std::uint64_t arg_segments = PyInt_AsUnsignedLongLongMask (value);
  self->obj->set_segments (arg_segments);

  return 0;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief imagefile.imagefile: segment_size getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
imagefile_imagefile_get_segment_size (imagefile_imagefile_o *self)
{
  return PyInt_FromLong (self->obj->get_segment_size ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief imagefile.imagefile: segment_size setter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static int
imagefile_imagefile_set_segment_size (imagefile_imagefile_o *self, PyObject *value, void *closure)
{
  if (value == nullptr)
    {
      PyErr_SetString (PyExc_TypeError, "cannot delete 'segment_size' attribute");
      return -1;
    }

  if (!PyInt_Check (value) && !PyLong_Check (value))
    {
      PyErr_SetString (PyExc_TypeError, "invalid type for 'segment_size' attribute");
      return -1;
    }

  // set segment_size
  std::uint64_t arg_segment_size = PyInt_AsUnsignedLongLongMask (value);
  self->obj->set_segment_size (arg_segment_size);

  return 0;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief imagefile.imagefile: drive_vendor getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
imagefile_imagefile_get_drive_vendor (imagefile_imagefile_o *self)
{
  return PyString_FromString (self->obj->get_drive_vendor ().c_str ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief imagefile.imagefile: drive_vendor setter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static int
imagefile_imagefile_set_drive_vendor (imagefile_imagefile_o *self, PyObject *value, void *closure)
{
  if (value == nullptr)
    {
      PyErr_SetString (PyExc_TypeError, "cannot delete 'drive_vendor' attribute");
      return -1;
    }

  if (!PyString_Check (value))
    {
      PyErr_SetString (PyExc_TypeError, "invalid type for 'drive_vendor' attribute");
      return -1;
    }

  // set drive_model
  const char *arg_drive_vendor = PyString_AsString (value);
  self->obj->set_drive_vendor (arg_drive_vendor);

  return 0;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief imagefile.imagefile: drive_model getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
imagefile_imagefile_get_drive_model (imagefile_imagefile_o *self)
{
  return PyString_FromString (self->obj->get_drive_model ().c_str ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief imagefile.imagefile: drive_model setter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static int
imagefile_imagefile_set_drive_model (imagefile_imagefile_o *self, PyObject *value, void *closure)
{
  if (value == nullptr)
    {
      PyErr_SetString (PyExc_TypeError, "cannot delete 'drive_model' attribute");
      return -1;
    }

  if (!PyString_Check (value))
    {
      PyErr_SetString (PyExc_TypeError, "invalid type for 'drive_model' attribute");
      return -1;
    }

  // set drive_model
  const char *arg_drive_model = PyString_AsString (value);
  self->obj->set_drive_model (arg_drive_model);

  return 0;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief imagefile.imagefile: drive_serial_number getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
imagefile_imagefile_get_drive_serial_number (imagefile_imagefile_o *self)
{
  return PyString_FromString (self->obj->get_drive_serial_number ().c_str ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief imagefile.imagefile: drive_serial_number setter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static int
imagefile_imagefile_set_drive_serial_number (imagefile_imagefile_o *self, PyObject *value, void *closure)
{
  if (value == nullptr)
    {
      PyErr_SetString (PyExc_TypeError, "cannot delete 'drive_serial_number' attribute");
      return -1;
    }

  if (!PyString_Check (value))
    {
      PyErr_SetString (PyExc_TypeError, "invalid type for 'drive_serial_number' attribute");
      return -1;
    }

  // set drive_serial_number
  const char *arg_drive_serial_number = PyString_AsString (value);
  self->obj->set_drive_serial_number (arg_drive_serial_number);

  return 0;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief imagefile.imagefile: acquisition_user getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
imagefile_imagefile_get_acquisition_user (imagefile_imagefile_o *self)
{
  return PyString_FromString (self->obj->get_acquisition_user ().c_str ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief imagefile.imagefile: acquisition_user setter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static int
imagefile_imagefile_set_acquisition_user (imagefile_imagefile_o *self, PyObject *value, void *closure)
{
  if (value == nullptr)
    {
      PyErr_SetString (PyExc_TypeError, "cannot delete 'acquisition_user' attribute");
      return -1;
    }

  if (!PyString_Check (value))
    {
      PyErr_SetString (PyExc_TypeError, "invalid type for 'acquisition_user' attribute");
      return -1;
    }

  // set acquisition_user
  const char *arg_acquisition_user = PyString_AsString (value);
  self->obj->set_acquisition_user (arg_acquisition_user);

  return 0;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief imagefile.imagefile: acquisition_time getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
imagefile_imagefile_get_acquisition_time (imagefile_imagefile_o *self)
{
  auto ret_value = self->obj->get_acquisition_time ();
  auto ret_date = ret_value.get_date ();
  auto ret_time = ret_value.get_time ();

  return PyDateTime_FromDateAndTime (ret_date.get_year (), ret_date.get_month (), ret_date.get_day (), ret_time.get_hour (), ret_time.get_minute (), ret_time.get_second (), 0);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief imagefile.imagefile: acquisition_time setter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static int
imagefile_imagefile_set_acquisition_time (imagefile_imagefile_o *self, PyObject *value, void *closure)
{
  if (value == nullptr)
    {
      PyErr_SetString (PyExc_TypeError, "cannot delete 'acquisition_time' attribute");
      return -1;
    }

  if (!PyDateTime_Check (value))
    {
      PyErr_SetString (PyExc_TypeError, "invalid type for 'acquisition_time' attribute");
      return -1;
    }

  // set acquisition_time
  mobius::datetime::datetime arg_acquisition_time (
    PyDateTime_GET_YEAR (value),
    PyDateTime_GET_MONTH (value),
    PyDateTime_GET_DAY (value),
    PyDateTime_DATE_GET_HOUR (value),
    PyDateTime_DATE_GET_MINUTE (value),
    PyDateTime_DATE_GET_SECOND (value));
  self->obj->set_acquisition_time (arg_acquisition_time);

  return 0;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief imagefile.imagefile: acquisition_tool getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
imagefile_imagefile_get_acquisition_tool (imagefile_imagefile_o *self)
{
  return PyString_FromString (self->obj->get_acquisition_tool ().c_str ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief imagefile.imagefile: acquisition_tool setter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static int
imagefile_imagefile_set_acquisition_tool (imagefile_imagefile_o *self, PyObject *value, void *closure)
{
  if (value == nullptr)
    {
      PyErr_SetString (PyExc_TypeError, "cannot delete 'acquisition_tool' attribute");
      return -1;
    }

  if (!PyString_Check (value))
    {
      PyErr_SetString (PyExc_TypeError, "invalid type for 'acquisition_tool' attribute");
      return -1;
    }

  // set acquisition_tool
  const char *arg_acquisition_tool = PyString_AsString (value);
  self->obj->set_acquisition_tool (arg_acquisition_tool);

  return 0;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief imagefile.imagefile: acquisition_platform getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
imagefile_imagefile_get_acquisition_platform (imagefile_imagefile_o *self)
{
  return PyString_FromString (self->obj->get_acquisition_platform ().c_str ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief imagefile.imagefile: acquisition_platform setter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static int
imagefile_imagefile_set_acquisition_platform (imagefile_imagefile_o *self, PyObject *value, void *closure)
{
  if (value == nullptr)
    {
      PyErr_SetString (PyExc_TypeError, "cannot delete 'acquisition_platform' attribute");
      return -1;
    }

  if (!PyString_Check (value))
    {
      PyErr_SetString (PyExc_TypeError, "invalid type for 'acquisition_platform' attribute");
      return -1;
    }

  // set acquisition_platform
  const char *arg_acquisition_platform = PyString_AsString (value);
  self->obj->set_acquisition_platform (arg_acquisition_platform);

  return 0;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief imagefile.imagefile: hash_md5 getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
imagefile_imagefile_get_hash_md5 (imagefile_imagefile_o *self)
{
  return PyString_FromString (self->obj->get_hash_md5 ().c_str ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief imagefile.imagefile: hash_md5 setter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static int
imagefile_imagefile_set_hash_md5 (imagefile_imagefile_o *self, PyObject *value, void *closure)
{
  if (value == nullptr)
    {
      PyErr_SetString (PyExc_TypeError, "cannot delete 'hash_md5' attribute");
      return -1;
    }

  if (!PyString_Check (value))
    {
      PyErr_SetString (PyExc_TypeError, "invalid type for 'hash_md5' attribute");
      return -1;
    }

  // set hash_md5
  const char *arg_hash_md5 = PyString_AsString (value);
  self->obj->set_hash_md5 (arg_hash_md5);

  return 0;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief imagefile.imagefile: new_reader
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
imagefile_imagefile_f_new_reader (imagefile_imagefile_o *self, PyObject *args)
{
  io_reader_o *ret = nullptr;

  try
    {
      ret = mobius_io_api->io_reader_tp_alloc ();
      if (ret != nullptr)
        ret->obj = new mobius::io::reader (self->obj->new_reader ());
    }
  catch (const std::runtime_error& e)
    {
      PyErr_SetString (PyExc_IOError, e.what ());
    }

  return (PyObject *) ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief imagefile.imagefile: new_writer
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
imagefile_imagefile_f_new_writer (imagefile_imagefile_o *self, PyObject *args)
{
  io_writer_o *ret = nullptr;

  try
    {
      ret = mobius_io_api->io_writer_tp_alloc ();
      if (ret != nullptr)
        ret->obj = new mobius::io::writer (self->obj->new_writer ());
    }
  catch (const std::runtime_error& e)
    {
      PyErr_SetString (PyExc_IOError, e.what ());
    }

  return (PyObject *) ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief imagefile module initialisation function
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
PyMODINIT_FUNC
initimagefile ()
{
  if (PyType_Ready (&imagefile_imagefile_t) < 0)
    return;

  PyObject* module = Py_InitModule3 (
    "imagefile",
    imagefile_methods,
    "Mobius Forensic Toolkit API wrapper"
  );

  if (module == nullptr)
    return;

  PyDateTime_IMPORT;
  
  // import mobius.io module
  PyObject *io_module = PyImport_ImportModule ("mobius.io");
  if (io_module == nullptr)
    return;

  mobius_io_api = (mobius_io_capi *) PyCapsule_Import ("mobius.io.capi", 0);
  if (mobius_io_api == nullptr)
    return;

  // add types
  Py_INCREF (&imagefile_imagefile_t);
  PyModule_AddObject (module, "imagefile", (PyObject *) &imagefile_imagefile_t);
  
  // add typelist var
  PyObject *typelist = PyList_New (0);
  if (typelist == nullptr)
    return;
  
  for (auto i : mobius::imagefile::imagefile::typelist)
    {
      PyObject *t = PyTuple_New (4);
      PyTuple_SetItem (t, 0, PyString_FromString (i.type.c_str ()));
      PyTuple_SetItem (t, 1, PyString_FromString (i.description.c_str ()));
      PyTuple_SetItem (t, 2, PyString_FromString (i.extensions.c_str ()));
      PyTuple_SetItem (t, 3, PyBool_FromLong (i.is_writeable));
      PyList_Append (typelist, t);
      
  //{"ewf", "EWF image file", "e01|ewf", true},
      
    }

  PyModule_AddObject (module, "typelist", (PyObject *) typelist);
}
