// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include <mobius/system/device_list.h>
#include <mobius/exception.inc>
#include <libudev.h>
#include <vector>
#include <stdexcept>

namespace mobius
{
namespace system
{
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief implementation struct
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=        
struct device_list_impl
{
  struct udev *udev = nullptr;
  struct udev_enumerate *udev_enumerate = nullptr;
  std::vector <device> devices;

   device_list_impl ();  
  ~device_list_impl ();
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief create shared implementation
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
device_list_impl::device_list_impl ()
{
  udev = udev_new ();

  if (!udev)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("cannot allocated udev struct"));

  udev_enumerate = udev_enumerate_new (udev);
  if (!udev_enumerate)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("cannot allocated udev_enumerate struct"));
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief destroy shared implementation
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
device_list_impl::~device_list_impl ()
{
  if (udev_enumerate)
    udev_enumerate_unref (udev_enumerate);
  
  if (udev)
    udev_unref (udev);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief create object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
device_list::device_list ()
{
  impl_ = std::make_shared <device_list_impl>();
  scan ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief scan system devices
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
device_list::scan ()
{
  udev_enumerate_scan_devices (impl_->udev_enumerate);
  struct udev_list_entry *entry = udev_enumerate_get_list_entry (impl_->udev_enumerate);
  impl_->devices.clear ();

  while (entry)
    {
      const char *path = udev_list_entry_get_name (entry);
      device dev (udev_device_new_from_syspath (impl_->udev, path));
      impl_->devices.push_back (dev);
      entry = udev_list_entry_get_next (entry);
    }
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief create iterator to first device
//! \return const iterator
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
device_list::const_iterator
device_list::begin () const
{
  return impl_->devices.begin ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief create iterator pass to last device
//! \return const iterator
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
device_list::const_iterator
device_list::end () const
{
  return impl_->devices.end ();
}

} // namespace system
} // namespace mobius
