// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include "disk_impl_imagefile.h"
#include <mobius/datetime/conv_iso_string.h>
#include <mobius/uri.h>

namespace mobius
{
namespace disk
{
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief constructor
//! \param url imagefile URL
//! \param imgtype imagefile type
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
disk_impl_imagefile::disk_impl_imagefile (const std::string& url, const std::string& imgtype)
  : url_ (url),
    imgtype_ (imgtype),
    imagefile_ (url_, imgtype_)
{
  mobius::uri uri (url);
  set_name ("imagefile " + uri.get_basename ());

  if (imagefile_.is_available ())
    {
      set_size (imagefile_.get_size ());
      set_sector_size (imagefile_.get_sector_size ());
      imgtype_ = imagefile_.get_type ();
    }
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief check whether disk is available
//! \return true/false
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
bool
disk_impl_imagefile::is_available () const
{
  return imagefile_.is_available ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief create new reader
//! \return new reader
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
const mobius::io::reader
disk_impl_imagefile::new_reader () const
{
  return imagefile_.new_reader ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief (re)scan disk
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
disk_impl_imagefile::scan ()
{
  imagefile_ = mobius::imagefile::imagefile (url_, imgtype_);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief get metadata
//! \return disk metadata
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
const mobius::metadata
disk_impl_imagefile::get_metadata () const
{
  return mobius::metadata
  {
    {"name", "name", "std::string", get_name ()},
    {"url", "URL", "std::string", url_},
    {"imagetype", "image file type", "std::string", imgtype_},
    {"size", "size", "std::uint64_t", std::to_string (imagefile_.get_size ()) + " bytes"},
    {"sectors", "sectors", "std::uint64_t", std::to_string (imagefile_.get_sectors ())},
    {"sector_size", "sector size", "std::uint32_t", std::to_string (imagefile_.get_sector_size ()) + " bytes"},
    {"segments", "number of segments", "std::uint32_t", std::to_string (imagefile_.get_segments ())},
    {"segment_size", "segment size", "std::uint64_t", std::to_string (imagefile_.get_segment_size ()) + " bytes"},
    {"model", "drive model", "std::string", imagefile_.get_drive_model ()},
    {"serial", "drive serial number", "std::string", imagefile_.get_drive_serial_number ()},
    {"acquisition_user", "acquisition user", "std::string", imagefile_.get_acquisition_user ()},
    {"acquisition_time", "acquisition time", "mobius::datetime", datetime_to_iso_string (imagefile_.get_acquisition_time ())},
    {"acquisition_tool", "acquisition tool", "std::string", imagefile_.get_acquisition_tool ()},
    {"acquisition_platform", "acquisition platform", "std::string", imagefile_.get_acquisition_platform ()},
    {"hash_md5", "MD5 hash", "std::string", imagefile_.get_hash_md5 ()}
  };
}

} // namespace disk
} // namespace mobius
