// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief  C++ API module wrapper
//!\author Eduardo Aguiar
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include <Python.h>
#include "filesystem_ext2.h"
#include "filesystem_hfs.h"
#include "filesystem_iso.h"
#include "filesystem_ntfs.h"
#include "filesystem_vfat.h"
#include "disk/disk.h"

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief function get_filesystems
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
PyObject*
func_get_filesystems (PyObject *self, PyObject *args)
{
  // parse arguments
  PyObject *arg_disk;

  if (!PyArg_ParseTuple (args, "O!", &disk_disk_t, &arg_disk))
    return nullptr;

  mobius::disk::disk *disk_p = ((disk_disk_o *) arg_disk)->obj;

  // run C++ code
  std::vector <mobius::filesystem::filesystem> filesystems;
  
  try
    {
      filesystems = mobius::filesystem::get_filesystems (*disk_p);
    }
  catch (const std::runtime_error& e)
    {
      Py_DECREF (self);
      PyErr_SetString (PyExc_Exception, e.what ());
      self = NULL;
    }

  // create list
  PyObject *ret = PyList_New (0);
  if (ret == nullptr)
    return nullptr;

  // populate list
  for (auto fs : filesystems)
    {
      PyObject *item = nullptr;

      // create filesystem object
      if (fs.get_type () == "ext2")
        {
          item = _PyObject_New (&filesystem_filesystem_ext2_t);
	  ((filesystem_filesystem_ext2_o *) item)->obj = new mobius::filesystem::filesystem_ext2 (fs);
	}

      else if (fs.get_type () == "hfs")
        {
          item = _PyObject_New (&filesystem_filesystem_hfs_t);
	  ((filesystem_filesystem_hfs_o *) item)->obj = new mobius::filesystem::filesystem_hfs (fs);
	}

      else if (fs.get_type () == "iso")
        {
          item = _PyObject_New (&filesystem_filesystem_iso_t);
	  ((filesystem_filesystem_iso_o *) item)->obj = new mobius::filesystem::filesystem_iso (fs);
	}

      else if (fs.get_type () == "ntfs")
        {
          item = _PyObject_New (&filesystem_filesystem_ntfs_t);
	  ((filesystem_filesystem_ntfs_o *) item)->obj = new mobius::filesystem::filesystem_ntfs (fs);
	}

      else if (fs.get_type () == "vfat")
        {
          item = _PyObject_New (&filesystem_filesystem_vfat_t);
	  ((filesystem_filesystem_vfat_o *) item)->obj = new mobius::filesystem::filesystem_vfat (fs);
	}

      else
        {
          PyErr_SetString (PyExc_TypeError, "unknown filesystem type");
          Py_CLEAR (ret);
          return nullptr;
	}

      // append object to the list
      PyList_Append (ret, (PyObject *) item);
   }

  return ret;
}
