// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019,2020 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#define PY_SSIZE_T_CLEAN        // PEP 353

#include "pymobius.h"
#include <bytesobject.h>
#include <mobius/exception.inc>
#include <stdexcept>

namespace mobius
{
namespace py
{
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Get function argument list size
//! \param args Argument tuple
//! \return Number of items
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::uint32_t
get_arg_size (PyObject *args) noexcept
{
  return PyTuple_Size (args);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Get function argument
//! \param args Argument tuple
//! \param idx Argument index
//! \return Argument
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
PyObject *
get_arg (PyObject *args, std::uint32_t idx)
{
  if (idx < PyTuple_Size (args))
    return PyTuple_GetItem (args, idx);

  throw std::out_of_range ("Argument index out of range");
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Get function argument as string
//! \param args Argument tuple
//! \param idx Argument index
//! \return Argument
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::string
get_arg_as_std_string (PyObject *args, std::uint32_t idx)
{
  return pystring_as_std_string (get_arg (args, idx));
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Get function argument as string (with default value)
//! \param args Argument tuple
//! \param idx Argument index
//! \param value Default value
//! \return Argument
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::string
get_arg_as_std_string (PyObject *args, std::uint32_t idx, const std::string& value)
{
  if (idx < get_arg_size (args))
    return pystring_as_std_string (get_arg (args, idx));

  return value;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Get function argument as bytearray
//! \param args Argument tuple
//! \param idx Argument index
//! \return Argument
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
mobius::bytearray
get_arg_as_bytearray (PyObject *args, std::uint32_t idx)
{
  return pybytes_as_bytearray (get_arg (args, idx));
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Get function argument as bytearray (with default value)
//! \param args Argument tuple
//! \param idx Argument index
//! \param value Default value
//! \return Argument
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
mobius::bytearray
get_arg_as_bytearray (PyObject *args, std::uint32_t idx, const mobius::bytearray& value)
{
  if (idx < get_arg_size (args))
    return pybytes_as_bytearray (get_arg (args, idx));

  return value;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Get function argument as datetime
//! \param args Argument tuple
//! \param idx Argument index
//! \return Argument
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
mobius::datetime::datetime
get_arg_as_datetime (PyObject *args, std::uint32_t idx)
{
  return pydatetime_as_datetime (get_arg (args, idx));
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Get function argument as datetime (with default value)
//! \param args Argument tuple
//! \param idx Argument index
//! \param value Default value
//! \return Argument
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
mobius::datetime::datetime
get_arg_as_datetime (
  PyObject *args,
  std::uint32_t idx,
  const mobius::datetime::datetime& value)
{
  if (idx < get_arg_size (args))
    return pydatetime_as_datetime (get_arg (args, idx));

  return value;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Get function argument as bool
//! \param args Argument tuple
//! \param idx Argument index
//! \return Argument
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
bool
get_arg_as_bool (PyObject *args, std::uint32_t idx)
{
  return pybool_as_bool (get_arg (args, idx));
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Get function argument as bool (with default value)
//! \param args Argument tuple
//! \param idx Argument index
//! \param value Default value
//! \return Argument
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
bool
get_arg_as_bool (PyObject *args, std::uint32_t idx, bool value)
{
  if (idx < get_arg_size (args))
    return pybool_as_bool (get_arg (args, idx));
  
  return value;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Check string
//! \param value Python object
//! \return true/false
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
bool
pystring_check (PyObject *value)
{
#if PY_MAJOR_VERSION > 2
  return PyUnicode_Check (value);
#else
  return PyString_Check (value) || PyUnicode_Check (value);
#endif
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Create PyUnicode from std::string
//! \param s C++ string
//! \return Python Unicode object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
PyObject *
pystring_from_std_string (const std::string& s)
{
  return PyUnicode_FromString (s.c_str ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Create std::string from PyUnicode
//! \param value Python String
//! \return C++ string
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::string
pystring_as_std_string (PyObject *value)
{
#if PY_MAJOR_VERSION > 2
  const char *buffer = PyUnicode_AsUTF8 (value);
#else
  const char *buffer = nullptr;
  PyObject *pystr = nullptr;

  if (PyString_Check (value))
    buffer = PyString_AsString (value);

  else if (PyUnicode_Check (value))
    {
      pystr = PyUnicode_AsUTF8String (value);

      if (pystr)
        buffer = PyString_AsString (pystr);
    }
#endif

  std::string str;

  if (buffer)
    str = buffer;

  else
    throw std::invalid_argument (mobius::MOBIUS_EXCEPTION_MSG ("Invalid Python string"));

#if PY_MAJOR_VERSION < 3
  if (pystr)
    Py_DECREF (pystr);
#endif

  return str;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Check PyBytes
//! \param value Python object
//! \return true/false
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
bool
pybytes_check (PyObject *value)
{
  return PyBytes_Check (value);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Create PyBytes from mobius::bytearray
//! \param array Bytearray
//! \return Python Bytes object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
PyObject *
pybytes_from_bytearray (const mobius::bytearray& array)
{
  return PyBytes_FromStringAndSize ((const char *) array.data (), array.size ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Create mobius::bytearray from PyBytes
//! \param value Python Object
//! \return Bytearray
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
mobius::bytearray
pybytes_as_bytearray (PyObject *value)
{
  char *buffer;
  Py_ssize_t length;

  if (PyBytes_AsStringAndSize (value, &buffer, &length) == -1)
    throw std::runtime_error (mobius::MOBIUS_EXCEPTION_MSG ("Invalid Python bytearray"));

  return mobius::bytearray (reinterpret_cast <const std::uint8_t *> (buffer), length);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Check PyBool
//! \param value Python object
//! \return true/false
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
bool
pybool_check (PyObject *value)
{
  return PyBool_Check (value);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Create PyBytes from mobius::bytearray
//! \param b Bool value
//! \return Python Bool object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
PyObject *
pybool_from_bool (bool b)
{
  PyObject *ret = b ? Py_True : Py_False;
  Py_INCREF (ret);
  
  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Create bool from PyBool
//! \param value Python Object
//! \return bool value
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
bool
pybool_as_bool (PyObject *value)
{
  return value == Py_True;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Create PyLong from std::uint64_t
//! \param value Value
//! \return Python Unicode object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
PyObject *
pylong_from_std_uint64_t (std::uint64_t value)
{
  return PyLong_FromUnsignedLongLong (value);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Create std::uint64_t from PyLong
//! \param value Python Object
//! \return std::uint64_t
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::uint64_t
pylong_as_std_uint64_t (PyObject *value)
{
  return PyLong_AsUnsignedLongLong (value);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Create PyLong from int
//! \param value Value
//! \return Python Unicode object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
PyObject *
pylong_from_int (int value)
{
  return PyLong_FromLongLong (value);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Create int from PyLong
//! \param value Python Object
//! \return int
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
int
pylong_as_int (PyObject *value)
{
  return PyLong_AsLong (value);
}

} // namespace py
} // namespace mobius
