// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019,2020 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include "cipher_des.h"
#include <mobius/exception.inc>
#include <stdexcept>
#include <iostream>

static constexpr bool DEBUG = false;

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Key permutation table
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static constexpr std::uint64_t PC_1[] =
{
  0x0000000000000080, 0x0000000000008000, 0x0000000000800000, 0x0000000080000000,
  0x0000008000000000, 0x0000800000000000, 0x0080000000000000, 0x8000000000000000,
  0x0000000000000040, 0x0000000000004000, 0x0000000000400000, 0x0000000040000000,
  0x0000004000000000, 0x0000400000000000, 0x0040000000000000, 0x4000000000000000,
  0x0000000000000020, 0x0000000000002000, 0x0000000000200000, 0x0000000020000000,
  0x0000002000000000, 0x0000200000000000, 0x0020000000000000, 0x2000000000000000,
  0x0000000000000010, 0x0000000000001000, 0x0000000000100000, 0x0000000010000000,
  0x0000000000000002, 0x0000000000000200, 0x0000000000020000, 0x0000000002000000,
  0x0000000200000000, 0x0000020000000000, 0x0002000000000000, 0x0200000000000000,
  0x0000000000000004, 0x0000000000000400, 0x0000000000040000, 0x0000000004000000,
  0x0000000400000000, 0x0000040000000000, 0x0004000000000000, 0x0400000000000000,
  0x0000000000000008, 0x0000000000000800, 0x0000000000080000, 0x0000000008000000,
  0x0000000800000000, 0x0000080000000000, 0x0008000000000000, 0x0800000000000000,
  0x0000001000000000, 0x0000100000000000, 0x0010000000000000, 0x1000000000000000
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Key left shift
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static constexpr int LSHIFTS[] = {1, 1, 2, 2, 2, 2, 2, 2, 1, 2, 2, 2, 2, 2, 2, 1};

//! \brief kn(1-16) permutation
static constexpr std::uint64_t PC_2[] =
{
  0x0000040000000000, 0x0000008000000000, 0x0000200000000000, 0x0000000100000000,
  0x0080000000000000, 0x0008000000000000, 0x0020000000000000, 0x0000000010000000,
  0x0000020000000000, 0x0004000000000000, 0x0000000800000000, 0x0000400000000000,
  0x0000000200000000, 0x0000002000000000, 0x0000100000000000, 0x0010000000000000,
  0x0000000040000000, 0x0001000000000000, 0x0000010000000000, 0x0002000000000000,
  0x0000000020000000, 0x0000001000000000, 0x0000080000000000, 0x0040000000000000,
  0x0000000000008000, 0x0000000000000010, 0x0000000002000000, 0x0000000000080000,
  0x0000000000000200, 0x0000000000000002, 0x0000000004000000, 0x0000000000010000,
  0x0000000000000020, 0x0000000000000800, 0x0000000000800000, 0x0000000000000100,
  0x0000000000001000, 0x0000000000000080, 0x0000000000020000, 0x0000000000000001,
  0x0000000000400000, 0x0000000000000008, 0x0000000000000400, 0x0000000000004000,
  0x0000000000000040, 0x0000000000100000, 0x0000000008000000, 0x0000000001000000,
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Initial permutation table
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static constexpr std::uint64_t IP[] =
{
  0x0000000000000040, 0x0000000000004000, 0x0000000000400000, 0x0000000040000000,
  0x0000004000000000, 0x0000400000000000, 0x0040000000000000, 0x4000000000000000,
  0x0000000000000010, 0x0000000000001000, 0x0000000000100000, 0x0000000010000000,
  0x0000001000000000, 0x0000100000000000, 0x0010000000000000, 0x1000000000000000,
  0x0000000000000004, 0x0000000000000400, 0x0000000000040000, 0x0000000004000000,
  0x0000000400000000, 0x0000040000000000, 0x0004000000000000, 0x0400000000000000,
  0x0000000000000001, 0x0000000000000100, 0x0000000000010000, 0x0000000001000000,
  0x0000000100000000, 0x0000010000000000, 0x0001000000000000, 0x0100000000000000,
  0x0000000000000080, 0x0000000000008000, 0x0000000000800000, 0x0000000080000000,
  0x0000008000000000, 0x0000800000000000, 0x0080000000000000, 0x8000000000000000,
  0x0000000000000020, 0x0000000000002000, 0x0000000000200000, 0x0000000020000000,
  0x0000002000000000, 0x0000200000000000, 0x0020000000000000, 0x2000000000000000,
  0x0000000000000008, 0x0000000000000800, 0x0000000000080000, 0x0000000008000000,
  0x0000000800000000, 0x0000080000000000, 0x0008000000000000, 0x0800000000000000,
  0x0000000000000002, 0x0000000000000200, 0x0000000000020000, 0x0000000002000000,
  0x0000000200000000, 0x0000020000000000, 0x0002000000000000, 0x0200000000000000,
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Expand permutation table
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static constexpr std::uint64_t E[] =
{
  0x0000000000000001, 0x0000000080000000, 0x0000000040000000, 0x0000000020000000,
  0x0000000010000000, 0x0000000008000000, 0x0000000010000000, 0x0000000008000000,
  0x0000000004000000, 0x0000000002000000, 0x0000000001000000, 0x0000000000800000,
  0x0000000001000000, 0x0000000000800000, 0x0000000000400000, 0x0000000000200000,
  0x0000000000100000, 0x0000000000080000, 0x0000000000100000, 0x0000000000080000,
  0x0000000000040000, 0x0000000000020000, 0x0000000000010000, 0x0000000000008000,
  0x0000000000010000, 0x0000000000008000, 0x0000000000004000, 0x0000000000002000,
  0x0000000000001000, 0x0000000000000800, 0x0000000000001000, 0x0000000000000800,
  0x0000000000000400, 0x0000000000000200, 0x0000000000000100, 0x0000000000000080,
  0x0000000000000100, 0x0000000000000080, 0x0000000000000040, 0x0000000000000020,
  0x0000000000000010, 0x0000000000000008, 0x0000000000000010, 0x0000000000000008,
  0x0000000000000004, 0x0000000000000002, 0x0000000000000001, 0x0000000080000000
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief P permutation table
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static constexpr std::uint64_t P[] =
{
  0x0000000000010000, 0x0000000002000000, 0x0000000000001000, 0x0000000000000800,
  0x0000000000000008, 0x0000000000100000, 0x0000000000000010, 0x0000000000008000,
  0x0000000080000000, 0x0000000000020000, 0x0000000000000200, 0x0000000000000040,
  0x0000000008000000, 0x0000000000004000, 0x0000000000000002, 0x0000000000400000,
  0x0000000040000000, 0x0000000001000000, 0x0000000000000100, 0x0000000000040000,
  0x0000000000000001, 0x0000000000000020, 0x0000000020000000, 0x0000000000800000,
  0x0000000000002000, 0x0000000000080000, 0x0000000000000004, 0x0000000004000000,
  0x0000000000000400, 0x0000000000200000, 0x0000000010000000, 0x0000000000000080,
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Final permutation table
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static constexpr std::uint64_t FP[] =
{
  0x0000000001000000, 0x0100000000000000, 0x0000000000010000, 0x0001000000000000,
  0x0000000000000100, 0x0000010000000000, 0x0000000000000001, 0x0000000100000000,
  0x0000000002000000, 0x0200000000000000, 0x0000000000020000, 0x0002000000000000,
  0x0000000000000200, 0x0000020000000000, 0x0000000000000002, 0x0000000200000000,
  0x0000000004000000, 0x0400000000000000, 0x0000000000040000, 0x0004000000000000,
  0x0000000000000400, 0x0000040000000000, 0x0000000000000004, 0x0000000400000000,
  0x0000000008000000, 0x0800000000000000, 0x0000000000080000, 0x0008000000000000,
  0x0000000000000800, 0x0000080000000000, 0x0000000000000008, 0x0000000800000000,
  0x0000000010000000, 0x1000000000000000, 0x0000000000100000, 0x0010000000000000,
  0x0000000000001000, 0x0000100000000000, 0x0000000000000010, 0x0000001000000000,
  0x0000000020000000, 0x2000000000000000, 0x0000000000200000, 0x0020000000000000,
  0x0000000000002000, 0x0000200000000000, 0x0000000000000020, 0x0000002000000000,
  0x0000000040000000, 0x4000000000000000, 0x0000000000400000, 0x0040000000000000,
  0x0000000000004000, 0x0000400000000000, 0x0000000000000040, 0x0000004000000000,
  0x0000000080000000, 0x8000000000000000, 0x0000000000800000, 0x0080000000000000,
  0x0000000000008000, 0x0000800000000000, 0x0000000000000080, 0x0000008000000000
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief S box 1 table
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static constexpr std::uint32_t S1[] =
{
  0xe0000000, 0x00000000, 0x40000000, 0xf0000000, 0xd0000000, 0x70000000, 0x10000000, 0x40000000,
  0x20000000, 0xe0000000, 0xf0000000, 0x20000000, 0xb0000000, 0xd0000000, 0x80000000, 0x10000000,
  0x30000000, 0xa0000000, 0xa0000000, 0x60000000, 0x60000000, 0xc0000000, 0xc0000000, 0xb0000000,
  0x50000000, 0x90000000, 0x90000000, 0x50000000, 0x00000000, 0x30000000, 0x70000000, 0x80000000,
  0x40000000, 0xf0000000, 0x10000000, 0xc0000000, 0xe0000000, 0x80000000, 0x80000000, 0x20000000,
  0xd0000000, 0x40000000, 0x60000000, 0x90000000, 0x20000000, 0x10000000, 0xb0000000, 0x70000000,
  0xf0000000, 0x50000000, 0xc0000000, 0xb0000000, 0x90000000, 0x30000000, 0x70000000, 0xe0000000,
  0x30000000, 0xa0000000, 0xa0000000, 0x00000000, 0x50000000, 0x60000000, 0x00000000, 0xd0000000
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief S box 2 table
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static constexpr std::uint32_t S2[] =
{
  0x0f000000, 0x03000000, 0x01000000, 0x0d000000, 0x08000000, 0x04000000, 0x0e000000, 0x07000000,
  0x06000000, 0x0f000000, 0x0b000000, 0x02000000, 0x03000000, 0x08000000, 0x04000000, 0x0e000000,
  0x09000000, 0x0c000000, 0x07000000, 0x00000000, 0x02000000, 0x01000000, 0x0d000000, 0x0a000000,
  0x0c000000, 0x06000000, 0x00000000, 0x09000000, 0x05000000, 0x0b000000, 0x0a000000, 0x05000000,
  0x00000000, 0x0d000000, 0x0e000000, 0x08000000, 0x07000000, 0x0a000000, 0x0b000000, 0x01000000,
  0x0a000000, 0x03000000, 0x04000000, 0x0f000000, 0x0d000000, 0x04000000, 0x01000000, 0x02000000,
  0x05000000, 0x0b000000, 0x08000000, 0x06000000, 0x0c000000, 0x07000000, 0x06000000, 0x0c000000,
  0x09000000, 0x00000000, 0x03000000, 0x05000000, 0x02000000, 0x0e000000, 0x0f000000, 0x09000000
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief S box 3 table
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static constexpr std::uint32_t S3[] =
{
  0x00a00000, 0x00d00000, 0x00000000, 0x00700000, 0x00900000, 0x00000000, 0x00e00000, 0x00900000,
  0x00600000, 0x00300000, 0x00300000, 0x00400000, 0x00f00000, 0x00600000, 0x00500000, 0x00a00000,
  0x00100000, 0x00200000, 0x00d00000, 0x00800000, 0x00c00000, 0x00500000, 0x00700000, 0x00e00000,
  0x00b00000, 0x00c00000, 0x00400000, 0x00b00000, 0x00200000, 0x00f00000, 0x00800000, 0x00100000,
  0x00d00000, 0x00100000, 0x00600000, 0x00a00000, 0x00400000, 0x00d00000, 0x00900000, 0x00000000,
  0x00800000, 0x00600000, 0x00f00000, 0x00900000, 0x00300000, 0x00800000, 0x00000000, 0x00700000,
  0x00b00000, 0x00400000, 0x00100000, 0x00f00000, 0x00200000, 0x00e00000, 0x00c00000, 0x00300000,
  0x00500000, 0x00b00000, 0x00a00000, 0x00500000, 0x00e00000, 0x00200000, 0x00700000, 0x00c00000
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief S box 4 table
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static constexpr std::uint32_t S4[] =
{
  0x00070000, 0x000d0000, 0x000d0000, 0x00080000, 0x000e0000, 0x000b0000, 0x00030000, 0x00050000,
  0x00000000, 0x00060000, 0x00060000, 0x000f0000, 0x00090000, 0x00000000, 0x000a0000, 0x00030000,
  0x00010000, 0x00040000, 0x00020000, 0x00070000, 0x00080000, 0x00020000, 0x00050000, 0x000c0000,
  0x000b0000, 0x00010000, 0x000c0000, 0x000a0000, 0x00040000, 0x000e0000, 0x000f0000, 0x00090000,
  0x000a0000, 0x00030000, 0x00060000, 0x000f0000, 0x00090000, 0x00000000, 0x00000000, 0x00060000,
  0x000c0000, 0x000a0000, 0x000b0000, 0x00010000, 0x00070000, 0x000d0000, 0x000d0000, 0x00080000,
  0x000f0000, 0x00090000, 0x00010000, 0x00040000, 0x00030000, 0x00050000, 0x000e0000, 0x000b0000,
  0x00050000, 0x000c0000, 0x00020000, 0x00070000, 0x00080000, 0x00020000, 0x00040000, 0x000e0000
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief S box 5 table
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static constexpr std::uint32_t S5[] =
{
  0x00002000, 0x0000e000, 0x0000c000, 0x0000b000, 0x00004000, 0x00002000, 0x00001000, 0x0000c000,
  0x00007000, 0x00004000, 0x0000a000, 0x00007000, 0x0000b000, 0x0000d000, 0x00006000, 0x00001000,
  0x00008000, 0x00005000, 0x00005000, 0x00000000, 0x00003000, 0x0000f000, 0x0000f000, 0x0000a000,
  0x0000d000, 0x00003000, 0x00000000, 0x00009000, 0x0000e000, 0x00008000, 0x00009000, 0x00006000,
  0x00004000, 0x0000b000, 0x00002000, 0x00008000, 0x00001000, 0x0000c000, 0x0000b000, 0x00007000,
  0x0000a000, 0x00001000, 0x0000d000, 0x0000e000, 0x00007000, 0x00002000, 0x00008000, 0x0000d000,
  0x0000f000, 0x00006000, 0x00009000, 0x0000f000, 0x0000c000, 0x00000000, 0x00005000, 0x00009000,
  0x00006000, 0x0000a000, 0x00003000, 0x00004000, 0x00000000, 0x00005000, 0x0000e000, 0x00003000
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief S box 6 table
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static constexpr std::uint32_t S6[] =
{
  0x00000c00, 0x00000a00, 0x00000100, 0x00000f00, 0x00000a00, 0x00000400, 0x00000f00, 0x00000200,
  0x00000900, 0x00000700, 0x00000200, 0x00000c00, 0x00000600, 0x00000900, 0x00000800, 0x00000500,
  0x00000000, 0x00000600, 0x00000d00, 0x00000100, 0x00000300, 0x00000d00, 0x00000400, 0x00000e00,
  0x00000e00, 0x00000000, 0x00000700, 0x00000b00, 0x00000500, 0x00000300, 0x00000b00, 0x00000800,
  0x00000900, 0x00000400, 0x00000e00, 0x00000300, 0x00000f00, 0x00000200, 0x00000500, 0x00000c00,
  0x00000200, 0x00000900, 0x00000800, 0x00000500, 0x00000c00, 0x00000f00, 0x00000300, 0x00000a00,
  0x00000700, 0x00000b00, 0x00000000, 0x00000e00, 0x00000400, 0x00000100, 0x00000a00, 0x00000700,
  0x00000100, 0x00000600, 0x00000d00, 0x00000000, 0x00000b00, 0x00000800, 0x00000600, 0x00000d00
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief S box 7 table
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static constexpr std::uint32_t S7[] =
{
  0x00000040, 0x000000d0, 0x000000b0, 0x00000000, 0x00000020, 0x000000b0, 0x000000e0, 0x00000070,
  0x000000f0, 0x00000040, 0x00000000, 0x00000090, 0x00000080, 0x00000010, 0x000000d0, 0x000000a0,
  0x00000030, 0x000000e0, 0x000000c0, 0x00000030, 0x00000090, 0x00000050, 0x00000070, 0x000000c0,
  0x00000050, 0x00000020, 0x000000a0, 0x000000f0, 0x00000060, 0x00000080, 0x00000010, 0x00000060,
  0x00000010, 0x00000060, 0x00000040, 0x000000b0, 0x000000b0, 0x000000d0, 0x000000d0, 0x00000080,
  0x000000c0, 0x00000010, 0x00000030, 0x00000040, 0x00000070, 0x000000a0, 0x000000e0, 0x00000070,
  0x000000a0, 0x00000090, 0x000000f0, 0x00000050, 0x00000060, 0x00000000, 0x00000080, 0x000000f0,
  0x00000000, 0x000000e0, 0x00000050, 0x00000020, 0x00000090, 0x00000030, 0x00000020, 0x000000c0
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief S box 8 table
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static constexpr std::uint32_t S8[] =
{
  0x0000000d, 0x00000001, 0x00000002, 0x0000000f, 0x00000008, 0x0000000d, 0x00000004, 0x00000008,
  0x00000006, 0x0000000a, 0x0000000f, 0x00000003, 0x0000000b, 0x00000007, 0x00000001, 0x00000004,
  0x0000000a, 0x0000000c, 0x00000009, 0x00000005, 0x00000003, 0x00000006, 0x0000000e, 0x0000000b,
  0x00000005, 0x00000000, 0x00000000, 0x0000000e, 0x0000000c, 0x00000009, 0x00000007, 0x00000002,
  0x00000007, 0x00000002, 0x0000000b, 0x00000001, 0x00000004, 0x0000000e, 0x00000001, 0x00000007,
  0x00000009, 0x00000004, 0x0000000c, 0x0000000a, 0x0000000e, 0x00000008, 0x00000002, 0x0000000d,
  0x00000000, 0x0000000f, 0x00000006, 0x0000000c, 0x0000000a, 0x00000009, 0x0000000d, 0x00000000,
  0x0000000f, 0x00000003, 0x00000003, 0x00000005, 0x00000005, 0x00000006, 0x00000008, 0x0000000b
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Convert int value to binary string
//! \param value int value
//! \return binary string
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
template <typename T>
static const std::string
bin (T value)
{
  std::string r;
  for (int i = sizeof (T) * 8 - 1; i >= 0; i--)
    {
      r += (value & (std::uint64_t (1) << i)) ? '1' : '0';
      if (!(i & 7))
        r += ' ';
    }
  return r;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Permutate bits from value, according to a given bit table
//! \param value
//! \param table bit mask table
//! \param size number of entries of table
//! \return permutated value
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static std::uint64_t
permutate (std::uint64_t value, const std::uint64_t table[], std::size_t size) noexcept
{
  std::uint64_t rc = 0;

  for (std::size_t i = 0; i < size; i++)
    {
      rc <<= 1;
      if (value & table[i])
        rc |= 1;
    }

  return rc;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Cipher function f (FIPS 46-3)
//! \param r R value
//! \param kn n-subkey
//! \return f value
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static std::uint64_t
compute_f (std::uint32_t r, std::uint64_t kn) noexcept
{
  std::uint64_t f = permutate (r, E, 48) ^ kn;

  if (DEBUG)
    std::cout << "K + E(r) = " << bin (f) << std::endl;

  std::uint32_t sbox =
    S1[(f >> 42) & 0x3f] | S2[(f >> 36) & 0x3f] |
    S3[(f >> 30) & 0x3f] | S4[(f >> 24) & 0x3f] |
    S5[(f >> 18) & 0x3f] | S6[(f >> 12) & 0x3f] |
    S7[(f >> 6) & 0x3f] | S8[f & 0x3f];

  if (DEBUG)
    std::cout << "sbox = " << bin (sbox) << std::endl;

  return permutate (sbox, P, 32);
}

namespace mobius
{
namespace crypt
{
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Constructor
//! \param key encryption/decryption password
//! \param mode cipher block mode
//! \param iv initialization vector
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
cipher_des::cipher_des (
  const mobius::bytearray& key,
  const std::string& mode,
  const mobius::bytearray& iv
)
  : cipher_block (8, mode, iv)
{
  // transform key into uint64_t value
  std::uint64_t k;

  if (key.size () == 7)
    {
      k = (std::uint64_t (key[0] & 0xfe) << 56) |
          (std::uint64_t (key[0] & 0x01) << 55) |
          (std::uint64_t (key[1] & 0xfc) << 47) |
          (std::uint64_t (key[1] & 0x03) << 46) |
          (std::uint64_t (key[2] & 0xf8) << 38) |
          (std::uint64_t (key[2] & 0x07) << 37) |
          (std::uint64_t (key[3] & 0xf0) << 29) |
          (std::uint64_t (key[3] & 0x0f) << 28) |
          (std::uint64_t (key[4] & 0xe0) << 20) |
          (std::uint64_t (key[4] & 0x1f) << 19) |
          (std::uint64_t (key[5] & 0xc0) << 11) |
          (std::uint64_t (key[5] & 0x3f) << 10) |
          (std::uint64_t (key[6] & 0x80) << 2) |
          (std::uint64_t (key[6] & 0x7f) << 1);
    }

  else if (key.size () == 8)
    {
      k = (std::uint64_t (key[0]) << 56) |
          (std::uint64_t (key[1]) << 48) |
          (std::uint64_t (key[2]) << 40) |
          (std::uint64_t (key[3]) << 32) |
          (std::uint64_t (key[4]) << 24) |
          (std::uint64_t (key[5]) << 16) |
          (std::uint64_t (key[6]) << 8) |
          (std::uint64_t (key[7]));
    }

  else
    throw std::out_of_range (MOBIUS_EXCEPTION_MSG ("key size must be either 7 or 8 bytes"));

  // generate subkeys (kn_[i])
  std::uint64_t kp = permutate (k, PC_1, 56);
  std::uint32_t cn = kp >> 28;
  std::uint32_t dn = kp & 0xfffffff;

  for (auto i = 0; i < 16; i++)
    {
      cn = ((cn << LSHIFTS[i]) | (cn >> (28 - LSHIFTS[i]))) & 0xfffffff;
      dn = ((dn << LSHIFTS[i]) | (dn >> (28 - LSHIFTS[i]))) & 0xfffffff;
      kn_[i] = permutate ((std::uint64_t (cn) << 28) | dn, PC_2, 48);

      if (DEBUG)
        std::cout << "k" << i + 1 << " = " << bin (kn_[i]) << std::endl;
    }
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Encrypt/decrypt a data block
//! \param data data block
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
cipher_des::_crypt_block (mobius::bytearray& data, bool encryption) noexcept
{
  // decode data buffer as uint64_t value
  std::uint64_t m = (std::uint64_t (data[0]) << 56) |
      (std::uint64_t (data[1]) << 48) |
      (std::uint64_t (data[2]) << 40) |
      (std::uint64_t (data[3]) << 32) |
      (std::uint64_t (data[4]) << 24) |
      (std::uint64_t (data[5]) << 16) |
      (std::uint64_t (data[6]) << 8) |
      (std::uint64_t (data[7]));

  // apply the initial permutation
  std::uint64_t ip = permutate (m, IP, 64);
  std::uint32_t l = ip >> 32;
  std::uint32_t r = ip;

  if (DEBUG)
    std::cout << "ip = " << bin (ip) << std::endl;

  // 16 rounds
  std::uint32_t l_old;

  for (auto i = 0; i < 16; i++)
    {
      l_old = l;
      l = r;
      r = l_old ^ compute_f (r, kn_[encryption ? i : 15 - i]);

      if (DEBUG)
        {
          std::cout << "l" << i + 1 << " = " << bin (l) << std::endl;
          std::cout << "r" << i + 1 << " = " << bin (r) << std::endl;
        }
    }

  // apply the final permutation and return value
  m = permutate ((std::uint64_t (r) << 32) | l, FP, 64);

  // encode uint64_t value back into data buffer
  data[0] = m >> 56;
  data[1] = m >> 48;
  data[2] = m >> 40;
  data[3] = m >> 32;
  data[4] = m >> 24;
  data[5] = m >> 16;
  data[6] = m >> 8;
  data[7] = m;
}

} // namespace crypt
} // namespace mobius
