// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include <mobius/imagefile/imagefile.h>
#include <mobius/imagefile/imagefile_impl_null.h>
#include <mobius/imagefile/ewf/imagefile_impl.h>
#include <mobius/imagefile/imagefile_impl_raw.h>
#include <mobius/imagefile/imagefile_impl_split.h>
#include <mobius/imagefile/imagefile_impl_dossier.h>
#include <mobius/imagefile/imagefile_impl_solo.h>
#include <mobius/imagefile/imagefile_impl_talon.h>
#include <mobius/io/file.h>

namespace mobius
{
namespace imagefile
{
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief imagefile types available
// id, description, file extensions, writeable (true/false)
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
const std::vector <imagefile::info> imagefile::typelist =
{
  {"ewf", "EWF image file", "E01|e01|ewf", true},
  {"solo", "ICS Solo image file", "txt", false},
  {"dossier", "Logicube Dossier image file", "LOG", false},
  {"talon", "Logicube Talon image file", "LOG", false},
  {"raw", "raw image file", "raw|dd|img|*", true},
  {"split", "split image file", "001", true}
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief construct object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
imagefile::imagefile () : is_valid_ (false)
{
  impl_ = std::make_shared <imagefile_impl_null> ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief construct object
//! \param url imagefile URL
//! \param type imagefile type (default = "autodetect")
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
imagefile::imagefile (const std::string& url, const std::string& type)
  : url_ (url), is_valid_ (true)

{
  // detect imagefile type, if type == "autodetect"
  std::string i_type;

  if (type == "autodetect")
    {
      if (ewf::imagefile_impl::is_instance (url_))
        i_type = "ewf";

      else if (imagefile_impl_dossier::is_instance (url_))
        i_type = "dossier";

      else if (imagefile_impl_talon::is_instance (url_))
        i_type = "talon";

      else if (imagefile_impl_solo::is_instance (url_))
        i_type = "solo";

      else if (imagefile_impl_split::is_instance (url_))
        i_type = "split";

      else
        i_type = "raw";
    }
  else
    i_type = type;

  // create impl_ according to i_type
  if (i_type == "ewf")
    impl_ = std::make_shared <ewf::imagefile_impl> (url_);

  else if (i_type == "raw")
    impl_ = std::make_shared <imagefile_impl_raw> (url_);

  else if (i_type == "split")
    impl_ = std::make_shared <imagefile_impl_split> (url_);

  else if (i_type == "dossier")
    impl_ = std::make_shared <imagefile_impl_dossier> (url_);

  else if (i_type == "solo")
    impl_ = std::make_shared <imagefile_impl_solo> (url_);

  else if (i_type == "talon")
    impl_ = std::make_shared <imagefile_impl_talon> (url_);

  else
    {
      impl_ = std::make_shared <imagefile_impl_null> ();
      is_valid_ = false;
    }
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief check if imagefile is is_available
//! \return true/false
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
bool
imagefile::is_available () const
{
  mobius::io::file f (url_);
  return f && f.exists ();
}

} // namespace imagefile
} // namespace mobius
