// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019,2020,2021,2022,2023,2024 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include "ui.h"
#include <mobius/core/resource.h>
#include <mobius/exception.inc>
#include <stdexcept>

namespace mobius
{
namespace ui
{
namespace
{
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Global UI implementation
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static std::shared_ptr <ui_impl_base> g_impl_;
static std::string g_impl_id_;

} // namespace

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Set current UI implementation class
//! \param id UI implementation ID
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
set_implementation (const std::string& id)
{
  if (g_impl_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("UI implementation already set"));

  // search implementation
  auto resource = mobius::core::get_resource ("ui.implementation." + id);
  
  if (resource)
    {
      g_impl_ = resource.get_value <resource_type> () ();
      g_impl_id_ = id;
    }
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Get current UI implementation class
//! \return Smart pointer to UI implementation class
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::shared_ptr <ui_impl_base>
get_implementation ()
{
  // if g_impl_ is not set, create using first UI implementation available
  if (!g_impl_)
    {
      auto resources = mobius::core::get_resources ("ui.implementation");

      if (resources.size () == 0)
        throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("no UI implementation found"));

      else
        {
          g_impl_ = resources[0].get_value <resource_type> () ();
          g_impl_id_ = resources[0].get_id ();
        }
    }

  // return implementation instance
  return g_impl_;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Constructor
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
ui::ui ()
  : impl_ (get_implementation ())
{
}

} // namespace ui
} // namespace mobius
