/* This file is part of the MPFRCPP Library.

  Copyright (c) 2006 -- 2007 Alexey V. Beshenov <bav.272304@gmail.com>.

  The MPFRCPP Library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public License as
  published by the Free Software Foundation; either version 2.1 of the
  License, or (at your option) any later version.

  The MPFRCPP Library is distributed in the hope that it will be
  useful, but WITHOUT ANY WARRANTY; without even the implied warranty
  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with the MPFRCPP Library; see the file COPYING.LIB. If
  not, write to the Free Software Foundation, Inc., 51 Franklin Street,
  Fifth Floor, Boston, MA 02110-1301, USA. */

/**
 * @file constants.hpp
 * @date 2007-04-03
 * MPFRCPP numeric constants.
 * INTERNAL HEADER, NOT TO BE USED DIRECTLY.
 */

#ifndef INCLUDED_BY_MPFRCPP
    #error THIS IS AN INTERNAL HEADER ONLY, NOT TO BE USED DIRECTLY
#endif

#ifndef MPFRCPP_FUNCTIONS_CONSTANTS
#define MPFRCPP_FUNCTIONS_CONSTANTS

namespace mpfrcpp {

    //------------------------------------------------------------

    /**
     * Numeric constants
     */

    class ConstantClass : public NumericFunction {
            bool cache_;
        public :

            ConstantClass () throw() : NumericFunction (), cache_( true ) {}
            ConstantClass ( const Precision& pr, const RoundMode& rm ) throw() :
                NumericFunction (), cache_( true ) {}
            ConstantClass ( const Precision& pr ) throw() :
                NumericFunction (), cache_( true ) {}
            ConstantClass ( const RoundMode& rm ) throw() :
                NumericFunction (), cache_( true ) {}

            Real log2 () const throw ();
            Real log2 ( const Precision &, const RoundMode & ) const throw ();
            Real log2 ( const Precision & ) const throw ();
            Real log2 ( const RoundMode & ) const throw ();

            Real pi () const throw ();
            Real pi ( const Precision &, const RoundMode & ) const throw ();
            Real pi ( const Precision & ) const throw ();
            Real pi ( const RoundMode & ) const throw ();

            Real Euler () const throw ();
            Real Euler ( const Precision &, const RoundMode & ) const throw ();
            Real Euler ( const Precision & ) const throw ();
            Real Euler ( const RoundMode & ) const throw ();

            Real Catalan () const throw ();
            Real Catalan ( const Precision &, const RoundMode & ) const throw ();
            Real Catalan ( const Precision & ) const throw ();
            Real Catalan ( const RoundMode & ) const throw ();

            void cache ( bool c ) throw () {
                cache_ = c;
            }
            bool isCached () const throw () {
                return cache_;
            }
            static void freeCache () throw () {
                mpfr_free_cache ();
            }

    };

    //------------------------------------------------------------

    /**
     * log 2, 0,693...
     */

    Real ConstantClass::log2 () const throw () {
        Real x ( getPrecision() );
        mpfr_const_log2 ( x.getMpfrT(), getRoundMode().getMpfrRndT() );
        if ( !isCached () )
            freeCache();
        return x;
    }

    Real ConstantClass::log2 ( const Precision &p, const RoundMode &r )
    const throw () {
        Real x ( p );
        mpfr_const_log2 ( x.getMpfrT(), r.getMpfrRndT() );
        if ( !isCached () )
            freeCache();
        return x;
    }

    Real ConstantClass::log2 ( const Precision &p )
    const throw () {
        Real x ( p );
        mpfr_const_log2 ( x.getMpfrT(), getRoundMode().getMpfrRndT() );
        if ( !isCached () )
            freeCache();
        return x;
    }

    Real ConstantClass::log2 ( const RoundMode &r )
    const throw () {
        Real x ( getPrecision() );
        mpfr_const_log2 ( x.getMpfrT(), r.getMpfrRndT() );
        if ( !isCached () )
            freeCache();
        return x;
    }

    //------------------------------------------------------------

    /**
     * pi, 3.1415926...
     */

    Real ConstantClass::pi () const throw () {
        Real x ( getPrecision() );
        mpfr_const_pi ( x.getMpfrT(), getRoundMode().getMpfrRndT() );
        if ( !isCached () )
            freeCache();
        return x;
    }

    Real ConstantClass::pi ( const Precision &p, const RoundMode &r )
    const throw () {
        Real x ( p );
        mpfr_const_pi ( x.getMpfrT(), r.getMpfrRndT() );
        if ( !isCached () )
            freeCache();
        return x;
    }

    Real ConstantClass::pi ( const Precision &p ) const throw () {
        Real x ( p );
        mpfr_const_pi ( x.getMpfrT(), getRoundMode().getMpfrRndT() );
        if ( !isCached () )
            freeCache();
        return x;
    }

    Real ConstantClass::pi ( const RoundMode &r ) const throw () {
        Real x ( getPrecision() );
        mpfr_const_pi ( x.getMpfrT(), r.getMpfrRndT() );
        if ( !isCached () )
            freeCache();
        return x;
    }

    //------------------------------------------------------------

    /**
     * The Euler-Mascheroni constant, 0.577...
     */

    Real ConstantClass::Euler () const throw () {
        Real x ( getPrecision() );
        mpfr_const_euler ( x.getMpfrT(), getRoundMode().getMpfrRndT() );
        if ( !isCached () )
            freeCache();
        return x;
    }

    Real ConstantClass::Euler ( const Precision &p, const RoundMode &r )
    const throw () {
        Real x ( p );
        mpfr_const_euler ( x.getMpfrT(), r.getMpfrRndT() );
        if ( !isCached () )
            freeCache();
        return x;
    }

    Real ConstantClass::Euler ( const Precision &p ) const throw () {
        Real x ( p );
        mpfr_const_euler ( x.getMpfrT(), getRoundMode().getMpfrRndT() );
        if ( !isCached () )
            freeCache();
        return x;
    }

    Real ConstantClass::Euler ( const RoundMode &r ) const throw () {
        Real x ( getPrecision() );
        mpfr_const_euler ( x.getMpfrT(), r.getMpfrRndT() );
        if ( !isCached () )
            freeCache();
        return x;
    }

    //------------------------------------------------------------

    /**
     * The Catalan constant, 0.915...
     */

    Real ConstantClass::Catalan () const throw () {
        Real x ( getPrecision() );
        mpfr_const_catalan ( x.getMpfrT(), getRoundMode().getMpfrRndT() );
        if ( !isCached () )
            freeCache();
        return x;
    }

    Real ConstantClass::Catalan ( const Precision &p, const RoundMode &r )
    const throw () {
        Real x ( p );
        mpfr_const_catalan ( x.getMpfrT(), r.getMpfrRndT() );
        if ( !isCached () )
            freeCache();
        return x;
    }

    Real ConstantClass::Catalan ( const Precision &p ) const throw () {
        Real x ( p );
        mpfr_const_catalan ( x.getMpfrT(), getRoundMode().getMpfrRndT() );
        if ( !isCached () )
            freeCache();
        return x;
    }

    Real ConstantClass::Catalan ( const RoundMode &r ) const throw () {
        Real x ( getPrecision() );
        mpfr_const_catalan ( x.getMpfrT(), r.getMpfrRndT() );
        if ( !isCached () )
            freeCache();
        return x;
    }

    //------------------------------------------------------------

}    // namespace mpfrcpp

#endif    // MPFRCPP_FUNCTIONS_CONSTANTS
