/*
SDX: Documentary System in XML.
Copyright (C) 2000, 2001, 2002  Ministere de la culture et de la communication (France), AJLSM

Ministere de la culture et de la communication,
Mission de la recherche et de la technologie
3 rue de Valois, 75042 Paris Cedex 01 (France)
mrt@culture.fr, michel.bottin@culture.fr

AJLSM, 17, rue Vital Carles, 33000 Bordeaux (France)
sevigny@ajlsm.com

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the
Free Software Foundation, Inc.
59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
or connect to:
http://www.fsf.org/copyleft/gpl.html
*/
package fr.gouv.culture.sdx.framework;

import fr.gouv.culture.sdx.application.Application;
import fr.gouv.culture.sdx.documentbase.DocumentBase;
import fr.gouv.culture.sdx.documentbase.LuceneDocumentBase;
import fr.gouv.culture.sdx.exception.SDXException;
import fr.gouv.culture.sdx.exception.SDXExceptionCode;
import fr.gouv.culture.sdx.pipeline.Pipeline;
import fr.gouv.culture.sdx.search.lucene.analysis.AnalyzerManager;
import fr.gouv.culture.sdx.user.SuperuserInformation;
import fr.gouv.culture.sdx.utils.Utilities;
import fr.gouv.culture.sdx.utils.constants.Node;
import org.apache.avalon.excalibur.io.FileUtil;
import org.apache.avalon.framework.activity.Disposable;
import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.activity.Startable;
import org.apache.avalon.framework.activity.Suspendable;
import org.apache.avalon.framework.component.*;
import org.apache.avalon.framework.configuration.*;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.logger.LogEnabled;
import org.apache.avalon.framework.logger.Logger;
import org.apache.avalon.framework.thread.ThreadSafe;
import org.apache.cocoon.Constants;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.util.StringUtils;
import org.apache.cocoon.xml.XMLizable;
import org.apache.excalibur.source.SourceUtil;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import java.io.*;
import java.net.MalformedURLException;
import java.net.URL;
import java.rmi.RemoteException;
import java.rmi.registry.LocateRegistry;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Locale;

/**
 * The SDX framework represents a working installation of SDX within an execution container.
 * <p>
 * The framework serves two main purposes : (1) handle global configuration parameters and
 * (2) make SDX applications visible and usable.
 * <p>
 * An execution container is responsible for building one and only one SDX framework, and
 * then make it available. It must be built before any call to other SDX methods via any API.
 * For now, only a servlet container is supported, and for this container the building
 * of the framework occurs at the servlet container's startup. This could change in the near future.
 */

/*
Implementation Info :

Framework - class must implement parent interface that defines role
Component - class must implement it if is to be a component
ComponentSelector - gives this object the ability to select components of the same ROLE, but are of different use (needed for
    communication between sdx apps
Contextualizable - allows us to acquire a Cocoon 2 Context
Recomposable - added to the class implementation list because it allows this class to select components within the parent
    component manager (CM), Cocoon's CM, this will be necessary when a Framework wants to get a local or remote pooled
    DataSource component from the parent CM, to be recomposed if necessary if the system is changed;
Configurable - allows configuration within parent CM
Configuration - allows configuration by via XML file
Reconfigurable - allows configuration within parent CM if params are changed by user
Initializable - interface is implemented to allow the Framework component to create new datasource components
(I assume this is allowable via web interface)
Disposable - self explanatory
Startable - self explanatory
Suspendable - The Suspendable interface is used when a component will need to temporarily
    halt execution of a component. The execution may be halted so that you can
    reconfigure/ recompose/recontextualize component.
Parameterizable - needed if and when we move to a more configurable system of allowing users to specify config file placement
RequestLifecycleComponent - Objects implementing this marker interface have a lifecycle of one acceptRequest. This means if one acceptRequest
    is looking up several times an object implementing this interface, it's always the same object. In addition, the first time
    this object is looked up during a acceptRequest, the setup() method is called
Poolable - needed to manage instances of our component, see http://xml.apache.org/cocoon/developing/avalon.html (subheading: Pooling configuration) for more info.
LogEnable - new interface for logging

For the order in which the methods from these implementations are honored, please see:
http://jakarta.apache.org/avalon/framework/reference-the-lifecycle.html

-rbp29/03/02
*/


/**Core unit of SDX, loads applications, pipelines, etc.
 *TODOJavadoc: better documentation-rbp
 */
public class FrameworkImpl implements Framework, Component, ComponentSelector,
        Contextualizable, Composable, Configurable, /*Parameterizable,*/ Reconfigurable, Initializable,
        Disposable, Startable, Suspendable, LogEnabled, XMLizable, ThreadSafe {

    /************
     Class members
     ************/

    /** Avalon logger to write information. */
    private org.apache.avalon.framework.logger.Logger logger;
    /** The component manager. */
    private ComponentManager manager;
    /** The framework's configuration object.*/
    private Configuration sdxConf;
    /** The framework's context. */
    private Context context;
    /** The servlet "context directory" : file:/{TOMCAT_HOME}/webapps/{sdx}/. */
    private String contextPath;
    /** The directory for dynamic library loading : file:/{TOMCAT_HOME}/webapps/{sdx}/WEB-INF/lib/. */
    private String libPath;
    /** The directory for this framework's configuration : file:/{TOMCAT_HOME}/webapps/{sdx}/WEB-INF/{sdx}/. */
    private String sdxConfPath;
    /** The framework's working directory : {sdx}. */
    private File workDir = null;
    /**The upload dir for the framework within the workDir*/
    private File context_upload_dir = null;
    /** The framework's locale (found either in the configuration file or by a call to </code>Locale.getDefault()</code>). */
    private Locale locale;
    /**The xml:lang attribute value which has been passed to this framework's configuration. */
    private String xmlLang = "";  //TODO : why retain it ? what does it bring more to locale ? -pb
    /** The list of registered applications (by path names). */
    private Hashtable registeredAppsByPath;
    /** The list of registered applications (by ids). */
    private Hashtable registeredAppsById;
    /** The list of pipelines (by ids). */
    private Hashtable pipes; //TODOpb : understand what those pipelines can be used for :-)
    /** Whether the super-user has been set or not. */
    private boolean isSuperUserSet = false;
    /** The super-user's password. */
    private String suEncryptedPasswd = null;
    /** Whether the RMI registry exists or not.*/
    private boolean isRmiRegistryCreated = false;
    /** The path to a security policy file for the jvm if rmi is to be used*/
    //private String securityPolicy = ""; //no longer needed, but maybe in the future
    /** The RMI host name : machine name or IP address.*/
    private String rmiHost = SDX_DEFAULT_RMI_HOST;
    /** The RMI port.*/
    private int rmiPort = SDX_DEFAULT_RMI_PORT;
    /** The manager for analyzers. */
    private AnalyzerManager analyzerMgr = new AnalyzerManager();
    /**Default encoding value to be used for application-wide encoding functions*/
    private String encoding = DEFAULT_ENCODING;


    /***********************
     Directory and file names
     ***********************/

    /** The servlet-engine relative path to the framework's configuration directory : WEB-INF/{sdx}/. */
    private final String CONFIGURATION_DIR_PATH = "WEB-INF" + File.separator + "sdx" + File.separator;
    /** The directory where applications are registered : WEB-INF/{sdx}/applications/. */
    private File sdxAppsDir = null; //TODO : make a constant of it as it is determinist ? -pb
    /** The framework's configuration filename. */
    public static final String CONFIGURATION_FILE_NAME = "sdx.xconf";
    /** The directory name where framework's applications are registered. */
    private final String APPLICATIONS_DIR = "applications";
    /** The directory name where application configuration files reside. */
    private final String APPLICATION_CONFIGURATION_DIRECTORY = "conf"; //TODO : move to Application.java ? -pb
    /** The configuration file name of an application. */
    public static final String APP_CONFIG_FILENAME = "application.xconf"; //TODO : move to Application.java ? -pb
    /** The servlet-engine relative path to the framework's librairies : WEB-INF/lib/. */
    private final String LIBRARY_DIR_PATH = "WEB-INF" + "/" + "lib" + "/";

    /**************************************************************
     String representation of the keys used by the properties object

     TODO :
     Most of these keys seem to be redundant with class members...
     I know that these keys may be used by other classes to access properties, but, IMHO
     we should offer accessors to this properties rather than :
     1) expose the properties object to other classes : it should be "protected"
     2) expose keys to other classes : they alos should be "protected"
     Wha't your mind about tihs? -pb
     **************************************************************/

    /** String representation for a key in the Properties object : framework's configuration file. */
    public static final String SDX_CONF = "sdxConfFile";
    /** String representation for a key in the Properties object : framework's applications names. */
    public static final String APP_PATH_NAME = "appPathName";
    /** String representation for a key in the Properties object : framework's applications configuration directories. */
    public static final String APP_CONF_PATH = "appConfDirPath";
    /** String representation for a key in the Properties object :  framework's applications configurations. */
    public static final String APP_CONF = "appConf";
    /** String representation for a key in the Properties object : framework's context directory. */
    public static final String SDX_APP_PATH = "cocoonAppPath"; //TODO : mmmh, somewhat misleading... -pb
    /** String representation for a key in the Properties object : framework's configuration directory. */
    public static final String SDX_CONF_PATH = "sdxConfPath";
    /** String representation for a key in the Properties object :  framework's libraries. */
    public static final String LIB_PATH = "libPath";
    /** String representation for a key in the Properties object : security policy. */
    public static final String SECURITY_POLICY = "securityPolicy";
    /** String representation for a key in the Properties object : famework's RMI host. */
    public static final String RMI_HOST = "rmiHost";
    /** String representation for a key in the Properties object : famework's RMI port. */
    public static final String RMI_PORT = "rmiPort";
    /** String representation for a key in the Properties object :  framework's analyzer manager. */
    public static final String ANALYZER_MGR = "analyzerMgr";

    /*************************************************************************
     Child element names of the configuration element in the configuration file
     *************************************************************************/

    /**Attribute name for configuration file/also used as hashtable key*/
    public static final String ATTRIBUTE_NAME_ENCODING = "encoding";
    private final String ATTRIBUTE_USE = "use"; //TODO : apparently for entity resolver. Move it there ! -pb
    /** The element used define the use of RMI. */
    private final String ELEMENT_NAME_RMI = "rmi-registry";
    /** The implied attribute used to define RMI policy file. */
    //private String ATTRIBUTE_NAME_SECURITY = "security"; //no longer needed, but maybe in the future
    /** The implied attribute used to define RMI host. */
    private String ATTRIBUTE_NAME_PORT = "port";
    /** The implied attribute used to define RMI port. */
    private String ATTRIBUTE_NAME_HOST = "host";

    /********************
     Useful default values
     ********************/

    /** The file name in which is stored the super-user's info. */
    private String SUPER_USER_FILE_NAME = "su"; //TODO : As it is used right now, it is final -pb
    /** Default RMI policy file*/
    private final String SDX_DEFAULT_SECURITY_POLICY = "rmi.policy"; //TODO : make it static ? -pb
    /** Default RMI host*/
    public static final String SDX_DEFAULT_RMI_HOST = "localhost";
    /** Default RMI port*/
    public static final int SDX_DEFAULT_RMI_PORT = 9000;


    /** Sets the logger for this framework
     *
     * @param logger   The logger.
     */
    public void enableLogging(org.apache.avalon.framework.logger.Logger logger) {
        /*when our component is loaded, we take the logger which Coocoon assigns to our component based upon it's
        defaults and logkit.xconf and set it here, so we can retrieve it when necessary or pass it to other classes*/
        this.logger = logger;
    }

    /**
     * Contextualize this class.
     *
     * @param context   The context provided by Cocoon.
     * @throws ContextException
     */
    public void contextualize(Context context) throws ContextException {
        if (context == null) throw new ContextException("Context provided was null", null);

        this.context = context;
        this.context_upload_dir = (File) this.context.get(Constants.CONTEXT_UPLOAD_DIR);
        //getting a working directory
        if (this.workDir == null) {
            //getting the default one
            this.workDir = (File) context.get(Constants.CONTEXT_WORK_DIR);
        }

        //if none was provided, get a context path
        org.apache.cocoon.environment.Context ctx =
                (org.apache.cocoon.environment.Context) context.get(Constants.CONTEXT_ENVIRONMENT_CONTEXT);

        // Determine the context directory, preferably as a file
        // FIXME (SW) - this is purposely redundant with some code in CocoonServlet
        // to have the same rootPath. How to avoid this ?
        try {
            String rootPath = ctx.getRealPath("/");
            if (rootPath != null) {
                this.contextPath = new File(rootPath).toURL().toExternalForm();
            } else {
                String webInf = ctx.getResource("/WEB-INF").toExternalForm();
                this.contextPath = webInf.substring(0, webInf.length() - "WEB-INF".length());
            }
        } catch (MalformedURLException e) {
            Utilities.logWarn(logger, "Could not get context directory", e);
            this.contextPath = "";
        }

        this.libPath = this.contextPath + LIBRARY_DIR_PATH;
    }

    /**
     * Sets the framework ComponentManager.
     *
     * @param manager  The ComponentManager. Provided by Cocoon.
     * @throws ComponentException
     */
    public void compose(ComponentManager manager) throws ComponentException {
        if (manager == null) {
            SDXException sdxE = new SDXException(null, SDXExceptionCode.ERROR_COMPONENT_MANAGER_NULL, null, null);
            Utilities.logWarn(logger, sdxE.getMessage(), sdxE);
        }
        this.manager = manager;
    }

    /**
     * Configures the framework.
     *
     * @param configuration  The configuration object provided at startup by Cocoon, using the <sdx-framework> element in cocoon.xconf.
     * @throws ConfigurationException
     */
    public void configure(Configuration configuration) throws ConfigurationException {
        Utilities.checkConfiguration(configuration);
        /*We must configure the framework itself, but this actually takes place in the intialize() method as we only allow Cocoon to honor
        the LogEnabled implementation by assigning us a logger through its use of the Avalon LogKit architecture in /WEB-INF/logkit.configuration file.*/
        /*
            TODO: Things to configure for the framework :
                - the name and description, in multiple languages
                - preferred language?
                - some kind of security
        */
    }

    private void configureApplications() throws SDXException {
        /*
         * creating the path to the directory which contains files for registering applications
         * within the framework
         * should be similar to {TOMCAT_HOME}/webapps/{sdx | cocoonAppName}/WEB-INF/sdx/applications
         */
        String sdxAppsPath = this.sdxConfPath + APPLICATIONS_DIR;
        //testing the directory for the path, to ensure it is available and we have access to it
        this.sdxAppsDir = Utilities.checkDirectory(sdxAppsPath, logger);

        /*
        * retrieving a list of files in the "applications" directory
        * those filenames are *also* application names
        * this is *the* trick !
        * TODO : what is the rationale behind this trick ? Can't have an XML file ? -pb
        *TODORemoveQ&A: it is easier to add and remove applications based upon a empty file rather than an xml file-rbp
        */
        String[] appNames = this.sdxAppsDir.list();
        //if we don't have any apps to configure we throw an except.
        if (appNames == null || appNames.length == 0) {
            String[] args = new String[1];
            args[0] = this.sdxAppsDir.getAbsolutePath();
            throw new SDXException(logger, SDXExceptionCode.ERROR_NO_APPS_FOR_CONFIG, args, null);
        }

        //ensuring we have the appropriate hashtables to work with
        //TODO : instantiate those Hashtable when they are declared ? -pb
        if (registeredAppsById == null) registeredAppsById = new Hashtable();
        if (registeredAppsByPath == null) registeredAppsByPath = new Hashtable();

        Utilities.logInfo(logger, "\tConfiguring Applications...");
        //iterating over the applications' list, should never be null, but rather empty (size zero) because we create the directory if it doesn't exist
        //TODO : not clear : what should not be null ? what is the behaviour when "it" is empty ? -pb
        //TODORemoveQ&A: i was referring to the string array "appNames"
        for (int i = 0; i < appNames.length; i++) {
            try {
                configureApplication(appNames[i]);
            } catch (SDXException e) {
                //log a message saying application config failed
                //by itself, the creation of the SDXException should log this kind of message
                //we don't want all application configurations to fail, so we won't throw this farther out
            }
        }
    }

    private synchronized void configureApplication(String appPath) throws SDXException {
        /*
         * ensuring the existence of application's directory
         * should be similar to {TOMCAT_HOME}/webapps/{sdx | cocoonAppName}/{myAppName}/
         */
        String appDirPath = this.contextPath + appPath + File.separator;
        File appDir = new File(appDirPath);
        if (!appDir.exists()) {
            //there is no directory corresponding to the appName provided from the registration directory
            String[] args = new String[1];
            args[0] = appDirPath;
            //by itself, the creation of the SDXException should log the message
            throw new SDXException(logger, SDXExceptionCode.ERROR_NO_APP_DIRECTORY, args, null);
        }

        /*
        * creating a path to the directory in which the application's specific configuration file should exist
        * should be similar to {TOMCAT_HOME}/webapps/{sdx | cocoonAppName}/{myAppName}/conf/
        */
        String appConfDirPath = this.contextPath + appPath + File.separator + APPLICATION_CONFIGURATION_DIRECTORY + File.separator;
        //testing the directory for the path, to ensure it is available and we have access to it
        Utilities.checkDirectory(appConfDirPath, logger);
        //String confFilePath = appConfDirPath + APP_CONFIG_FILENAME + CONFIGURATION_FILE_EXTENSION;

        /*Parameterizable
        * the one line below will allow us to make an sdx app highly configurable to the user as to where they may place their
        * application specific configuration files, the path value of the confFile attribute in the sdx.xconf must be a path relative
        * to the sdx installation "myapp/blah/blahh/app.xml" or "myconfigfiles/app1.xml"-rbp11/04/02
        */
        //String confFilePath = contextPath + appConfig.getAttribute("confFile");

        /*
        * creating a path to the application specific configuration file
        * should be similar to {TOMCAT_HOME}/webapps/{sdx | cocoonAppName}/{myAppName}/conf/application.xconf
        */
        String appConfFilePath = appConfDirPath + APP_CONFIG_FILENAME;
        //creating a file object from the path
        File appConfFile = new File(appConfFilePath);
        Utilities.logInfo(logger, "\tLoading Application Configuration file from :  " + appConfFile.toString());
        if (!appConfFile.exists()) {
            String[] args = new String[2];
            args[0] = appPath;
            args[1] = appConfFilePath;
            //by itself, the creation of the SDXException should log the message
            throw new SDXException(logger, SDXExceptionCode.ERROR_NO_APP_CONFIG_FILE, args, null);
        }
        //building a configuration object
        DefaultConfigurationBuilder appConfigBuild = new DefaultConfigurationBuilder(true);
        //populating the configuration object with the sdx.xconf configuration file
        Configuration appConf = null;
        try {
            appConf = appConfigBuild.buildFromFile(appConfFile);
        } catch (Exception e) {
            String[] args = new String[2];
            args[0] = appPath;
            args[1] = appConfFile.getAbsolutePath();
            //by itself, the creation of the SDXException should log the message
            throw new SDXException(logger, SDXExceptionCode.ERROR_APP_CONFIG_FILE_READ, args, e);
        }

        if (appConf == null) {
            String[] args = new String[2];
            args[0] = appPath;
            args[1] = appConfFile.getAbsolutePath();
            //by itself, the creation of the SDXException should log the message
            throw new SDXException(logger, SDXExceptionCode.ERROR_APP_CONFIG_FILE_READ, args, null);
        }
        /*if the rmi registry is not already created and the appConf object has any document bases
        available for remote searching, we create the rmi registry*/
        if (!this.isRmiRegistryCreated) createRmiRegistry(appConf);
        //creating application's properties object
        Hashtable props = getProperties();
        //adding the application's name to application's properties
        props.put(APP_PATH_NAME, appPath);
        //adding the path information to application's properties
        props.put(APP_CONF_PATH, appConfDirPath);
        //adding the application configuration object to the properties object for use if needed withing the application configuration
        props.put(APP_CONF, appConf);
        try {
            //creating a new application object
            Application app = new Application();
            //setting the logger for the application object
            app.enableLogging(logger.getChildLogger(appPath));
            //setting the context
            app.contextualize(this.context);
            //passing component manager (actually belonging to Cocoon) to the application object
            app.compose(this.manager);
            //passing the application's properties to the application object
            app.setProperties(props);
            //TODO : shouldn't the below message be delegated to Application class ? -pb
            Utilities.logInfo(logger, "\tConfiguring application..." + app.toString());
            //configuring the application with the configuration object build from application.xconf
            app.configure(appConf);
            //initializing the application
            app.init();
            //adding the application object to the framework's data structures
            addApplication(app);
            //TODO : review message ? something like "sucessfully added to framework" -pb
            Utilities.logInfo(logger, "\tApplication Configuration Successful, the application, \"" + app.getId() + "\", was configured !");
        } catch (ComponentException e) {
            String[] args = new String[1];
            args[0] = appDirPath;
            throw new SDXException(logger, SDXExceptionCode.ERROR_CONFIGURE_APP, args, e);
        } catch (ConfigurationException e) {
            String[] args = new String[1];
            args[0] = appDirPath;
            throw new SDXException(logger, SDXExceptionCode.ERROR_CONFIGURE_APP, args, e);
        } catch (ContextException e) {
            String[] args = new String[1];
            args[0] = appDirPath;
            throw new SDXException(logger, SDXExceptionCode.ERROR_CONFIGURE_APP, args, e);
        }

    }

    private void configurePipelines() throws SDXException {
        //getting an array of configuration objects from each of the <sdx:pipeline> subElements of <sdx:pipelines> element
        //safe code here : if the element "pipelines" does not exist, it is created in memory, so never null (source : Avalon javadocs)
        Configuration[] pipesConfList = sdxConf.getChild(Pipeline.ELEMENT_NAME_PIPELINES).getChildren(Pipeline.ELEMENT_NAME);

        /*We no longer require the single pipeline for results transformations
        //testing if we have something
        if (pipesConfList == null || pipesConfList.length == 0) {
            String[] args = new String[1];
            //getting the location of the configuration file
            args[0] = sdxConf.getLocation();
            //by itself, the creation of the SDXException should log the message
            throw new SDXException(logger, SDXExceptionCode.ERROR_NO_PIPELINES_IN_CONFIG, args, null);
        } */

        //ensuring we have the appropriate hashtables to work with
        //TODO : instantiate those Hashtable when they are declared ? -pb
        if (pipes == null) pipes = new Hashtable();

        if (pipesConfList != null) {
            //iterating over the applications' list, should never be null, but rather empty because we create the directory if it doesn't exist
            //TODO : see comment above regarding null/empty -pb
            //TODORemoveQ&A: bad copy paste, sorry:(-
            for (int i = 0; i < pipesConfList.length; i++) {
                Utilities.logInfo(logger, "\tConfiguring Pipelines...");
                try {
                    //reading the pipeline id attribute
                    String pipeId = pipesConfList[i].getAttribute(Pipeline.ATTRIBUTE_ID);
                    //checking the configuration value
                    //TODO : isn't it the *id* ? -pb
                    //TODORemoveQ&A:yes but we verify that it is not an empty string-rbp
                    Utilities.checkConfAttributeValue(Pipeline.ATTRIBUTE_ID, pipeId, pipesConfList[i].getLocation());
                    //reading the pipeline type attribute
                    String pipeType = pipesConfList[i].getAttribute(Pipeline.ATTRIBUTE_TYPE);
                    //checking the configuration value
                    Utilities.checkConfAttributeValue(Pipeline.ATTRIBUTE_TYPE, pipeType, pipesConfList[i].getLocation());
                    //building the fully qualified class name based on the type
                    String pipeClassName = Pipeline.PACKAGE_QUALNAME + pipeType.substring(0, 1).toUpperCase() + pipeType.substring(1, pipeType.length()).toLowerCase() + Pipeline.CLASS_NAME_SUFFIX;
                    //getting the class from the class name
                    Class pipeClass = null;
                    try {
                        pipeClass = Class.forName(pipeClassName);
                    } catch (ClassNotFoundException e) {
                        //logging the first failure
                        String[] args = new String[2];
                        args[0] = pipesConfList[i].getAttribute(Pipeline.ATTRIBUTE_ID);
                        args[1] = e.getMessage();
                        //by itself, the creation of the SDXException should log the message
                        SDXException sdxE = new SDXException(null, SDXExceptionCode.ERROR_CONFIGURE_PIPELINE, args, e);
                        Utilities.logWarn(logger, sdxE.getMessage(), null);
                        //trying again, with the attribute's value, hopefully the user is providing a fully qualified attribute name
                        pipeClassName = pipeType;
                        try {
                            pipeClass = Class.forName(pipeClassName);
                        } catch (ClassNotFoundException e1) {
                            String[] args2 = new String[2];
                            args2[0] = pipesConfList[i].getAttribute(Pipeline.ATTRIBUTE_ID);
                            args2[1] = e1.getMessage();
                            //by itself, the creation of the SDXException should log the message
                            throw new SDXException(logger, SDXExceptionCode.ERROR_CONFIGURE_PIPELINE, args2, e1);
                        }
                    }
                    //building a new instance of a Pipeline object
                    Object obj = pipeClass.newInstance();
                    if (obj == null) {
                        String[] args = new String[1];
                        args[0] = pipeClassName;
                        throw new SDXException(logger, SDXExceptionCode.ERROR_NEW_OBJECT_INSTANCE_NULL, args, null);
                    }
                    //testing to see wether the object implements the fr.gouv.culture.sdx.pipeline.Pipeline Interface
                    if (!(obj instanceof Pipeline)) {
                        //the object doesn't implement our interface
                        String[] args = new String[3];
                        args[0] = "Pipeline";
                        args[1] = pipeClass.getName();
                        args[2] = pipeType;
                        //by itself, the creation of the SDXException should log the message
                        throw new SDXException(logger, SDXExceptionCode.ERROR_CLASS_NOT_INSTANCEOF_SDX_INTERFACE, args, null);
                    }
                    //the object does implement our interface
                    //casting it into a DocumentBase object
                    Pipeline pipe = (Pipeline) obj;
                    //setting the DB's logger
                    pipe.enableLogging(logger.getChildLogger(pipeId));
                    //passing component manager (actually Cocoon's one) to the DB
                    //this also allows us to get our framework component
                    pipe.compose(manager);
                    //configuring the DB
                    pipe.configure(pipesConfList[i]);
                    //adding the pipe to the list
                    pipes.put(pipe.getId(), pipe);
                    Utilities.logInfo(logger, "\tPipeline Configuration Successful, the application, \"" + pipe.getId() + "\", was configured !");
                } catch (InstantiationException e) {
                    String[] args = new String[2];
                    args[0] = pipesConfList[i].getLocation();
                    args[1] = e.getMessage();
                    //by itself, the creation of the SDXException should log the message
                    new SDXException(logger, SDXExceptionCode.ERROR_CONFIGURE_PIPELINE, args, e);
                    //we don't want all document base configurations to fail, so we won't throw this farther out
                } catch (IllegalAccessException e) {
                    String[] args = new String[2];
                    args[0] = pipesConfList[i].getLocation();
                    args[1] = e.getMessage();
                    //by itself, the creation of the SDXException should log the message
                    new SDXException(logger, SDXExceptionCode.ERROR_CONFIGURE_PIPELINE, args, e);
                    //we don't want all document base configurations to fail, so we won't throw this farther out
                } catch (ComponentException e) {
                    String[] args = new String[2];
                    args[0] = pipesConfList[i].getLocation();
                    args[1] = e.getMessage();
                    //by itself, the creation of the SDXException should log the message
                    new SDXException(logger, SDXExceptionCode.ERROR_CONFIGURE_PIPELINE, args, e);
                    //we don't want all document base configurations to fail, so we won't throw this farther out
                } catch (ConfigurationException e) {
                    //log a message saying application config failed
                    Utilities.logException(logger, e);
                    //we don't want all document base configurations to fail, so we won't throw this farther out
                } catch (SDXException e) {
                    //log a message saying application config failed
                    //by itself, the creation of the SDXException should log this kind of message
                    //we don't want all document base configurations to fail, so we won't throw this farther out
                }
            }
        }
    }

    private void createRmiRegistry(Configuration configuration) throws SDXException {

        //scanning the application configuration for any "remote-access" elements
        Configuration[] docBases = configuration.getChild(DocumentBase.ELEMENT_NAME_DOCUMENT_BASES).getChildren(DocumentBase.ELEMENT_NAME_DOCUMENT_BASE);
        boolean remoteDocBase = false;
        for (int i = 0; i < docBases.length; i++) {
            Configuration docBas = docBases[i];
            if (docBas.getAttributeAsBoolean(LuceneDocumentBase.DBELEM_ATTRIBUTE_REMOTE_ACCESS, false)) {
                remoteDocBase = docBas.getAttributeAsBoolean(LuceneDocumentBase.DBELEM_ATTRIBUTE_REMOTE_ACCESS, false);
            }
        }

        //if we have a valid configuration from one of the documentbase configurations we create the registry
        if (remoteDocBase) {
            //the following code should be done once and only once if there are multiple remote indices declarations
            //this should be done only once, but this with two documentbases that have remote-access elements-rbp
            //when tomcat is loaded it will load the policy file, see tcat/conf/catalina.policy-rbp
            /*dont wan't to do this anymore as it causes problems upon tomcat loading
            if (!Utilities.checkString(System.getField("java.security.policy"))) {
                if (Utilities.checkString(this.securityPolicy))
                    System.setProperty("java.security.policy", this.securityPolicy);
                //System.getProperties().put("java.security.policy", rmiPolicy);
            }
            */
            //checking for the tomcat security manager
            if (System.getSecurityManager() == null) {
                //tomcat wasn't started with the security option
                /*TODO?:should we fail completely here on configuration
                 or simply log a failure message saying that remote objects will not be available*/
                //i think we should fail, this prevents the user from having to restart the server again
                throw new SDXException(logger, SDXExceptionCode.ERROR_NO_SECURITY_MANAGER, null, null);
            }

            try {

                if (Utilities.checkString(this.rmiHost)) {
                    //setting the rmi server host name if it isn't already set
                    if (!Utilities.checkString(System.getProperty("java.rmi.server.hostname")))
                        System.setProperty("java.rmi.server.hostname", this.rmiHost);
                    else {//we already have a hostname
                        if (!this.rmiHost.equalsIgnoreCase(System.getProperty("java.rmi.server.hostname"))) {
                            //we need a message saying that the two host names do not match
                            String[] args = new String[3];
                            args[0] = "java.rmi.server.hostname";
                            args[1] = System.getProperty("java.rmi.server.hostname");
                            args[2] = rmiHost;
                            SDXException sdxE = new SDXException(null, SDXExceptionCode.ERROR_SYSTEM_PROPERTY, args, null);
                            Utilities.logWarn(logger, null, sdxE);
                        }
                    }
                }

                String rmiCodebase = this.libPath;
                if (Utilities.checkString(rmiCodebase)) {
                    //we set the code base if it isn't already set,ad we have a good value
                    if (!Utilities.checkString(System.getProperty("java.rmi.server.codebase"))) {
                        System.setProperty("java.rmi.server.codebase", rmiCodebase);
                        System.setProperty("java.rmi.server.useCodebaseOnly", "true");
                    } else {//we have a code base already
                        if (!rmiCodebase.equalsIgnoreCase(System.getProperty("java.rmi.server.codebase"))) {
                            //we need a message saying that the two code base's do not match
                            String[] args = new String[3];
                            args[0] = "java.rmi.server.codebase";
                            args[1] = System.getProperty("java.rmi.server.codebase");
                            args[2] = rmiCodebase;
                            SDXException sdxE = new SDXException(null, SDXExceptionCode.ERROR_SYSTEM_PROPERTY, args, null);
                            Utilities.logWarn(logger, null, sdxE);
                        }
                    }
                }
            } catch (SecurityException e) {
                //we don't have the appropriate permissions to do these, should fail and pass message along telling some one to edit the tomcat policy file
                throw new SDXException(logger, SDXExceptionCode.ERROR_SYSTEM_PRIVILEGES, null, e);
            }

            try {
                LocateRegistry.createRegistry(this.rmiPort);
                this.isRmiRegistryCreated = true;
            } catch (RemoteException e) {
                throw new SDXException(logger, SDXExceptionCode.ERROR_CREATE_RMI_REGISTRY, null, e);
            }


        }

    }

    /** Reconfigures the component
     *
     * @param conf  Currently a <code>null</code> value is supported and this calls
     * re-initializes the Framework by calling intialize(), USE WITH CAUTION
     * TODO : please explain -pb
     *@see #initialize
     *
     */
    public void reconfigure(Configuration conf) throws ConfigurationException {
        //TODO?:can we make this better?-rbp
        if (conf == null) {
            try {
                this.contextualize(this.context);
                this.initialize();
            } catch (Exception e) {
                throw new ConfigurationException(e.getMessage(), e);
            }
        }
    }

    /**Reconfigures and application based upon an id
     *
     * @param appId The id of the application to be reconfigured
     */
    public void reconfigureApplication(String appId) throws SDXException {
        if (Utilities.checkString(appId)) {
            //getting the application for which the reconfigure is desired
            Application app = this.getApplicationById(appId);
            if (app != null) {
                String appPath = app.getPath();
                //removing the application from the data structures
                app = (Application) this.registeredAppsById.remove(appId);
                app = (Application) this.registeredAppsByPath.remove(appPath);
                app = null;
                //System.runFinalization();
                //System.gc();
                //re-adding the application
                this.addApplication(appPath);
            }
        }
    }

    /*Parameterizable
    * the code lines below will allow us to make an sdx app configurable to the user as to where he or she may place there
    * sdx config file by creating an "<sdx-framework>" child element
    * <parameter name="sdx-config-file" value="webapps/mySdxIntallDirName/mypath....mysdx.xml"/>
    * within the their cocoon.xconf (the value attribute must be a path relative to the sdx installation
    * "myapp/blah/blahh/app.xml" or "myconfigfiles/app1.xml"), if we implement this we must test relative paths
    * like "../blah/myconfigfile.xml"-rbp11/04/02Configure this class with parameters from the sdxconfig file-rbp08/04/02
    */
    /*
    public void parameterize(Parameters params) {
        //setting our default configuration file location
        //this.sdxConfPath = (FileUtil.toFile(new URL(contextPath))).getCanonicalPath() + File.separator;
        //this.sdxConfPath = params.getParameter("sdx-config-file", this.sdxConfPath);
    }
    */

    /**
     * Initializes the framework and builds the necessary application, pipeline, and analyzer manager objects.
     *
     * @throws Exception
     */
    public void initialize() throws Exception {
        //calling the internal synchronized init
        this.init();
    }

    /**
     * Initializes the framework and builds the necessary application, pipeline, and analyzer manager objects.
     *
     * @throws Exception
     */
    private synchronized void init() throws Exception {
        //resetting the contextPath to ensure we have a cross-platform compliant assignment of the path
        //should be similar to {TOMCAT_HOME}/webapps/{sdx | cocoonAppName}/
        this.contextPath = (FileUtil.toFile(new URL(this.contextPath))).getCanonicalPath() + File.separator;
        Utilities.logInfo(logger, "\tContext directory is " + this.contextPath);

        /* Parameterizable
        * the one line below helps use enforce relative path format if we are to move to a more configurable method of allowing
        * users to place config files-rbp11/04/02, but in what path format?
    * this.sdxConfPath = this.contextPath
        */

        /*
        * setting the sdx configuration path webapps
        * should be similar to {TOMCAT_HOME}/webapps/{sdx | cocoonAppName}/WEB-INF/sdx/
        */
        this.sdxConfPath = this.contextPath + CONFIGURATION_DIR_PATH;
        //testing the directory for the configuration path, to ensure it is available and we have access to it
        Utilities.checkDirectory(this.sdxConfPath, logger);

        /*
        * creating the path to the base configuration file
        * should be similar to {TOMCAT_HOME}/webapps/{sdx | cocoonAppName}/WEB-INF/sdx/sdx.xconf
        */
        String sdxConfFilePath = this.sdxConfPath + CONFIGURATION_FILE_NAME;
        //creating a file object from the path
        File sdxConfFile = new File(sdxConfFilePath);
        //verifying we have a file to work with
        if (!sdxConfFile.exists()) {
            String[] args = new String[2];
            args[0] = CONFIGURATION_FILE_NAME;
            args[1] = this.sdxConfPath;
            //by itself, the creation of the SDXException should log the message
            throw new SDXException(logger, SDXExceptionCode.ERROR_NO_SDX_CONFIG_FILE, args, null);
        } else {
            //if we have it, we proceed
            Utilities.logInfo(logger, "\tLoading Framework Configuration file :  " + sdxConfFilePath);
            Utilities.logInfo(logger, "\tConfiguring framework... : " + this.toString());

            //building a configuration object
            //passing "true" to allow namespace support
            DefaultConfigurationBuilder frameConfigBuild = new DefaultConfigurationBuilder(true);
            //populating the config object with the sdx.xconf configuration file
            try {
                this.sdxConf = frameConfigBuild.buildFromFile(sdxConfFile);
            } catch (Exception e) {
                String[] args = new String[1];
                args[0] = sdxConfFile.getAbsolutePath();
                //by itself, the creation of the SDXException should log the message
                throw new SDXException(logger, SDXExceptionCode.ERROR_SDX_CONFIG_FILE_READ, args, e);
            }
            /* old code no longer necessary with new application registering process, but will be useful in the near future
            //at this point, we have an array of "application" elements from sdx.xconf
            Configuration[] appElems = sdxConf.getChildren();
            */
            Configuration rmiConf = sdxConf.getChild(ELEMENT_NAME_RMI, false);
            if (rmiConf != null) {
                int port = rmiConf.getAttributeAsInteger(ATTRIBUTE_NAME_PORT, SDX_DEFAULT_RMI_PORT);
                String host = rmiConf.getAttribute(ATTRIBUTE_NAME_HOST, SDX_DEFAULT_RMI_HOST);
                //if value's were provided we verify that they are non empty strings
                Utilities.checkConfAttributeValue(ATTRIBUTE_NAME_HOST, this.rmiHost, rmiConf.getLocation());
                Utilities.checkConfAttributeValue(ATTRIBUTE_NAME_PORT, Integer.toString(this.rmiPort), rmiConf.getLocation());
                this.rmiPort = port;
                this.rmiHost = host;
                /*no longer needed, but maybe in the future
                String security = rmiConf.getAttribute(ATTRIBUTE_NAME_SECURITY, SDX_DEFAULT_SECURITY_POLICY);
                Utilities.checkConfAttributeValue(ATTRIBUTE_NAME_SECURITY, security, rmiConf.getLocation());
                //resolving the policy file
                File policy = Utilities.resolveFile(logger, rmiConf.getLocation(), getProperties(), security, false);
                //saving a pointer to the policy file
                this.securityPolicy = FileUtil.removeExtension(policy.toURL().toExternalForm());
                */

            }

            this.xmlLang = sdxConf.getAttribute(Node.Name.XML_LANG, null);
            this.locale = Utilities.buildLocale(sdxConf, null);
            this.encoding = sdxConf.getAttribute(ATTRIBUTE_NAME_ENCODING, DEFAULT_ENCODING);

        }

        //buliding the analyzer manager, should be done before configuring applications as it will be needed at that time
        //setting the logger for the analyzer manager object
        analyzerMgr.enableLogging(this.logger);
        //passing component manager (actually Cocoon's one) to the analyzer manager object
        analyzerMgr.compose(manager);

        //building and configuring the applications
        try {
            configureApplications();
        } catch (SDXException e) {
            //by itself, the creation of the SDXException should log the message

        }

        try {
            configureSuperUser();
        } catch (SDXException e) {
            //by itself, the creation of the SDXException should log the message
        }

        //building and configuring the pipelines
        try {
            configurePipelines();
        } catch (SDXException e) {
            //by itself, the creation of the SDXException should log the message

        }

    }

    /** Starts the component, but currently has no function. */
    public void start() {
        //TODO:implement
    }

    /** Suspends the component, but currently has no function. */
    public void suspend() {
        //TODO:implement
    }

    /** Stops the component, but currently has no function. */
    public void stop() {
        //TODO:implement
    }

    /** Disposes the component, but currently has no function. */
    public void dispose() {
        //TODO:implement
    }

    /** Resumes the component after it has been suspended, but currently has no function. */
    public void resume() {
        //TODO:implement
    }

    /** Check to see if a Component exists for a hint, but currently has no function and only returns false. */
    public boolean hasComponent(java.lang.Object hint) {
        return false;
        //TODO:implement
    }

    /** Select the Component associated with the given hint, but currently has no function and only returns null. */
    public Component select(Object hint) throws ComponentException {
        return null;
        //TODO:implement
    }

    /** Releases the Component when we are finished with it. */
    public void release(Component component) {
    }

    private Hashtable getProperties() {
        //creating the properties object
        Hashtable props = new Hashtable();
        //adding the path to this Cocoon application
        props.put(SDX_APP_PATH, this.contextPath);
        //adding the upload dir path for this Cocoon application
        props.put(Constants.CONTEXT_UPLOAD_DIR, this.context_upload_dir);
        //adding the path to the sdx configuration dir
        props.put(SDX_CONF_PATH, this.sdxConfPath);
        //adding sdx' configuration file object to application's properties : useful for defaults
        props.put(SDX_CONF, sdxConf);
        //adding the path to the 'lib' directory
        props.put(LIB_PATH, this.libPath);
        //adding the analyzer manager to the properties
        //analyzer manager should never be null
        props.put(ANALYZER_MGR, this.analyzerMgr);
        //adding the default encoding
        props.put(ATTRIBUTE_NAME_ENCODING, this.encoding);
        //adding the rmi port
        if (new Integer(this.rmiPort) != null) props.put(RMI_PORT, new Integer(this.rmiPort));
        //adding the rmi host
        if (Utilities.checkString(this.rmiHost)) props.put(RMI_HOST, this.rmiHost);
        //adding the security policy
        //no longer needed, but maybe in the future
        //if (Utilities.checkString(this.securityPolicy)) props.put(SECURITY_POLICY, this.securityPolicy);
        return props;
    }

    /**Add's an application based upon a path (ie. directory name)
     * containing the application's configuration file, etc.
     *
     * @param appPath   The directory name for the application under the sdx installation
     *                  (example: sdxworld)
     */
    public synchronized void addApplication(String appPath) throws SDXException {
        if (Utilities.checkString(appPath)) {
            //create the empty file
            File appFile = new File(this.sdxAppsDir, appPath);

            try {
                if (appFile != null) appFile.createNewFile();
            } catch (IOException e) {
                String[] args = new String[1];
                args[0] = appFile.getAbsolutePath();
                throw new SDXException(logger, SDXExceptionCode.ERROR_CREATE_APP_FILE, null, e);
            }
            //configure the application
            this.configureApplication(appPath);
        }

    }

    /**
     * Saves an application in framework's data structures.
     *
     * @param app	The application object. Normally created during framework's initialization.
     */
    private void addApplication(Application app) throws SDXException {
        //we should always have an application object passed in
        //TODO: we need checks here, as the user may
        //adding value pairs to hashtables
        //we should always have a valid getPath() if we get this far in the configuration
        String appKey = app.getPath();
        if (this.registeredAppsByPath.containsKey(appKey)) {
            String[] args = new String[1];
            args[0] = "An application with the path name, " + appKey + ", already exists";
            throw new SDXException(logger, SDXExceptionCode.ERROR_GENERIC, args, null);
        }

        this.registeredAppsByPath.put(appKey, app);

        appKey = app.getId();
        if (this.registeredAppsById.containsKey(appKey)) {
            String[] args = new String[1];
            args[0] = "An application with the id, " + appKey + ", already exists";
            throw new SDXException(logger, SDXExceptionCode.ERROR_GENERIC, args, null);
        }
        //we should always have a valid getId() if we get this far in the configuration
        this.registeredAppsById.put(appKey, app);
    }

    /**Removes and application from the framework based upon an id
     *
     * @param appId
     */
    public synchronized void removeApplication(String appId) throws SDXException {
        if (Utilities.checkString(appId)) {
            //getting the application for which the reconfigure is desired
            Application app = this.getApplicationById(appId);
            if (app != null) {
                String appPath = app.getPath();
                this.registeredAppsById.remove(appId);
                this.registeredAppsByPath.remove(appPath);
                File appFile = new File(this.sdxAppsDir, appPath);
                if (appFile != null && appFile.isFile() && appFile.exists())
                    appFile.delete();
            }
        }
    }

    private void configureSuperUser() throws SDXException {
        try {
            getSuperUserInformationFromFile();
        } catch (SDXException sdxE) {
            /*TODO?:logging info here as the exception will be expected and startUp,because the file should not exist
             and it will be logged when it is created, also could be expected at reconfigure if someone has deleted
             the file?-rbp*/
            Utilities.logInfo(logger, sdxE.getMessage());
        }
    }

    /**
     * Gets the framework's logger.
     *
     * @return  The logger.
     */
    public Logger getLogger() {
        return logger;
    }

    /**Returns an Enumeration on the ids of the applications owned by this framework.
     *
     */
    public Enumeration getApplicationIds() {
        if (this.registeredAppsById != null) {
            return this.registeredAppsById.keys();
        } else
            return null;
    }

    /**
     * Gets an application identified by its id.
     *
     * @param id    The application's id.
     * @return      The application object.
     * @throws SDXException
     */
    public Application getApplicationById(String id) throws SDXException {
        Application app = (Application) registeredAppsById.get(id);
        if (app == null) {
            String[] args = new String[1];
            args[0] = id;
            //by itself, the creation of the SDXException should log the message
            throw new SDXException(logger, SDXExceptionCode.ERROR_UNKNOWN_APPLICATION, args, null);
        } else
            return app;
    }

    /**
     * Gets an application identified by its path.
     *
     * @param path    The application's path (the directory name for the application under the sdx installation, i.e. {TOMCAT_HOME}/webapps/{sdx}/{myApp}).
     * @return      The application object.
     * @throws SDXException
     */
    public Application getApplicationByPath(String path) throws SDXException {
        Application app = (Application) registeredAppsByPath.get(path);
        if (app == null) {
            String[] args = new String[1];
            args[0] = path;
            //by itself, the creation of the SDXException should log the message
            throw new SDXException(logger, SDXExceptionCode.ERROR_UNKNOWN_APPLICATION, args, null);
        } else
            return app;
    }

    /** Returns a new instance of the desired pipeline.
     *
     * @param id    The id of the desired query pipeline
     * @return
     */
    public Pipeline getPipeline(String id) throws SDXException {
        Pipeline pipe = (Pipeline) pipes.get(id);
        if (pipe == null) {
            String[] args = new String[1];
            args[0] = id;
            //by itself, the creation of the SDXException should log the message
            throw new SDXException(logger, SDXExceptionCode.ERROR_UNKNOWN_PIPELINE, args, null);
        } else
            return pipe.newInstance();
    }

    /**Set's the super user information
     *
     * @param initialUserId The user id,
     *                      <code>null</code> should be passed at initial set-up.
     * @param initialPasswd The existing super user password ,
     *                      <code>null</code> should be passed at initial set-up.
     * @param newUserId     The new user id.
     * @param newPasswd     The new password.
     * @param firstname     The first name
     * @param lastname      The last name
     * @param email         The email
     * @param xmlLang       A valid xml:lang attribute value
     * @throws SDXException
     */
    public void setSuperUser(String initialUserId, String initialPasswd, String newUserId, String newPasswd, String firstname, String lastname, String email, String xmlLang) throws SDXException {
        /*we need something such as public void setSuperuser(userid, initialPasswd, newPasswd) method in Framework, for setting
     userid/password for the superuser. The initial password should be the one encrypted in the passwd file if it exists, and
     null if not. An exception should be thrown if this method fails (it will fail if initialPasswd is not correct)*/
        if (!Utilities.checkString(newUserId))
            throw new SDXException(logger, SDXExceptionCode.ERROR_INVALID_NEW_SUPERUSER_ID, null, null);
        if (!Utilities.checkString(newPasswd))
            throw new SDXException(logger, SDXExceptionCode.ERROR_INVALID_NEW_SUPERUSER_PASSWORD, null, null);

        //verifying rights
        if (isSuperUserSet) validateSuperUser(initialUserId, initialPasswd);
        //otherwise we allow the sdx admin to create one

        //building strings
        String[] strings = new String[6];
        for (int i = 0; i < strings.length; i++) {
            //asssigning values
            if (i == 0) strings[i] = newUserId;
            if (i == 1) strings[i] = firstname;
            if (i == 2) strings[i] = lastname;
            if (i == 3) strings[i] = email;
            if (i == 4) strings[i] = xmlLang;
            if (i == 5) strings[i] = encryptPassword(newPasswd);

            //verifying values, so no strings are reported as "null"
            if (strings[i] == null) strings[i] = "";
        }

        //creating the string for our super user info file
        String data = Utilities.joinStrings(strings, ":");
        /*int lastColonIndex = 0;
        if (data.endsWith(":")) lastColonIndex = data.lastIndexOf(":");
        if (lastColonIndex > 0) data = data.substring(0, lastColonIndex);*/

        //if we have a good string we write the file
        if (!Utilities.checkString(data))
        //no good data, should rarely happen
            throw new SDXException(logger, SDXExceptionCode.ERROR_INVALID_NEW_SUPERUSER_DATA, null, null);

        //build the file and place it
        File superUser = new File(this.sdxConfPath, SUPER_USER_FILE_NAME);
        //output stream to file
        FileOutputStream suOut = null;
        try {
            suOut = new FileOutputStream(superUser);
            suOut.write(data.getBytes());
            //setting the necessary classfields
            this.isSuperUserSet = true;
        } catch (IOException e) {
            String[] args = new String[1];
            //giving the path information for the file we try to write
            args[0] = this.sdxConfPath + SUPER_USER_FILE_NAME;
            throw new SDXException(logger, SDXExceptionCode.ERROR_WRITE_NEW_SUPERUSER_FILE, args, e);
        } finally {
            if (suOut != null) {
                try {
                    suOut.close();
                } catch (IOException e) {
                    throw new SDXException(logger, SDXExceptionCode.ERROR_CLOSE_STREAM, null, e);
                }
            }
        }
    }

    /**Indicates whether the super user info is set*/
    public boolean isSuperUserSet() {
        /*we need something such as public boolean superuserIsSet() in Framework, where we return true if the superuser password has
        been changed (thus the passwd file exists and is valid) and false if not.*/
        try {
            getSuperUserInformationFromFile();
        } catch (SDXException e) {
            //do nothing here as the appropriate state for isSuperUserSet will be adjusted
        }
        return isSuperUserSet;
    }

    /**Validates the super user based on the provided
     * and password
     *
     * @param userid
     * @param passwd
     * @return  The SuperuserInformation object
     * @throws SDXException
     */
    public SuperuserInformation validateSuperUser(String userid, String passwd) throws SDXException {
        /*we need something such as validateSuperuser(userid, passwd) in the Framework interface, where, if userid and passwd are
        set correctly, a SuperuserInformation object is returned, and if not, an exception is thrown. If valid, the superuser should
        contain userid, name, email from the passwd file.*/
        SuperuserInformation su = this.getSuperUserInformationFromFile();
        if (isSuperUserSet) {
            if (su != null && userid.equals(su.getId())) {
                //at this point the id's match, so we check the password and then return the object
                checkPassword(passwd);
                return su;
            } else {
                String[] args = new String[2];
                args[0] = userid;
                args[1] = passwd;
                throw new SDXException(logger, SDXExceptionCode.ERROR_VALIDATE_SUPERUSER, args, null);
            }
        } else {
            throw new SDXException(logger, SDXExceptionCode.ERROR_SUPERUSER_NOT_SET, null, null);
        }
    }

    /**Builds the super user information object based on a file and set's the encrypted password class field*/
    private SuperuserInformation getSuperUserInformationFromFile() throws SDXException {
        SuperuserInformation su = null;
        File superUser = new File(this.sdxConfPath, SUPER_USER_FILE_NAME);
        //we read the file
        FileInputStream suIs = null;
        byte[] bytes = null;
        try {
            suIs = new FileInputStream(superUser);
            bytes = new byte[suIs.available()];
            suIs.read(bytes);
        } catch (IOException e) {
            Logger localLogger;
            if (e instanceof FileNotFoundException)
            //we don't want to log this now
                localLogger = null;
            else
                localLogger = this.logger;

            this.isSuperUserSet = false;//we can't find the file so set the class field appropriately
            String[] args = new String[1];
            //giving the path information for the file we try to write
            args[0] = this.sdxConfPath + SUPER_USER_FILE_NAME;
            throw new SDXException(localLogger, SDXExceptionCode.ERROR_READ_NEW_SUPERUSER_FILE, args, e);
        } finally {
            if (suIs != null)
                try {
                    suIs.close();
                } catch (IOException e) {
                    throw new SDXException(logger, SDXExceptionCode.ERROR_CLOSE_STREAM, null, e);
                }
        }
        String data = null;
        if (bytes != null) data = new String(bytes);
        if (Utilities.checkString(data)) {
            //data should be in this form, {userid}:{firstname}:{lastname}:{email}:{xml:lang}:{encryptedpassword}
            String[] strings = StringUtils.split(data, ":");

            if (strings.length < 6) {
                //string not in the correct format, file badly changed or corrupted
                this.isSuperUserSet = false;
                String[] args = new String[1];
                //giving the path information for the file we try to write
                args[0] = this.sdxConfPath + SUPER_USER_FILE_NAME;
                throw new SDXException(logger, SDXExceptionCode.ERROR_INVALID_SUPERUSER_DATA_FROM_FILE, args, null);
            }

            String id = "";
            String firstname = "";
            String lastname = "";
            String email = "";
            String xmlLang = "";
            String passwd = "";
            for (int i = 0; i < strings.length; i++) {
                //asssigning values
                if (i == 0) id = strings[i];
                if (i == 1) firstname = strings[i];
                if (i == 2) lastname = strings[i];
                if (i == 3) email = strings[i];
                if (i == 4) xmlLang = strings[i];
                if (i == 5) passwd = strings[i];
            }

            su = new SuperuserInformation();
            su.setId(id);
            su.setFirstname(firstname);
            su.setLastname(lastname);
            su.setEmail(email);
            su.setPreferredLocale(Utilities.buildLocale(xmlLang, null, null));
            this.suEncryptedPasswd = passwd;

        }

        //setting the class field appropriately based upon our info
        if (su != null)
            this.isSuperUserSet = true;
        else
            this.isSuperUserSet = false;

        return su;

    }

    /**Compares the current super user's encrypted password
     * against the provided password after encryption.
     *
     * If both the super user password and the provided password are null
     * no exception is thrown, ie we have a match.
     *
     * @param passwd    The password to compare against the super user's password
     */
    private void checkPassword(String passwd) throws SDXException {
        /*it is probabaly a good idea to call getSuperUserInformationFromFile
        before this to ensure our encrypted password is up to date, but since
        we must do it just before in the only calling method it won't be necessary
        here until this method is used in other places-rbp*/
        if (passwd != suEncryptedPasswd) {
            String providedEncryptedPasswd = this.encryptPassword(passwd);
            if (!suEncryptedPasswd.equals(providedEncryptedPasswd))
                throw new SDXException(logger, SDXExceptionCode.ERROR_SUPERUSER_PASSWORD, null, null);
        }
    }

    /**Encrypts a password based on SHA and encoding base 64
     *
     * @param passwd    The password to encrypt
     * @return          The encrypted password, or <code>null</code> if the provided String was null
     * @throws SDXException
     */
    private String encryptPassword(String passwd) throws SDXException {
        if (passwd == null)
            return passwd;
        else {
            MessageDigest md = null;
            try {
                md = MessageDigest.getInstance("SHA");
            } catch (NoSuchAlgorithmException e) {
                //message digest algorithm is not available in the caller's environment
                throw new SDXException(logger, SDXExceptionCode.ERROR_GET_ENCRYPTION, null, e);
            }
            md.update(passwd.getBytes());
            return SourceUtil.encodeBASE64(md.digest());
        }
    }

    /**Returns the xml:lang attibute value from the configuration*/
    public String getXmlLang() {
        return xmlLang;
    }

    /**Returns the locale for the framework*/
    public Locale getLocale() {
        return locale;
    }

    /**
     * Could send an XML representation of something, but currently has no function.
     *
     * @param handler   A SAX content handler to feed with events.
     * @throws SAXException
     * @throws ProcessingException
     */
    public void toSAX(ContentHandler handler) throws SAXException, ProcessingException {

        // TODO: what XML structure and information here ?

    }

    public String getEncoding() {
        return encoding;
    }

}
