/* Copyright (c) 2010 - 2012 Daniel Thiele, Axel Wachtler
 All rights reserved.

 Redistribution and use in source and binary forms, with or without
 modification, are permitted provided that the following conditions
 are met:

 * Redistributions of source code must retain the above copyright
 notice, this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 notice, this list of conditions and the following disclaimer in the
 documentation and/or other materials provided with the distribution.
 * Neither the name of the authors nor the names of its contributors
 may be used to endorse or promote products derived from this software
 without specific prior written permission.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 POSSIBILITY OF SUCH DAMAGE. */

/* $Id$ */
/**
 * @file
 * @brief Wireless bootloader application, resides in bootloader section
 *
 * @author Daniel Thiele
 *         Dietzsch und Thiele PartG
 *         Bautzner Str. 67 01099 Dresden
 *         www.ib-dt.de daniel.thiele@ib-dt.de
 *
 * @ingroup grpAppWiBo
 *
 *
 *
 * Description of compile flavours:
 *
 * WIBO_FLAVOUR_KEYPRESS
 *   run bootloader only if key is pressed
 *
 * WIBO_FLAVOUR_KEYPRESS_KEYNB
 *   only valid if WIBO_FLAVOUR_KEYPRESS is defined, the number of the key
 *   that is evaluated if stay in bootloader or jump to application
 *
 *
 *
 * WIBO_FLAVOUR_MAILBOX
 *   flag from application to stay in bootloader, typically a certain register with a secret value
 *
 * WIBO_FLAVOUR_MAILBOX_REGISTER
 *    the register to find secret value
 *    example: #define WIBO_FLAVOUR_MAILBOX_REGISTER (GPRR0)
 * WIBO_FLAVOUR_MAILBOX_CODE
 *    the secret value
 *    example: #define WIBO_FLAVOUR_MAILBOX_CODE (0xB5)
 */

/* avr-libc inclusions */
#include <avr/io.h>
#include <avr/boot.h>
#include <avr/pgmspace.h>
#include <util/crc16.h>
#include <string.h>

/* uracoli inclusions */
#include <board.h>
#include <ioutil.h>
#include <transceiver.h>
#include <p2p_protocol.h>

#define _SW_VERSION_ (0x01)

/* serial debug printout via USART0, using uracoli ioutil/hif module */
#ifndef SERIALDEBUG
#define SERIALDEBUG (0)
#endif

#if SERIALDEBUG > 0
#include <avr/interrupt.h>
#include <hif.h>
#endif

/* incoming frames are collected here */
static union
{
	uint8_t data[MAX_FRAME_SIZE];
	p2p_hdr_t hdr;
	p2p_wibo_data_t wibo_data;
	p2p_wibo_finish_t wibo_finish;
	p2p_wibo_target_t wibo_target;
} rxbuf;

#define PAGEBUFSIZE (SPM_PAGESIZE)

/* the only outgoing command, create in SRAM here
 * the values assigned here never have to changed
 */
static p2p_ping_cnf_t pingrep = {
		.hdr.cmd = P2P_PING_CNF,
		.hdr.fcf = 0x8841, /* short addressing, frame type: data, no ACK requested */
		.version = _SW_VERSION_,
		.appname = "wibo",
		.boardname = BOARD_NAME,
};

/* collect memory page data here */
static uint8_t pagebuf[PAGEBUFSIZE];

/* IEEE802.15.4 parameters for communication */
static node_config_t nodeconfig;

/*
 * \brief Program a page
 * (1) Erase the page containing at the given address
 * (2) Fill the page buffer
 * (3) Write the page buffer
 * (4) Wait for finish
 *
 * @param addr The address containing the page to program
 * @param *buf Pointer to buffer of page content
 */
static inline void boot_program_page(uint32_t addr, uint8_t *buf)
{
	uint16_t i;

#if SERIALDEBUG > 0
	printf("Write Flash addr=%04lX", addr);
	for (i = 0; i < SPM_PAGESIZE; i++)
	{
		if (!(i % 16))
		{
			putchar('\n');
			putchar(' ');
			putchar(' ');
		}
		printf(" %02X", buf[i]);
	}
	putchar('\n');
#endif

	boot_page_erase(addr);
	boot_spm_busy_wait();

	i = SPM_PAGESIZE;
	do
	{
		/* Set up little-endian word */
		uint16_t w = *buf++;
		w += (*buf++) << 8;

		boot_page_fill(addr + SPM_PAGESIZE - i, w);

		i -= 2;
	} while (i);

	boot_page_write(addr);
	boot_spm_busy_wait();
}

#if SERIALDEBUG > 0

int putc(int c, FILE *stream)
{
	//HIF_UART_DATA = c;
	return c;
}

static FILE usart0_stdio = FDEV_SETUP_STREAM(putc, NULL, _FDEV_SETUP_WRITE);
#endif

int __attribute__((OS_main)) main(void);
int main(void)
{
	void (*app)(void) = 0x0000;

	/* do not initialize variables to save code space, BSS segment sets all variables to zero
	 *
	 * compiler throws warning, please ignore
	 */
	uint8_t *ptr;
	uint8_t tmp;
	uint16_t datacrc = 0; /* checksum for received data */
	uint16_t addr = 0;
	uint16_t pagebufidx = 0;

	/* Target for program data 
		'F' : flash memory
		'E' : eeprom memory
		'X' : None, dry run
	*/
	uint8_t target = 'F'; /* for backwards compatibility */

#if defined(WIBO_FLAVOUR_KEYPRESS) || defined(WIBO_FLAVOUR_MAILBOX)
	uint8_t run_bootloader = 0;
#endif

	/* only stay in bootloader if key is pressed */
#if defined(WIBO_FLAVOUR_KEYPRESS)
#if defined(NO_KEYS)
#error "No Keys defined for WIBO_FLAVOUR_KEYPRESS"
#endif
	KEY_INIT();
	if(KEY_GET() != 0)
	{
		run_bootloader = 1;
	}
#endif /* defined(WIBO_FLAVOUR_KEYPRESS) */

#if defined(WIBO_FLAVOUR_MAILBOX)

#if !defined(WIBO_FLAVOUR_MAILBOX_REGISTER) || !defined(WIBO_FLAVOUR_MAILBOX_CODE)
#error "WIBO_FLAVOUR_MAILBOX not defined correctly"
#endif
	if(WIBO_FLAVOUR_MAILBOX_REGISTER == WIBO_FLAVOUR_MAILBOX_CODE)
	{
		run_bootloader = 1;
	}
	//WIBO_MAILBOX_CLR();
#endif /* defined(WIBO_FLAVOUR_MAILBOX) */

#if defined(WIBO_FLAVOUR_KEYPRESS) || defined(WIBO_FLAVOUR_MAILBOX)
	if(run_bootloader == 0)
	{
		app();
	}
#endif

	LED_INIT();
	LED_SET(0);

	get_node_config(&nodeconfig);

	trx_io_init(DEFAULT_SPI_RATE);
	TRX_RESET_LOW();
	TRX_SLPTR_LOW();
	TRX_RESET_HIGH();

	trx_reg_write(RG_TRX_STATE, CMD_FORCE_TRX_OFF);

	/* BAD: better way would be to abstract for all boards even if it makes no sense (megaRF variants) */
#if defined(DI_TRX_IRQ)
	DI_TRX_IRQ();
#endif

#if (RADIO_TYPE == RADIO_AT86RF230A) || (RADIO_TYPE == RADIO_AT86RF230B)
	trx_reg_write(RG_PHY_TX_PWR, 0x80); /* set TX_AUTO_CRC bit, and TX_PWR = max */
#else
	trx_reg_write(RG_TRX_CTRL_1, 0x20); /* set TX_AUTO_CRC bit */
#endif

	/* setup network addresses for auto modes */
	pingrep.hdr.pan = nodeconfig.pan_id;
	pingrep.hdr.src = nodeconfig.short_addr;

	trx_set_panid(nodeconfig.pan_id);
	trx_set_shortaddr(nodeconfig.short_addr);

	/* use register write to save code space, overwrites Bits CCA_REQUEST CCA_MODE[1] CCA_MODE[0]
	 * which is accepted
	 */
	trx_reg_write(RG_PHY_CC_CCA, nodeconfig.channel);

#if RADIO_TYPE == RADIO_AT86RF212

	/* reset value, BPSK-40 */
	/* trx_reg_write(RG_TRX_CTRL_2, 0x24); */

	/* +5dBm acc. to datasheet AT86RF212 table 7-15 */
	trx_reg_write(RG_PHY_TX_PWR, 0x84);
#endif /* RADIO_TYPE == RADIO_AT86RF212 */

	trx_reg_read(RG_IRQ_STATUS);

#if RADIO_TYPE == RADIO_ATMEGA128RFA1_A ||\
      RADIO_TYPE == RADIO_ATMEGA128RFA1_B ||\
      RADIO_TYPE == RADIO_ATMEGA128RFA1_C ||\
      RADIO_TYPE == RADIO_ATMEGA128RFA1_D
	trx_reg_write(RG_IRQ_MASK, TRX_IRQ_RX_END | TRX_IRQ_TX_END);
#else
	trx_reg_write(RG_IRQ_MASK, TRX_IRQ_TRX_END);
#endif
	trx_reg_write(RG_CSMA_SEED_0, nodeconfig.short_addr);
	trx_reg_write(RG_TRX_STATE, CMD_RX_AACK_ON);

#if SERIALDEBUG > 0

	stdout = stderr = &usart0_stdio;
	puts("WIBO Bootlapp Serial Debug");
	printf("PANID: %04X SHORTADDR: %04X CHANNEL: %d\n",
			nodeconfig.pan_id, nodeconfig.short_addr, nodeconfig.channel);
#endif

	for (;;)
	{
		LED_CLR(0);

#if RADIO_TYPE == RADIO_ATMEGA128RFA1_A ||\
      RADIO_TYPE == RADIO_ATMEGA128RFA1_B ||\
      RADIO_TYPE == RADIO_ATMEGA128RFA1_C ||\
      RADIO_TYPE == RADIO_ATMEGA128RFA1_D
		while (!(trx_reg_read(RG_IRQ_STATUS) & TRX_IRQ_RX_END));
		trx_reg_write(RG_IRQ_STATUS, TRX_IRQ_RX_END); /* clear the flag */
#else
		while (!(trx_reg_read(RG_IRQ_STATUS) & TRX_IRQ_TRX_END))
			;
#endif
		trx_frame_read(rxbuf.data, sizeof(rxbuf.data) / sizeof(rxbuf.data[0]),
				&tmp); /* dont use LQI, write into tmp variable */

		LED_SET(0);
		/* light as long as actions are running */

		/* prepare header of possible answer frame */

		switch (rxbuf.hdr.cmd)
		{

		case P2P_PING_REQ:
			pingrep.hdr.dst = rxbuf.hdr.src;
			pingrep.hdr.seq++;
			pingrep.crc = datacrc;

			trx_reg_write(RG_TRX_STATE, CMD_TX_ARET_ON);
			trx_frame_write(sizeof(p2p_ping_cnf_t) + sizeof(BOARD_NAME) + 2,
					(uint8_t*) &pingrep);
			TRX_SLPTR_HIGH();
			TRX_SLPTR_LOW();

#if SERIALDEBUG > 0
			puts("Ping");
#endif

#if RADIO_TYPE == RADIO_ATMEGA128RFA1_A ||\
      RADIO_TYPE == RADIO_ATMEGA128RFA1_B ||\
      RADIO_TYPE == RADIO_ATMEGA128RFA1_C ||\
      RADIO_TYPE == RADIO_ATMEGA128RFA1_D
			while (!(trx_reg_read(RG_IRQ_STATUS) & TRX_IRQ_TX_END));
			trx_reg_write(RG_IRQ_STATUS, TRX_IRQ_TX_END); /* clear the flag */
#else
			while (!(trx_reg_read(RG_IRQ_STATUS) & TRX_IRQ_TRX_END))
				;
#endif
			trx_reg_write(RG_TRX_STATE, CMD_RX_AACK_ON);
			break;

		case P2P_WIBO_TARGET:
			target = rxbuf.wibo_target.targmem;
#if SERIALDEBUG > 0
			printf("Set Target to %c", target);
#endif
			break;

		case P2P_WIBO_RESET:
#if SERIALDEBUG > 0
			puts("Reset");
#endif
			addr = SPM_PAGESIZE; /* misuse as counter */
			ptr = pagebuf;
			do
			{
				*ptr++ = 0xFF;
			} while (--addr);
			addr = 0;
			datacrc = 0;
			pagebufidx = 0;
			break;

		case P2P_WIBO_DATA:
			tmp = rxbuf.wibo_data.dsize;
#if SERIALDEBUG > 0
			printf("Data[%d]", rxbuf.wibo_data.dsize);
			ptr = rxbuf.wibo_data.data;
			do
			{
				printf(" %02X", *ptr++);
			}while (--tmp);
			;
			putchar('\n');
#endif
			ptr = rxbuf.wibo_data.data;
			do
			{
				datacrc = _crc_ccitt_update(datacrc, *ptr);
				if (target == 'F')
				{ /* Flash memory */
					pagebuf[pagebufidx++] = *ptr;
					if (pagebufidx >= PAGEBUFSIZE)
					{
						boot_program_page(addr, pagebuf);
						addr += SPM_PAGESIZE;
						pagebufidx = 0;
					}
				}
				else
				{ /* unknown target, dry run */

				}
				ptr++;
			} while (--tmp);
			break;

		case P2P_WIBO_FINISH:
#if SERIALDEBUG > 0
			puts("Finish");
#endif
			if (target == 'F')
			{ /* Flash memory */
				boot_program_page(addr, pagebuf);
				addr += SPM_PAGESIZE;
				pagebufidx = 0;
			}
			else
			{ /* unknown target, dry run */

			}
			break;

		case P2P_WIBO_EXIT:
#if SERIALDEBUG > 0
			puts("Exit");
#endif
			boot_rww_enable();
			app();
			break;

		default:
			/* unknown or unhandled command */
			break;
		};
	}
}

/* EOF */
