/* === includes ============================================================ */
#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/sleep.h>
#include <avr/wdt.h>
#include <util/delay.h>
#include <string.h>
/* uracoli libraries */
#include <board.h>
#include <radio.h>

#include "spiel.h"

/* === macros ============================================================== */
#define KEY_NONE (255)

/* === types =============================================================== */
typedef enum
{
    IDLE,
    WAIT_MY_KEY,
    WAIT_PEER_KEY,
} game_state_t;

/* === globals ============================================================= */

static volatile bool TxInProgress;
static volatile uint8_t LocalKey;
static uint32_t SysTime = 0;
static volatile int16_t SleepCounter;

/* === functions =========================================================== */

void xxo_deep_sleep(void);

/*
 * Sicherstellen, dass der Watchdog nach einem Watchdog-Reset wirklich
 * ruhig gestellt wird.  Siehe Datenblatt 13.4.2 (Seite 184 in
 * Dokument 8266C-MCU Wireless-08/11).
 *
 * Diese Funktion wird nicht mittels CALL gerufen, sondern vom Linker
 * in den Startup-Code eingebaut.  Daher ist sie "naked"
 * (d. h. bekommt weder Prolog noch Epilog, nicht einmal ein RET am
 * Ende) und wird in Section .init3 platziert.  Sie ist nur aus Sicht
 * der C-Syntax eine Funktion, da sich in C ausführbarer Code stets
 * innerhalb einer Funktion befinden muss.
 */
void get_mcusr(void) \
  __attribute__((naked)) \
  __attribute__((section(".init3")));
void get_mcusr(void)
{
  MCUSR = 0;
  wdt_disable();
}


uint8_t xxo_get_key(void)
{
uint8_t ret;
    cli();
    ret = LocalKey;
    LocalKey = KEY_NONE;
    sei();
    return ret;
}


void xxo_do_sleep(int16_t sleep_time)
{
    set_sleep_mode(SLEEP_MODE_IDLE);
    cli();
    SleepCounter = (sleep_time>>1);
    while (SleepCounter > 0)
    {
        sei();
        sleep_mode();
        cli();
    }
    sei();
}


ISR(TIMER0_OVF_vect)
{
    static uint8_t row = 0;
    uint8_t key;

    if (SleepCounter > 0)
    {
        SleepCounter--;
    }

    SysTime ++;

    key = update_pads(row);

    if ((key != KEY_NONE) && (LocalKey == KEY_NONE))
    {
        LocalKey = key;
    }

    display_leds(row);

    row ++;
    if (row > 2)
    {
        row = 0;
    }
}

ISR(WDT_vect)
{
    uint8_t key;

    /* Wir sind nur an Taste 4 interessiert, die in Reihe 1 liegt. */
    key = update_pads(1);

    if (key == 4)
    {
        LocalKey = key;

        /* 15 ms Timeout, Watchdog im Reset-Mode */
        wdt_enable(WDTO_15MS);
        for (;;)
        {
            /* Warten auf den Watchdog-Reset */
        }
    }
}



int main(void)
{
    node_config_t ncfg;
    game_state_t state;
    uint8_t mycolor, peercolor, local_key, radio_event, mylastkey, winner_color;

    state = IDLE;
    local_key = KEY_NONE;
    radio_event = KEY_NONE;
    mycolor = OFF;
    peercolor = OFF;
    mylastkey = NB_FIELDS;

    io_init();
    xxo_radio_init(&ncfg);
    led_cls();


    /* lösche Tasteneingaben */
    xxo_radio_get_event();
    xxo_get_key();
    leds_flash(1, 4, GREEN);
    leds_flash(1, 4, RED);

    while(1)
    {

        switch (state)
        {
            case IDLE:
                local_key = xxo_get_key();
                radio_event = xxo_radio_get_event();
                if(local_key == 4)
                {
                    xxo_send(REQUEST_GAME, &ncfg);
                }
                if((radio_event == REQUEST_GAME) && (mycolor == OFF))
                {
                    mycolor = GREEN;
                    peercolor = RED;
                    xxo_send(CONFIRM_GAME, &ncfg);
                    state = WAIT_MY_KEY;
                }
                else if (radio_event == CONFIRM_GAME)
                {
                    mycolor = RED;
                    peercolor = GREEN;
                    state = WAIT_PEER_KEY;
                }
                if((mycolor != OFF) && (state != IDLE))
                {
                    leds_flash(4, 4, mycolor);
                    led_set(4, OFF);
                }
                break;

            case WAIT_MY_KEY:
                local_key = xxo_get_key();
                if (local_key != KEY_NONE)
                {
                    if (OFF == led_get(local_key))
                    {
                        mylastkey = local_key;
                        led_set(local_key, mycolor);
                        xxo_send(local_key, &ncfg);
                        state = WAIT_PEER_KEY;
                    }
                }
                break;

            case WAIT_PEER_KEY:
                radio_event = xxo_radio_get_event();
                if (radio_event < NB_FIELDS)
                {
                    led_set(radio_event, peercolor);
                    state = WAIT_MY_KEY;
                }
                break;
        }

        if (state != IDLE)
        {
            winner_color = led_check_winner();

            if (OFF != winner_color)
            {
                if (winner_color != 4)
                {
                    leds_flood_fill(winner_color);
                    led_set(4, mycolor);
                }
                else
                {
                    uint8_t i;
                    for (i=0; i< 4;i++)
                    {
                        led_set(0, GREEN);
                        led_set(2, GREEN);
                        led_set(4, GREEN);
                        led_set(6, GREEN);
                        led_set(8, GREEN);
                        led_set(1, RED);
                        led_set(3, RED);
                        led_set(5, RED);
                        led_set(7, RED);
                        xxo_do_sleep(250);
                        led_set(0, RED);
                        led_set(2, RED);
                        led_set(4, RED);
                        led_set(6, RED);
                        led_set(8, RED);
                        led_set(1, GREEN);
                        led_set(3, GREEN);
                        led_set(5, GREEN);
                        led_set(7, GREEN);
                        xxo_do_sleep(250);
                    }
                }
                xxo_do_sleep(1500);
                led_cls();
                mycolor = OFF;
                peercolor = OFF;
                mylastkey = NB_FIELDS;
                state = IDLE;
                xxo_turn_off_radio();
                display_leds(0);
                display_leds(1);
                display_leds(2);
                xxo_deep_sleep();
            }
        }
        else
        {
            if(mylastkey < NB_FIELDS && state == WAIT_MY_KEY)
            {
                leds_flash(4, mylastkey, mycolor);
                led_set(mylastkey, mycolor);
            }

        }

        sleep_mode();
    }
    return 0;
}

#pragma GCC optimize ("-Os")
/*
 * Diese Funktion muss immer optimiert compiliert werden, damit das
 * Timing des WDCE-Bits eingehalten wird.
 */
void xxo_deep_sleep(void)
{
    set_sleep_mode(SLEEP_MODE_PWR_DOWN);

    PORTB &= ~(COL_LEDS | ROW_IO_MASK);
    DDRB = COL_LEDS | ROW_IO_MASK;
    PORTG = 0;
    DDRG = 0x3f;                /* kapazitive Touch-Pads */
    PORTD &= ~COL_PADS;
    DDRD = COL_PADS;

    // ungenutzte Portpins
    DDRE = 0xff;
    PORTE = 0;
    DDRF = 0xff;
    PORTF = 0;
    PORTD |= 0x0f;              /* Nutzer-Pins: I²C + UART 1 */
    PORTB |= 0xc0;

    cli();
    WDTCSR = _BV(WDCE);
    WDTCSR = _BV(WDIF) | _BV(WDIE); /* 15 ms interrupts */
    sei();
    TCCR0B = 0;

    sleep_mode();
}
