/* Copyright (c) 2011 - 2013 Daniel Thiele, Axel Wachtler
 All rights reserved.

 Redistribution and use in source and binary forms, with or without
 modification, are permitted provided that the following conditions
 are met:

 * Redistributions of source code must retain the above copyright
 notice, this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 notice, this list of conditions and the following disclaimer in the
 documentation and/or other materials provided with the distribution.
 * Neither the name of the authors nor the names of its contributors
 may be used to endorse or promote products derived from this software
 without specific prior written permission.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 POSSIBILITY OF SUCH DAMAGE. */

/**
 * @file Demo Application based on P2P protocol
 *       Measure temperature via RFA128 internal sensor and
 *       format to line and send via Wuart frames
 *
 * @brief
 * @ingroup
 */

#define ISRAVEN ( defined(ravrf230a) || defined(ravrf230b) )


/* === includes ============================================================ */

/* avr-libc inclusions */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <avr/wdt.h>
#include <util/delay.h>

/* uracoli inclusions */
#include "board.h"
#include "ioutil.h"
#include "transceiver.h"
#include "lw_mesh.h"

#if ISRAVEN
#include "hif.h"
#endif

#include "tlogger.h"
#include "meas.h"
#include "sensordata.h"

/* === macros ============================================================== */

#define APP_VERSION (0x01)
#define APP_NAME "lwm_tlogger"

#if defined(SUPPORTED_SENSORS)
	sensormask_t sensormask = SUPPORTED_SENSORS;
#else
	sensormask_t sensormask = SENSORDATA_NONE;
#warning "No supported sensors defined"
#endif

/* === types =============================================================== */

typedef enum
{
	APPSTATE_IDLE, APPSTATE_MEAS, APPSTATE_TRX,
} appstate_t;

/* === globals ============================================================= */

uint8_t mcusr_mirror __attribute__ ((section (".noinit")));

/* IEEE802.15.4 parameters for communication */
const node_config_t PROGMEM nc_flash =
{
	.short_addr = 0xFECA,
	.pan_id = 0x3412,
	.channel = 17
};

static node_config_t NodeConfig;

static NWK_DataReq_t request;
char message[MAX_LINE_LENGTH] = "-empty-";

#if ISRAVEN
static FILE hif_stream = FDEV_SETUP_STREAM(hif_putc, hif_getc, _FDEV_SETUP_RW);	
#endif

static volatile appstate_t appstate;
static volatile appstate_t nextstate;

/* === prototypes ========================================================== */

/* === functions =========================================================== */

/*
 * \brief Setup watchdog to serve as application timer
 *
 */
static inline void wdt_timersetup(uint8_t timeout)
{
	WDTCSR = (1 << WDCE) | (1 << WDE); /* Enable configuration change */
	WDTCSR = (1 << WDIF) | (1 << WDIE) | /* Enable Watchdog Interrupt Mode */
	(timeout & 0x08 ? _WD_PS3_MASK : 0x00) | (timeout & 0x07);
}

/*
 * \brief Watchdog ISR, used as application timer
 *
 */
ISR(WDT_vect, ISR_NOBLOCK)
{
	nextstate = APPSTATE_MEAS;
}


/*
 * \brief Initialize Application
 *
 */
void app_init(void)
{
	char cfg_location = '?';

	/* === read node configuration data ===================================== */
	/* 1st trial: read from EEPROM */
	if (get_node_config_eeprom(&NodeConfig, 0) == 0)
	{
		/* using EEPROM config */;
		cfg_location = 'E';
	}
	/* 2nd trial: read from FLASHEND */
	else if (get_node_config(&NodeConfig) == 0)
	{
		/* using FLASHEND config */;
		cfg_location = 'F';
	}
	/* 3rd trial: read default values compiled into the application */
	else
	{
		/* using application default config */;
		memcpy_P(&NodeConfig, (PGM_VOID_P) & nc_flash, sizeof(node_config_t));
		cfg_location = 'D';
	}

	lw_mesh_init(NodeConfig.pan_id, NodeConfig.short_addr, NodeConfig.channel);

#if defined(SR_RND_VALUE)
	/* TODO: if the random value is not available in HW, e.g. AT86RF230A/B,
	 * we need to find another method of seeding CSMA  */
	trx_reg_write(RG_CSMA_SEED_0, trx_bit_read(SR_RND_VALUE) | (trx_bit_read(SR_RND_VALUE) << 4));
	trx_reg_write(RG_CSMA_SEED_1, trx_bit_read(SR_RND_VALUE) | (trx_bit_read(SR_RND_VALUE) << 4));
#endif

	/* Welcome Blink */
	LED_SET(0);
	LED_SET(1);
	_delay_ms(20);
	LED_CLR(0);
	LED_CLR(1);
}


/*
 * \brief Save MCUSR to variable and disable watchdog
 * This must be done very early after reset, placing to .init3 section is done
 * in the forward declaration above
 *
 */
void __attribute__((naked)) __attribute__((section(".init3"))) get_mcusr(void)
{
	mcusr_mirror = MCUSR;
	MCUSR = 0;
	wdt_disable();
}

static void app_data_confirm(NWK_DataReq_t *req)
{
    if(req->status == NWK_SUCCESS_STATUS)
    {
        /* frame was sent successfully */
    }
    else
    {
        /* some error occurred */
    }
	
	nextstate = APPSTATE_IDLE;
}

uint8_t sensordata_fill_payload(uint8_t *ptr)
{
	float *fptr;
	uint8_t size = 0; /* to be returned */

	/* supported by all sensors (Raven + RFA) */
	request.data[size++] = SENSORDATA_VOLTAGE_AVR;
	fptr = (float*)&request.data[size];
	*fptr = measure_vmcu();
	size +=	sizeof(float);

#if defined(__AVR_ATmega128RFA1__)
	request.data[size++] = SENSORDATA_TEMPERATURE_RFA;
	fptr = (float*)&request.data[size];
	*fptr = measure_tmcu();
	size +=	sizeof(float);
#endif
	
	request.data[size] = SENSORDATA_NONE;
	
	return size;
}

int main()
{

#if defined(radiofaro)
	/* init all unused pins */
	DDRB = 0x00;  /* as input */
	PORTB = 0xFF; /* pullups */
	DDRD = 0x00;  /* as input */
	PORTD = 0xFF; /* pullups */
	DDRE = 0x00;  /* as input */
	PORTE = 0xFF; /* pullups */
	DDRF = 0x00;  /* as input */
	PORTF = 0xFF; /* pullups */
	DDRG = 0x00;  /* as input */
	PORTG = 0xFF; /* pullups */

	DIDR0 = 0xFF; /* disable all ADC inputs */
#endif

	LED_INIT();

	app_init();

    /* prepare data request */
    request.dstAddr     = COORD_ADDR;
    request.dstEndpoint = RX_ENDPOINT_ID;
    request.srcEndpoint = TX_ENDPOINT_ID;
    request.options     = 0;
    request.data        = message;
    request.size        = strlen(message);
    request.confirm     = app_data_confirm;

	wdt_reset();	
	wdt_timersetup(WDTO_4S);

	sei();

	measure_init();
	
	request.size = 0;
	request.data[request.size++] = SENSORDATA_CAPTION;
	sprintf(&request.data[request.size], "LWM Demo application: Temperature logger, Board %s", BOARD_NAME);
	request.size += strlen(message) + 1; /* plus NULL terminator */
	
	lw_mesh_data_req(&request);

	set_sleep_mode(SLEEP_MODE_IDLE);
	appstate = nextstate = APPSTATE_TRX;

	for (;;)
	{
		lw_mesh_task_handler();

		if (appstate != nextstate)
		{
			switch (nextstate)
			{
			case APPSTATE_IDLE:
				set_sleep_mode(SLEEP_MODE_PWR_DOWN);
				break;
			case APPSTATE_MEAS:
				LED_SET(0);
				DELAY_MS(5);
				LED_CLR(0);

				request.size = sensordata_fill_payload(request.data);
				
				nextstate = APPSTATE_TRX;

				/* Fake,
				 *
				 * fall through
				 */
			case APPSTATE_TRX:
				set_sleep_mode(SLEEP_MODE_IDLE);
				
				lw_mesh_data_req(&request);
				break;
			}
			appstate = nextstate;
		}
		else
		{
			/* until power management of LWM is clarified */
			if(appstate == APPSTATE_IDLE){
				// sleep_mode();
			}
		}
		
	} /* for(;;); */

	/* never reaches this point */

}
/* EOF */
