/*
 * Copyright (C) 2012 Google Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1.  Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 * 2.  Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 * 3.  Neither the name of Apple Computer, Inc. ("Apple") nor the names of
 *     its contributors may be used to endorse or promote products derived
 *     from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE AND ITS CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL APPLE OR ITS CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef PeriodicWave_h
#define PeriodicWave_h

#include <memory>
#include "core/typed_arrays/ArrayBufferViewHelpers.h"
#include "core/typed_arrays/DOMTypedArray.h"
#include "platform/audio/AudioArray.h"
#include "platform/bindings/ScriptWrappable.h"
#include "platform/wtf/Forward.h"
#include "platform/wtf/Vector.h"

namespace blink {

class BaseAudioContext;
class ExceptionState;
class PeriodicWaveOptions;

class PeriodicWave final : public GarbageCollectedFinalized<PeriodicWave>,
                           public ScriptWrappable {
  DEFINE_WRAPPERTYPEINFO();

 public:
  static PeriodicWave* CreateSine(float sample_rate);
  static PeriodicWave* CreateSquare(float sample_rate);
  static PeriodicWave* CreateSawtooth(float sample_rate);
  static PeriodicWave* CreateTriangle(float sample_rate);

  // Creates an arbitrary periodic wave given the frequency components (Fourier
  // coefficients).
  static PeriodicWave* Create(BaseAudioContext&,
                              const Vector<float>& real,
                              const Vector<float>& imag,
                              bool normalize,
                              ExceptionState&);

  static PeriodicWave* Create(BaseAudioContext*,
                              const PeriodicWaveOptions&,
                              ExceptionState&);

  virtual ~PeriodicWave();

  // Returns pointers to the lower and higher wave data for the pitch range
  // containing the given fundamental frequency. These two tables are in
  // adjacent "pitch" ranges where the higher table will have the maximum number
  // of partials which won't alias when played back at this fundamental
  // frequency. The lower wave is the next range containing fewer partials than
  // the higher wave.  Interpolation between these two tables can be made
  // according to tableInterpolationFactor.
  // Where values from 0 -> 1 interpolate between lower -> higher.
  void WaveDataForFundamentalFrequency(float,
                                       float*& lower_wave_data,
                                       float*& higher_wave_data,
                                       float& table_interpolation_factor);

  // Returns the scalar multiplier to the oscillator frequency to calculate wave
  // buffer phase increment.
  float RateScale() const { return rate_scale_; }

  // The size of the FFT to use based on the sampling rate.
  unsigned PeriodicWaveSize() const;

  // The number of ranges needed for the given sampling rate and FFT size.
  unsigned NumberOfRanges() const { return number_of_ranges_; }

  DEFINE_INLINE_TRACE() {}

 private:
  explicit PeriodicWave(float sample_rate);

  void GenerateBasicWaveform(int);

  size_t v8_external_memory_;

  float sample_rate_;
  unsigned number_of_ranges_;
  float cents_per_range_;

  // The lowest frequency (in Hertz) where playback will include all of the
  // partials.  Playing back lower than this frequency will gradually lose more
  // high-frequency information.  This frequency is quite low (~10Hz @ 44.1KHz)
  float lowest_fundamental_frequency_;

  float rate_scale_;

  // Maximum possible number of partials (before culling).
  unsigned MaxNumberOfPartials() const;

  unsigned NumberOfPartialsForRange(unsigned range_index) const;

  void AdjustV8ExternalMemory(int delta);

  // Creates tables based on numberOfComponents Fourier coefficients.
  void CreateBandLimitedTables(const float* real,
                               const float* imag,
                               unsigned number_of_components,
                               bool disable_normalization);
  Vector<std::unique_ptr<AudioFloatArray>> band_limited_tables_;
};

}  // namespace blink

#endif  // PeriodicWave_h
