/*
 * Copyright 2005 Maksim Orlovich <maksim@kde.org>
 * Copyright (C) 2006 Apple Computer, Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef XPathParser_h
#define XPathParser_h

#include <memory>
#include "base/macros.h"
#include "core/xml/XPathPredicate.h"
#include "core/xml/XPathStep.h"
#include "platform/wtf/Allocator.h"

namespace blink {

class ExceptionState;
class XPathNSResolver;

namespace XPath {

class Expression;
class LocationPath;
class Parser;

struct Token {
  STACK_ALLOCATED();
  int type;
  String str;
  Step::Axis axis;
  NumericOp::Opcode numop;
  EqTestOp::Opcode eqop;

  Token(int t) : type(t) {}
  Token(int t, const String& v) : type(t), str(v) {}
  Token(int t, Step::Axis v) : type(t), axis(v) {}
  Token(int t, NumericOp::Opcode v) : type(t), numop(v) {}
  Token(int t, EqTestOp::Opcode v) : type(t), eqop(v) {}
};

class Parser {
  STACK_ALLOCATED();

 public:
  Parser();
  ~Parser();

  XPathNSResolver* Resolver() const { return resolver_.Get(); }
  bool ExpandQName(const String& q_name,
                   AtomicString& local_name,
                   AtomicString& namespace_uri);

  Expression* ParseStatement(const String& statement,
                             XPathNSResolver*,
                             ExceptionState&);

  static Parser* Current() { return current_parser_; }

  int Lex(void* yylval);

  Member<Expression> top_expr_;
  bool got_namespace_error_;

  void RegisterString(String*);
  void DeleteString(String*);

 private:
  bool IsBinaryOperatorContext() const;

  void SkipWS();
  Token MakeTokenAndAdvance(int type, int advance = 1);
  Token MakeTokenAndAdvance(int type, NumericOp::Opcode, int advance = 1);
  Token MakeTokenAndAdvance(int type, EqTestOp::Opcode, int advance = 1);
  char PeekAheadHelper();
  char PeekCurHelper();

  Token LexString();
  Token LexNumber();
  bool LexNCName(String&);
  bool LexQName(String&);

  Token NextToken();
  Token NextTokenInternal();

  void Reset(const String& data);

  static Parser* current_parser_;

  unsigned next_pos_;
  String data_;
  int last_token_type_;
  Member<XPathNSResolver> resolver_;

  HashSet<std::unique_ptr<String>> strings_;
  DISALLOW_COPY_AND_ASSIGN(Parser);
};

}  // namespace XPath

}  // namespace blink

int xpathyyparse(blink::XPath::Parser*);
#endif
