// Copyright 2018 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CHROME_BROWSER_UI_WEBUI_SETTINGS_CHROMEOS_ACCOUNT_MANAGER_HANDLER_H_
#define CHROME_BROWSER_UI_WEBUI_SETTINGS_CHROMEOS_ACCOUNT_MANAGER_HANDLER_H_

#include <string>
#include <vector>

#include "base/macros.h"
#include "base/memory/weak_ptr.h"
#include "base/scoped_observer.h"
#include "chrome/browser/chromeos/account_mapper_util.h"
#include "chrome/browser/ui/webui/settings/settings_page_ui_handler.h"
#include "chromeos/account_manager/account_manager.h"
#include "components/signin/core/browser/account_tracker_service.h"
#include "services/identity/public/cpp/identity_manager.h"

namespace chromeos {
namespace settings {

class AccountManagerUIHandler : public ::settings::SettingsPageUIHandler,
                                public AccountManager::Observer,
                                public AccountTrackerService::Observer {
 public:
  // Accepts non-owning pointers to |AccountManager| and
  // |AccountTrackerService|. Both of these must outlive |this| instance.
  AccountManagerUIHandler(AccountManager* account_manager,
                          AccountTrackerService* account_tracker_service,
                          identity::IdentityManager* identity_manager);
  ~AccountManagerUIHandler() override;

  // WebUIMessageHandler implementation.
  void RegisterMessages() override;
  void OnJavascriptAllowed() override;
  void OnJavascriptDisallowed() override;

  // |AccountManager::Observer| overrides.
  // |AccountManager| is considered to be the source of truth for account
  // information.
  void OnTokenUpserted(const AccountManager::AccountKey& account_key) override;
  void OnAccountRemoved(const AccountManager::AccountKey& account_key) override;

  // |AccountTrackerService::Observer| overrides.
  void OnAccountUpdated(const AccountInfo& info) override;
  void OnAccountUpdateFailed(const std::string& account_id) override;
  void OnAccountRemoved(const AccountInfo& account_key) override;

 private:
  // WebUI "getAccounts" message callback.
  void HandleGetAccounts(const base::ListValue* args);

  // WebUI "addAccount" message callback.
  void HandleAddAccount(const base::ListValue* args);

  // WebUI "reauthenticateAccount" message callback.
  void HandleReauthenticateAccount(const base::ListValue* args);

  // WebUI "removeAccount" message callback.
  void HandleRemoveAccount(const base::ListValue* args);

  // WebUI "showWelcomeDialogIfRequired" message callback.
  void HandleShowWelcomeDialogIfRequired(const base::ListValue* args);

  // |AccountManager::GetAccounts| callback.
  void GetAccountsCallbackHandler(
      base::Value callback_id,
      std::vector<AccountManager::AccountKey> account_keys);

  // Refreshes the UI.
  void RefreshUI();

  // A non-owning pointer to |AccountManager|.
  AccountManager* const account_manager_;

  // A non-owning pointer to |AccountTrackerService|.
  AccountTrackerService* const account_tracker_service_;

  // A non-owning pointer to |IdentityManager|.
  identity::IdentityManager* const identity_manager_;

  chromeos::AccountMapperUtil account_mapper_util_;

  // An observer for |AccountManager|. Automatically deregisters when |this| is
  // destructed.
  ScopedObserver<AccountManager, AccountManager::Observer>
      account_manager_observer_;

  // An observer for |AccountTrackerService|. Automatically deregisters when
  // |this| is destructed.
  ScopedObserver<AccountTrackerService, AccountTrackerService::Observer>
      account_tracker_service_observer_;

  base::WeakPtrFactory<AccountManagerUIHandler> weak_factory_;
  DISALLOW_COPY_AND_ASSIGN(AccountManagerUIHandler);
};

}  // namespace settings
}  // namespace chromeos

#endif  // CHROME_BROWSER_UI_WEBUI_SETTINGS_CHROMEOS_ACCOUNT_MANAGER_HANDLER_H_
