/*
 * Copyright (C) 2008 Apple Inc. All Rights Reserved.
 * Copyright (C) 2010 Google, Inc. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef HTMLEntityParser_h
#define HTMLEntityParser_h

#include "core/CoreExport.h"
#include "platform/text/SegmentedString.h"
#include "wtf/Allocator.h"

namespace blink {

class DecodedHTMLEntity {
  STACK_ALLOCATED();

 private:
  // HTML entities contain at most four UTF-16 code units.
  static const unsigned kMaxLength = 4;

 public:
  DecodedHTMLEntity() : length(0) {}

  bool isEmpty() const { return !length; }

  void append(UChar c) {
    RELEASE_ASSERT(length < kMaxLength);
    data[length++] = c;
  }

  void append(UChar32 c) {
    if (U_IS_BMP(c)) {
      append(static_cast<UChar>(c));
      return;
    }
    append(U16_LEAD(c));
    append(U16_TRAIL(c));
  }

  unsigned length;
  UChar data[kMaxLength];
};

CORE_EXPORT bool consumeHTMLEntity(SegmentedString&,
                                   DecodedHTMLEntity& decodedEntity,
                                   bool& notEnoughCharacters,
                                   UChar additionalAllowedCharacter = '\0');

// Used by the XML parser.  Not suitable for use in HTML parsing.  Use
// consumeHTMLEntity instead.
size_t decodeNamedEntityToUCharArray(const char*, UChar result[4]);

}  // namespace blink

#endif  // HTMLEntityParser_h
