// Copyright 2014 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef MOJO_SERVICES_HTML_VIEWER_HTML_DOCUMENT_VIEW_H_
#define MOJO_SERVICES_HTML_VIEWER_HTML_DOCUMENT_VIEW_H_

#include "base/macros.h"
#include "base/memory/weak_ptr.h"
#include "mojo/public/cpp/application/lazy_interface_ptr.h"
#include "mojo/public/cpp/application/service_provider_impl.h"
#include "mojo/public/cpp/bindings/interface_impl.h"
#include "mojo/public/interfaces/application/application.mojom.h"
#include "mojo/services/public/cpp/view_manager/view_manager_client_factory.h"
#include "mojo/services/public/cpp/view_manager/view_manager_delegate.h"
#include "mojo/services/public/cpp/view_manager/view_observer.h"
#include "mojo/services/public/interfaces/content_handler/content_handler.mojom.h"
#include "mojo/services/public/interfaces/navigation/navigation.mojom.h"
#include "mojo/services/public/interfaces/network/url_loader.mojom.h"
#include "third_party/WebKit/public/web/WebFrameClient.h"
#include "third_party/WebKit/public/web/WebViewClient.h"

namespace base {
class MessageLoopProxy;
}

namespace mojo {

class WebMediaPlayerFactory;
class ViewManager;
class View;
class WebLayerTreeViewImpl;

// A view for a single HTML document.
class HTMLDocumentView : public mojo::InterfaceImpl<mojo::Application>,
                         public blink::WebViewClient,
                         public blink::WebFrameClient,
                         public ViewManagerDelegate,
                         public ViewObserver {
 public:
  // Load a new HTMLDocument with |response|.
  //
  // |service_provider_request| should be used to implement a
  // ServiceProvider which exposes services to the connecting application.
  // Commonly, the connecting application is the ViewManager and it will
  // request ViewManagerClient.
  //
  // |shell| is the Shell connection for this mojo::Application.
  HTMLDocumentView(URLResponsePtr response,
                   ShellPtr shell,
                   scoped_refptr<base::MessageLoopProxy> compositor_thread,
                   WebMediaPlayerFactory* web_media_player_factory);
  virtual ~HTMLDocumentView();

 private:
  // Application methods:
  void AcceptConnection(const String& requestor_url,
                        ServiceProviderPtr provider) override;
  void Initialize(Array<String> args) override;

  // WebViewClient methods:
  virtual blink::WebStorageNamespace* createSessionStorageNamespace();

  // WebWidgetClient methods:
  virtual void initializeLayerTreeView();
  virtual blink::WebLayerTreeView* layerTreeView();

  // WebFrameClient methods:
  virtual blink::WebMediaPlayer* createMediaPlayer(
      blink::WebLocalFrame* frame,
      const blink::WebURL& url,
      blink::WebMediaPlayerClient* client);
  virtual blink::WebMediaPlayer* createMediaPlayer(
      blink::WebLocalFrame* frame,
      const blink::WebURL& url,
      blink::WebMediaPlayerClient* client,
      blink::WebContentDecryptionModule* initial_cdm);
  virtual blink::WebFrame* createChildFrame(blink::WebLocalFrame* parent,
                                            const blink::WebString& frameName);
  virtual void frameDetached(blink::WebFrame*);
  virtual blink::WebCookieJar* cookieJar(blink::WebLocalFrame* frame);
  virtual blink::WebNavigationPolicy decidePolicyForNavigation(
      blink::WebLocalFrame* frame, blink::WebDataSource::ExtraData* data,
      const blink::WebURLRequest& request, blink::WebNavigationType nav_type,
      blink::WebNavigationPolicy default_policy, bool isRedirect);
  virtual void didAddMessageToConsole(
      const blink::WebConsoleMessage& message,
      const blink::WebString& source_name,
      unsigned source_line,
      const blink::WebString& stack_trace);
  virtual void didNavigateWithinPage(
      blink::WebLocalFrame* frame,
      const blink::WebHistoryItem& history_item,
      blink::WebHistoryCommitType commit_type);

  // ViewManagerDelegate methods:
  void OnEmbed(ViewManager* view_manager,
               View* root,
               ServiceProviderImpl* embedee_service_provider_impl,
               scoped_ptr<ServiceProvider> embedder_service_provider) override;
  void OnViewManagerDisconnected(ViewManager* view_manager) override;

  // ViewObserver methods:
  void OnViewBoundsChanged(View* view,
                           const Rect& old_bounds,
                           const Rect& new_bounds) override;
  void OnViewDestroyed(View* view) override;
  void OnViewInputEvent(View* view, const EventPtr& event) override;

  void Load(URLResponsePtr response);

  URLResponsePtr response_;
  ServiceProviderImpl exported_services_;
  scoped_ptr<ServiceProvider> embedder_service_provider_;
  ShellPtr shell_;
  LazyInterfacePtr<NavigatorHost> navigator_host_;
  blink::WebView* web_view_;
  View* root_;
  ViewManagerClientFactory view_manager_client_factory_;
  scoped_ptr<WebLayerTreeViewImpl> web_layer_tree_view_impl_;
  scoped_refptr<base::MessageLoopProxy> compositor_thread_;
  WebMediaPlayerFactory* web_media_player_factory_;

  base::WeakPtrFactory<HTMLDocumentView> weak_factory_;
  DISALLOW_COPY_AND_ASSIGN(HTMLDocumentView);
};

}  // namespace mojo

#endif  // MOJO_SERVICES_HTML_VIEWER_HTML_DOCUMENT_VIEW_H_
