#!/usr/bin/env python
# Copyright (c) 2012 The Chromium Authors. All rights reserved.
# Use of this source code is governed by a BSD-style license that can be
# found in the LICENSE file.

"""code generator for GL/GLES extension wrangler."""

import optparse
import os
import collections
import re
import platform
import sys
from subprocess import call
from collections import namedtuple

HEADER_PATHS = [
  '../../third_party/khronos',
  '../../third_party/mesa_headers',
  '.',
  '../../gpu',
]

UNCONDITIONALLY_BOUND_EXTENSIONS = set([
  'WGL_ARB_extensions_string',
  'WGL_EXT_extensions_string',
  'WGL_ARB_pbuffer',
  'GL_CHROMIUM_gles_depth_binding_hack', # crbug.com/448206
  'GL_CHROMIUM_glgetstringi_hack', # crbug.com/470396
  'GL_CHROMIUM_egl_khr_fence_sync_hack', # crbug.com/504758
  'GL_CHROMIUM_egl_android_native_fence_sync_hack', # crbug.com/775707
])

"""Function binding conditions can be specified manually by supplying a versions
array instead of the names array. Each version has the following keys:
   name: Mandatory. Name of the function. Multiple versions can have the same
         name but different conditions.
   extensions: Extra Extensions for which the function is bound. Only needed
               in some cases where the extension cannot be parsed from the
               headers.
   explicit_only: if True, only extensions in 'extensions' are considered.
   is_optional: True if the GetProcAddress can return NULL for the
                function.  This may happen for example when functions
                are added to a new version of an extension, but the
                extension string is not modified.
By default, the function gets its name from the first name in its names or
versions array. This can be overridden by supplying a 'known_as' key.

"""
GL_FUNCTIONS = [
{ 'return_type': 'void',
  'names': ['glActiveShaderProgram'],
  'arguments': 'GLuint pipeline, GLuint program', },
{ 'return_type': 'void',
  'names': ['glActiveTexture'],
  'arguments': 'GLenum texture', },
{ 'return_type': 'void',
  'names': ['glAttachShader'],
  'arguments': 'GLuint program, GLuint shader', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glBeginQuery' },
               { 'name': 'glBeginQueryARB' },
               { 'name': 'glBeginQueryEXT',
                 'extensions': ['GL_EXT_occlusion_query_boolean'] }],
  'arguments': 'GLenum target, GLuint id', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glBeginTransformFeedback' },
               { 'name': 'glBeginTransformFeedbackEXT',
                 'extension': ['GL_EXT_transform_feedback'] }],
  'arguments': 'GLenum primitiveMode', },
{ 'return_type': 'void',
  'names': ['glBindAttribLocation'],
  'arguments': 'GLuint program, GLuint index, const char* name', },
{ 'return_type': 'void',
  'names': ['glBindBuffer'],
  'arguments': 'GLenum target, GLuint buffer', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glBindBufferBase' },
               { 'name': 'glBindBufferBaseEXT',
                 'extension': ['GL_EXT_transform_feedback'] }],
  'arguments': 'GLenum target, GLuint index, GLuint buffer', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glBindBufferRange' },
               { 'name': 'glBindBufferRangeEXT',
                 'extension': ['GL_EXT_transform_feedback'] }],
  'arguments': 'GLenum target, GLuint index, GLuint buffer, GLintptr offset, '
               'GLsizeiptr size', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glBindFragDataLocation',
                 'extensions': ['GL_ARB_blend_func_extended'] },
               { 'name': 'glBindFragDataLocationEXT',
                 'extensions': ['GL_EXT_blend_func_extended'] }],
  'arguments': 'GLuint program, GLuint colorNumber, const char* name', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glBindFragDataLocationIndexed',
                 'extensions': ['GL_ARB_blend_func_extended'] },
               { 'name': 'glBindFragDataLocationIndexedEXT',
                 'extensions': ['GL_EXT_blend_func_extended'] }],
  'arguments':
      'GLuint program, GLuint colorNumber, GLuint index, const char* name',
},
{ 'return_type': 'void',
  'versions': [{ 'name': 'glBindFragmentInputLocationCHROMIUM',
                 'extensions': ['GL_CHROMIUM_path_rendering'] }],
  'arguments': 'GLuint program, GLint location, const char* name', },
{ 'return_type': 'void',
  'names': ['glBindFramebufferEXT', 'glBindFramebuffer'],
  'arguments': 'GLenum target, GLuint framebuffer', },
{ 'return_type': 'void',
  'known_as': 'glBindImageTextureEXT',
  'versions': [{ 'name': 'glBindImageTexture',
                 'extensions': ['GL_ARB_shader_image_load_store'] },
               { 'name': 'glBindImageTextureEXT',
                 'extensions': ['GL_EXT_shader_image_load_store'] }],
  'arguments': 'GLuint index, GLuint texture, GLint level, GLboolean layered,'
               'GLint layer, GLenum access, GLint format', },
{ 'return_type': 'void',
  'names': ['glBindProgramPipeline'],
  'arguments': 'GLuint pipeline', },
{ 'return_type': 'void',
  'names': ['glBindRenderbufferEXT', 'glBindRenderbuffer'],
  'arguments': 'GLenum target, GLuint renderbuffer', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glBindSampler',
                 'extensions': ['GL_ARB_sampler_objects'] }],
  'arguments': 'GLuint unit, GLuint sampler', },
{ 'return_type': 'void',
  'names': ['glBindTexture'],
  'arguments': 'GLenum target, GLuint texture', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glBindTransformFeedback',
                 'extensions': ['GL_ARB_transform_feedback2'] }],
  'arguments': 'GLenum target, GLuint id', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glBindUniformLocationCHROMIUM',
                 'extensions': ['GL_CHROMIUM_bind_uniform_location'] }],
  'arguments': 'GLuint program, GLint location, const char* name' },
{ 'return_type': 'void',
  'known_as': 'glBindVertexArrayOES',
  'versions': [{ 'name': 'glBindVertexArray',
                 'extensions': ['GL_ARB_vertex_array_object'], },
               { 'name': 'glBindVertexArrayOES' },
               { 'name': 'glBindVertexArrayAPPLE',
                 'extensions': ['GL_APPLE_vertex_array_object'] }],
  'arguments': 'GLuint array' },
{ 'return_type': 'void',
  'names': ['glBindVertexBuffer'],
  'arguments': 'GLuint bindingindex, GLuint buffer, GLintptr offset, '
               'GLsizei stride', },
{ 'return_type': 'void',
  'known_as': 'glBlendBarrierKHR',
  'versions': [{ 'name': 'glBlendBarrierNV',
                 'extensions': ['GL_NV_blend_equation_advanced'] },
               { 'name': 'glBlendBarrierKHR',
                 'extensions': ['GL_KHR_blend_equation_advanced'] }],
  'arguments': 'void' },
{ 'return_type': 'void',
  'names': ['glBlendColor'],
  'arguments': 'GLclampf red, GLclampf green, GLclampf blue, GLclampf alpha', },
{ 'return_type': 'void',
  'names': ['glBlendEquation'],
  'arguments': ' GLenum mode ', },
{ 'return_type': 'void',
  'known_as': 'glBlendEquationiOES',
  'versions': [
    { 'name': 'glBlendEquationi' },
    { 'name': 'glBlendEquationiOES', 'extensions':
      ['GL_OES_draw_buffers_indexed'] }
  ],
  'arguments': ' GLuint buf, GLenum mode ', },
{ 'return_type': 'void',
  'names': ['glBlendEquationSeparate'],
  'arguments': 'GLenum modeRGB, GLenum modeAlpha', },
{ 'return_type': 'void',
  'known_as': 'glBlendEquationSeparateiOES',
  'versions': [
    { 'name': 'glBlendEquationSeparatei' },
    { 'name': 'glBlendEquationSeparateiOES', 'extensions':
      ['GL_OES_draw_buffers_indexed'] }
  ],
  'arguments': 'GLuint buf, GLenum modeRGB, GLenum modeAlpha', },
{ 'return_type': 'void',
  'names': ['glBlendFunc'],
  'arguments': 'GLenum sfactor, GLenum dfactor', },
{ 'return_type': 'void',
  'known_as': 'glBlendFunciOES',
  'versions': [
    { 'name': 'glBlendFunci' },
    { 'name': 'glBlendFunciOES', 'extensions': ['GL_OES_draw_buffers_indexed'] }
  ],
  'arguments': 'GLuint buf, GLenum sfactor, GLenum dfactor', },
{ 'return_type': 'void',
  'names': ['glBlendFuncSeparate'],
  'arguments':
      'GLenum srcRGB, GLenum dstRGB, GLenum srcAlpha, GLenum dstAlpha', },
{ 'return_type': 'void',
  'known_as': 'glBlendFuncSeparateiOES',
  'versions': [
    { 'name': 'glBlendFuncSeparatei' },
    { 'name': 'glBlendFuncSeparateiOES', 'extensions':
      ['GL_OES_draw_buffers_indexed'] }
  ],
  'arguments':
      'GLuint buf, GLenum srcRGB, GLenum dstRGB, GLenum srcAlpha, '
      'GLenum dstAlpha', },
{ 'return_type': 'void',
  'versions' : [{'name': 'glBlitFramebuffer',
                 'extensions': ['GL_ARB_framebuffer_object']},
                {'name': 'glBlitFramebufferANGLE'},
                {'name': 'glBlitFramebufferEXT'}],
  'arguments': 'GLint srcX0, GLint srcY0, GLint srcX1, GLint srcY1, '
               'GLint dstX0, GLint dstY0, GLint dstX1, GLint dstY1, '
               'GLbitfield mask, GLenum filter', },
{ 'return_type': 'void',
  'names': ['glBufferData'],
  'arguments':
      'GLenum target, GLsizeiptr size, const void* data, GLenum usage', },
{ 'return_type': 'void',
  'names': ['glBufferSubData'],
  'arguments':
      'GLenum target, GLintptr offset, GLsizeiptr size, const void* data', },
{ 'return_type': 'GLenum',
  'names': ['glCheckFramebufferStatusEXT',
            'glCheckFramebufferStatus'],
  'arguments': 'GLenum target',
  'logging_code': """
  GL_SERVICE_LOG("GL_RESULT: " << GLEnums::GetStringEnum(result));
""", },
{ 'return_type': 'void',
  'names': ['glClear'],
  'arguments': 'GLbitfield mask', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glClearBufferfi' }],
  'arguments': 'GLenum buffer, GLint drawbuffer, const GLfloat depth, '
               'GLint stencil', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glClearBufferfv' }],
  'arguments': 'GLenum buffer, GLint drawbuffer, const GLfloat* value', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glClearBufferiv' }],
  'arguments': 'GLenum buffer, GLint drawbuffer, const GLint* value', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glClearBufferuiv' }],
  'arguments': 'GLenum buffer, GLint drawbuffer, const GLuint* value', },
{ 'return_type': 'void',
  'names': ['glClearColor'],
  'arguments': 'GLclampf red, GLclampf green, GLclampf blue, GLclampf alpha', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glClearDepth',
                 'extensions': ['GL_CHROMIUM_gles_depth_binding_hack'] }],
  'arguments': 'GLclampd depth', },
{ 'return_type': 'void',
  'names': ['glClearDepthf'],
  'arguments': 'GLclampf depth', },
{ 'return_type': 'void',
  'names': ['glClearStencil'],
  'arguments': 'GLint s', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glClearTexImage',
                 'extensions': ['GL_ARB_clear_texture'] },
               { 'name': 'glClearTexImageEXT',
                 'extensions': ['GL_EXT_clear_texture'] }],
  'arguments':
      'GLuint texture, GLint level, GLenum format, GLenum type, '
      'const GLvoid* data', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glClearTexSubImage',
                 'extensions': ['GL_EXT_clear_texture'] },
               { 'name': 'glClearTexSubImageEXT',
                 'extensions': ['GL_EXT_clear_texture'] }],
  'arguments':
      'GLuint texture, GLint level, GLint xoffset, GLint yoffset, '
      'GLint zoffset, GLint width, GLint height, GLint depth, GLenum format, '
      'GLenum type, const GLvoid* data', },
{ 'return_type': 'GLenum',
  'versions': [{ 'name': 'glClientWaitSync',
                 'extensions': ['GL_ARB_sync'] }],
  'arguments': 'GLsync sync, GLbitfield flags, GLuint64 timeout', },
{ 'return_type': 'GLenum',
  'versions': [{ 'name': 'glClientWaitSyncAPPLE',
                 'extensions': ['GL_APPLE_sync'] }],
  'arguments': 'GLsync sync, GLbitfield flags, GLuint64 timeout', },
{ 'return_type': 'void',
  'names': ['glColorMask'],
  'arguments':
      'GLboolean red, GLboolean green, GLboolean blue, GLboolean alpha', },
{ 'return_type': 'void',
  'known_as': 'glColorMaskiOES',
  'versions': [
    { 'name': 'glColorMaski' },
    { 'name': 'glColorMaskiOES', 'extensions': ['GL_OES_draw_buffers_indexed'] }
  ],
  'arguments':
      'GLuint buf, GLboolean red, GLboolean green, GLboolean blue, '
      'GLboolean alpha', },
{ 'return_type': 'void',
  'names': ['glCompileShader'],
  'arguments': 'GLuint shader', },
{ 'return_type': 'void',
  'names': ['glCompressedTexImage2D'],
  'arguments':
      'GLenum target, GLint level, GLenum internalformat, GLsizei width, '
      'GLsizei height, GLint border, GLsizei imageSize, const void* data', },
{ 'return_type': 'void',
  'versions': [{'name': 'glCompressedTexImage2DRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLenum target, GLint level, GLenum internalformat, GLsizei width, '
      'GLsizei height, GLint border, GLsizei imageSize, GLsizei dataSize, '
      'const void* data', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glCompressedTexImage3D' }],
  'arguments':
      'GLenum target, GLint level, GLenum internalformat, GLsizei width, '
      'GLsizei height, GLsizei depth, GLint border, GLsizei imageSize, '
      'const void* data', },
{ 'return_type': 'void',
  'versions': [{'name': 'glCompressedTexImage3DRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLenum target, GLint level, GLenum internalformat, GLsizei width, '
      'GLsizei height, GLsizei depth, GLint border, GLsizei imageSize, '
      'GLsizei dataSize, const void* data', },
{ 'return_type': 'void',
  'names': ['glCompressedTexSubImage2D'],
  'arguments':
      'GLenum target, GLint level, GLint xoffset, GLint yoffset, '
      'GLsizei width, GLsizei height, GLenum format, GLsizei imageSize, '
      'const void* data', },
{ 'return_type': 'void',
  'versions': [{'name': 'glCompressedTexSubImage2DRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLenum target, GLint level, GLint xoffset, GLint yoffset, '
      'GLsizei width, GLsizei height, GLenum format, GLsizei imageSize, '
      'GLsizei dataSize, const void* data', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glCompressedTexSubImage3D' }],
  'arguments':
      'GLenum target, GLint level, GLint xoffset, GLint yoffset, '
      'GLint zoffset, GLsizei width, GLsizei height, GLsizei depth, '
      'GLenum format, GLsizei imageSize, const void* data', },
{ 'return_type': 'void',
  'versions': [{'name': 'glCompressedTexSubImage3DRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLenum target, GLint level, GLint xoffset, GLint yoffset, '
      'GLint zoffset, GLsizei width, GLsizei height, GLsizei depth, '
      'GLenum format, GLsizei imageSize, GLsizei dataSize, '
      'const void* data', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glCopyBufferSubData' }],
  'arguments':
      'GLenum readTarget, GLenum writeTarget, GLintptr readOffset, '
      'GLintptr writeOffset, GLsizeiptr size', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glCopySubTextureCHROMIUM',
                 'extensions': ['GL_CHROMIUM_copy_texture'], }],
  'arguments':
      'GLuint sourceId, GLint sourceLevel, GLenum destTarget, GLuint destId, '
      'GLint destLevel, GLint xoffset, GLint yoffset, GLint x, GLint y, '
      'GLsizei width, GLsizei height, GLboolean unpackFlipY, '
      'GLboolean unpackPremultiplyAlpha, GLboolean unpackUnmultiplyAlpha', },
{ 'return_type': 'void',
  'names': ['glCopyTexImage2D'],
  'arguments':
      'GLenum target, GLint level, GLenum internalformat, GLint x, GLint y, '
      'GLsizei width, GLsizei height, GLint border', },
{ 'return_type': 'void',
  'names': ['glCopyTexSubImage2D'],
  'arguments':
      'GLenum target, GLint level, GLint xoffset, '
      'GLint yoffset, GLint x, GLint y, GLsizei width, GLsizei height', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glCopyTexSubImage3D' }],
  'arguments':
      'GLenum target, GLint level, GLint xoffset, GLint yoffset, '
      'GLint zoffset, GLint x, GLint y, GLsizei width, GLsizei height', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glCopyTextureCHROMIUM',
                 'extensions': ['GL_CHROMIUM_copy_texture'], }],
  'arguments':
      'GLuint sourceId, GLint sourceLevel, GLenum destTarget, GLuint destId, '
      'GLint destLevel, GLint internalFormat, GLenum destType, '
      'GLboolean unpackFlipY, GLboolean unpackPremultiplyAlpha, '
      'GLboolean unpackUnmultiplyAlpha', },
{ 'return_type': 'void',
  'names': ['glCoverageModulationNV'],
  'versions': [{ 'name': 'glCoverageModulationNV',
                 'extensions': ['GL_NV_framebuffer_mixed_samples'] },
               { 'name': 'glCoverageModulationCHROMIUM',
                 'extensions': ['GL_CHROMIUM_framebuffer_mixed_samples'] }],
  'arguments': 'GLenum components'},
{ 'return_type': 'void',
  'names': ['glCoverFillPathInstancedNV'],
  'versions': [{ 'name': 'glCoverFillPathInstancedNV',
                 'extensions': ['GL_NV_path_rendering'] },
               { 'name': 'glCoverFillPathInstancedCHROMIUM',
                 'extensions': ['GL_CHROMIUM_path_rendering'] }],
  'arguments': 'GLsizei numPaths, GLenum pathNameType, const void* paths, '
  'GLuint pathBase, GLenum coverMode, GLenum transformType, '
  'const GLfloat* transformValues' },
{ 'return_type': 'void',
  'names': ['glCoverFillPathNV'],
  'versions': [{ 'name': 'glCoverFillPathNV',
                 'extensions': ['GL_NV_path_rendering'] },
               { 'name': 'glCoverFillPathCHROMIUM',
                 'extensions': ['GL_CHROMIUM_path_rendering'] }],
  'arguments': 'GLuint path, GLenum coverMode' },
{ 'return_type': 'void',
  'names': ['glCoverStrokePathInstancedNV'],
  'versions': [{ 'name': 'glCoverStrokePathInstancedNV',
                 'extensions': ['GL_NV_path_rendering'] },
               { 'name': 'glCoverStrokePathInstancedCHROMIUM',
                 'extensions': ['GL_CHROMIUM_path_rendering'] }],
  'arguments': 'GLsizei numPaths, GLenum pathNameType, const void* paths, '
  'GLuint pathBase, GLenum coverMode, GLenum transformType, '
  'const GLfloat* transformValues' },
{ 'return_type': 'void',
  'names': ['glCoverStrokePathNV'],
  'versions': [{ 'name': 'glCoverStrokePathNV',
                 'extensions': ['GL_NV_path_rendering'] },
               { 'name': 'glCoverStrokePathCHROMIUM',
                 'extensions': ['GL_CHROMIUM_path_rendering'] }],
  'arguments': 'GLuint name, GLenum coverMode' },
{ 'return_type': 'void',
  'names': [ 'glCreateMemoryObjectsEXT' ],
  'arguments': 'GLsizei n, GLuint* memoryObjects', },
{ 'return_type': 'GLuint',
  'names': ['glCreateProgram'],
  'arguments': 'void', },
{ 'return_type': 'GLuint',
  'names': ['glCreateShader'],
  'arguments': 'GLenum type', },
{ 'return_type': 'GLuint',
  'names': ['glCreateShaderProgramv'],
  'arguments': 'GLenum type, GLsizei count, const char* const* strings', },
{ 'return_type': 'void',
  'names': ['glCullFace'],
  'arguments': 'GLenum mode', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glDebugMessageCallback' },
               { 'name': 'glDebugMessageCallbackKHR',
                 'extensions': ['GL_KHR_debug'] }],
  'arguments': 'GLDEBUGPROC callback, const void* userParam', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glDebugMessageControl' },
               { 'name': 'glDebugMessageControlKHR',
                 'extensions': ['GL_KHR_debug'] }],
  'arguments':
    'GLenum source, GLenum type, GLenum severity, GLsizei count, '
    'const GLuint* ids, GLboolean enabled', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glDebugMessageInsert' },
               { 'name': 'glDebugMessageInsertKHR',
                 'extensions': ['GL_KHR_debug'] }],
  'arguments':
    'GLenum source, GLenum type, GLuint id, GLenum severity, '
    'GLsizei length, const char* buf', },
{ 'return_type': 'void',
  'names': ['glDeleteBuffers'],
  'known_as': 'glDeleteBuffersARB',
  'arguments': 'GLsizei n, const GLuint* buffers', },
{ 'return_type': 'void',
  'known_as': 'glDeleteFencesAPPLE',
  'versions': [{ 'name': 'glDeleteFencesAPPLE',
                 'extensions': ['GL_APPLE_fence'] }],
  'arguments': 'GLsizei n, const GLuint* fences', },
{ 'return_type': 'void',
  'names': ['glDeleteFencesNV'],
  'arguments': 'GLsizei n, const GLuint* fences', },
{ 'return_type': 'void',
  'names': ['glDeleteFramebuffersEXT', 'glDeleteFramebuffers'],
  'arguments': 'GLsizei n, const GLuint* framebuffers', },
{ 'return_type': 'void',
  'names': [ 'glDeleteMemoryObjectsEXT' ],
  'versions': [{ 'name': 'glDeleteMemoryObjectsEXT',
                 'extensions': ['GL_EXT_memory_object'] }],
  'arguments': 'GLsizei n, const GLuint* memoryObjects', },
{ 'return_type': 'void',
  'names': ['glDeletePathsNV'],
  'versions': [{ 'name': 'glDeletePathsNV',
                 'extensions': ['GL_NV_path_rendering'] },
               { 'name': 'glDeletePathsCHROMIUM',
                 'extensions': ['GL_CHROMIUM_path_rendering'] }],
  'arguments': 'GLuint path, GLsizei range' },
{ 'return_type': 'void',
  'names': ['glDeleteProgram'],
  'arguments': 'GLuint program', },
{ 'return_type': 'void',
  'names': ['glDeleteProgramPipelines'],
  'arguments': 'GLsizei n, const GLuint* pipelines', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glDeleteQueries' },
               { 'name': 'glDeleteQueriesARB'},
               { 'name': 'glDeleteQueriesEXT',
                 'extensions': ['GL_EXT_occlusion_query_boolean'] }],
  'arguments': 'GLsizei n, const GLuint* ids', },
{ 'return_type': 'void',
  'names': ['glDeleteRenderbuffersEXT', 'glDeleteRenderbuffers'],
  'arguments': 'GLsizei n, const GLuint* renderbuffers', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glDeleteSamplers',
                 'extensions': ['GL_ARB_sampler_objects'] }],
  'arguments': 'GLsizei n, const GLuint* samplers', },
{ 'return_type': 'void',
  'names': ['glDeleteSemaphoresEXT'],
  'arguments': 'GLsizei n, const GLuint* semaphores', },
{ 'return_type': 'void',
  'names': ['glDeleteShader'],
  'arguments': 'GLuint shader', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glDeleteSync',
                 'extensions': ['GL_ARB_sync'] }],
  'arguments': 'GLsync sync', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glDeleteSyncAPPLE',
                 'extensions': ['GL_APPLE_sync'] }],
  'arguments': 'GLsync sync', },
{ 'return_type': 'void',
  'names': ['glDeleteTextures'],
  'arguments': 'GLsizei n, const GLuint* textures', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glDeleteTransformFeedbacks',
                 'extensions': ['GL_ARB_transform_feedback2'] }],
  'arguments': 'GLsizei n, const GLuint* ids', },
{ 'return_type': 'void',
  'known_as': 'glDeleteVertexArraysOES',
  'versions': [{ 'name': 'glDeleteVertexArrays',
                 'extensions': ['GL_ARB_vertex_array_object'], },
               { 'name': 'glDeleteVertexArraysOES' },
               { 'name': 'glDeleteVertexArraysAPPLE',
                 'extensions': ['GL_APPLE_vertex_array_object'] }],
  'arguments': 'GLsizei n, const GLuint* arrays' },
{ 'return_type': 'void',
  'names': ['glDepthFunc'],
  'arguments': 'GLenum func', },
{ 'return_type': 'void',
  'names': ['glDepthMask'],
  'arguments': 'GLboolean flag', },
{ 'return_type': 'void',
 'versions': [{ 'name': 'glDepthRange',
                'extensions': ['GL_CHROMIUM_gles_depth_binding_hack'] }],
  'arguments': 'GLclampd zNear, GLclampd zFar', },
{ 'return_type': 'void',
  'names': ['glDepthRangef'],
  'arguments': 'GLclampf zNear, GLclampf zFar', },
{ 'return_type': 'void',
  'names': ['glDetachShader'],
  'arguments': 'GLuint program, GLuint shader', },
{ 'return_type': 'void',
  'names': ['glDisable'],
  'arguments': 'GLenum cap', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glDisableExtensionANGLE',
                 'extensions': ['GL_ANGLE_request_extension'] }],
  'arguments': 'const char* name', },
{ 'return_type': 'void',
  'known_as': 'glDisableiOES',
  'versions': [
    { 'name': 'glDisablei' },
    { 'name': 'glDisableiOES', 'extensions': ['GL_OES_draw_buffers_indexed'] }
  ],
  'arguments': 'GLenum target, GLuint index', },
{ 'return_type': 'void',
  'names': ['glDisableVertexAttribArray'],
  'arguments': 'GLuint index', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glDiscardFramebufferEXT',
                 'extensions': ['GL_EXT_discard_framebuffer'] }],
  'arguments': 'GLenum target, GLsizei numAttachments, '
      'const GLenum* attachments' },
{ 'return_type': 'void',
  'names': ['glDispatchCompute'],
  'arguments': 'GLuint numGroupsX, GLuint numGroupsY, GLuint numGroupsZ', },
{ 'return_type': 'void',
  'names': ['glDispatchComputeIndirect'],
  'arguments': 'GLintptr indirect', },
{ 'return_type': 'void',
  'names': ['glDrawArrays'],
  'arguments': 'GLenum mode, GLint first, GLsizei count', },
{ 'return_type': 'void',
  'names': ['glDrawArraysIndirect'],
  'arguments': 'GLenum mode, const void* indirect', },
{ 'return_type': 'void',
  'known_as': 'glDrawArraysInstancedANGLE',
  'names': ['glDrawArraysInstancedARB', 'glDrawArraysInstancedANGLE',
            'glDrawArraysInstanced'],
  'arguments': 'GLenum mode, GLint first, GLsizei count, GLsizei primcount', },
{ 'return_type': 'void',
  'known_as': 'glDrawArraysInstancedBaseInstanceANGLE',
  #TODO(shrekshao): workaround when native support not available for cmd decoder
  'versions' : [{ 'name': 'glDrawArraysInstancedBaseInstance',
                 'extensions': ['GL_ARB_base_instance'] },
                { 'name': 'glDrawArraysInstancedBaseInstanceEXT' },
                { 'name': 'glDrawArraysInstancedBaseInstanceANGLE',
                 'extensions': ['GL_ANGLE_base_vertex_base_instance'] }],
  'arguments': 'GLenum mode, GLint first, GLsizei count, GLsizei primcount, '
  'GLuint baseinstance', },
{ 'return_type': 'void',
  'names': ['glDrawBuffer'],
  'arguments': 'GLenum mode', },
{ 'return_type': 'void',
  'names': ['glDrawBuffersARB', 'glDrawBuffersEXT', 'glDrawBuffers'],
  'arguments': 'GLsizei n, const GLenum* bufs', },
{ 'return_type': 'void',
  'names': ['glDrawElements'],
  'arguments':
      'GLenum mode, GLsizei count, GLenum type, const void* indices', },
{ 'return_type': 'void',
  'names': ['glDrawElementsIndirect'],
  'arguments': 'GLenum mode, GLenum type, const void* indirect', },
{ 'return_type': 'void',
  'known_as': 'glDrawElementsInstancedANGLE',
  'names': ['glDrawElementsInstancedARB', 'glDrawElementsInstancedANGLE',
            'glDrawElementsInstanced'],
  'arguments':
      'GLenum mode, GLsizei count, GLenum type, const void* indices, '
      'GLsizei primcount', },
{ 'return_type': 'void',
  'known_as': 'glDrawElementsInstancedBaseVertexBaseInstanceANGLE',
  #TODO(shrekshao): workaround when native support not available for cmd decoder
  'versions' : [{ 'name': 'glDrawElementsInstancedBaseVertexBaseInstance',
                 'extensions': ['GL_ARB_base_instance'] },
                { 'name': 'glDrawElementsInstancedBaseVertexBaseInstanceEXT' },
                { 'name': 'glDrawElementsInstancedBaseVertexBaseInstanceANGLE',
                 'extensions': ['GL_ANGLE_base_vertex_base_instance'] }],
  'arguments':
      'GLenum mode, GLsizei count, GLenum type, const void* indices, '
      'GLsizei primcount, GLint baseVertex, GLuint baseInstance', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glDrawRangeElements' }],
  'arguments': 'GLenum mode, GLuint start, GLuint end, GLsizei count, '
               'GLenum type, const void* indices', },
{ 'return_type': 'void',
  'names': ['glEGLImageTargetRenderbufferStorageOES'],
  'arguments': 'GLenum target, GLeglImageOES image', },
{ 'return_type': 'void',
  'names': ['glEGLImageTargetTexture2DOES'],
  'arguments': 'GLenum target, GLeglImageOES image', },
{ 'return_type': 'void',
  'names': ['glEnable'],
  'arguments': 'GLenum cap', },
{ 'return_type': 'void',
  'known_as': 'glEnableiOES',
  'versions': [
    { 'name': 'glEnablei' },
    { 'name': 'glEnableiOES', 'extensions': ['GL_OES_draw_buffers_indexed'] }
  ],
  'arguments': 'GLenum target, GLuint index', },
{ 'return_type': 'void',
  'names': ['glEnableVertexAttribArray'],
  'arguments': 'GLuint index', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glEndQuery' },
               { 'name': 'glEndQueryARB' },
               { 'name': 'glEndQueryEXT',
                 'extensions': ['GL_EXT_occlusion_query_boolean'] }],
  'arguments': 'GLenum target', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glEndTilingQCOM',
                 'extension': ['GL_QCOM_tiled_rendering'] }],
  'arguments': 'GLbitfield preserveMask', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glEndTransformFeedback' },
               { 'name': 'glEndTransformFeedbackEXT',
                 'extension': ['GL_EXT_transform_feedback'] }],
  'arguments': 'void', },
{ 'return_type': 'GLsync',
  'versions': [{ 'name': 'glFenceSync',
                 'extensions': ['GL_ARB_sync'] }],
  'arguments': 'GLenum condition, GLbitfield flags', },
{ 'return_type': 'GLsync',
  'versions': [{ 'name': 'glFenceSyncAPPLE',
                 'extensions': ['GL_APPLE_sync'] }],
  'arguments': 'GLenum condition, GLbitfield flags', },
{ 'return_type': 'void',
  'names': ['glFinish'],
  'arguments': 'void', },
{ 'return_type': 'void',
  'known_as': 'glFinishFenceAPPLE',
  'versions': [{ 'name': 'glFinishFenceAPPLE',
                 'extensions': ['GL_APPLE_fence'] }],
  'arguments': 'GLuint fence', },
{ 'return_type': 'void',
  'names': ['glFinishFenceNV'],
  'arguments': 'GLuint fence', },
{ 'return_type': 'void',
  'names': ['glFlush'],
  'arguments': 'void', },
{ 'return_type': 'void',
  'versions': [{'name': 'glFlushMappedBufferRange',
                'extensions': ['GL_ARB_map_buffer_range']},
               {'name': 'glFlushMappedBufferRangeEXT'}],
  'arguments': 'GLenum target, GLintptr offset, GLsizeiptr length', },
{ 'return_type': 'void',
  'versions': [{'name': 'glFramebufferParameteri'},
               {'name': 'glFramebufferParameteriMESA',
                'extensions': ['GL_MESA_framebuffer_flip_y']}],
  'arguments': 'GLenum target, GLenum pname, GLint param', },
{ 'return_type': 'void',
  'names': ['glFramebufferRenderbufferEXT', 'glFramebufferRenderbuffer'],
  'arguments':
      'GLenum target, GLenum attachment, GLenum renderbuffertarget, '
      'GLuint renderbuffer', },
{ 'return_type': 'void',
  'names': ['glFramebufferTexture2DEXT', 'glFramebufferTexture2D'],
  'arguments':
      'GLenum target, GLenum attachment, GLenum textarget, GLuint texture, '
      'GLint level', },
{ 'return_type': 'void',
 'versions': [{'name': 'glFramebufferTexture2DMultisampleEXT'},
              {'name': 'glFramebufferTexture2DMultisampleIMG'}],
  'arguments':
      'GLenum target, GLenum attachment, GLenum textarget, GLuint texture, '
      'GLint level, GLsizei samples', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glFramebufferTextureLayer' }],
  'arguments': 'GLenum target, GLenum attachment, GLuint texture, GLint level, '
               'GLint layer', },
{ 'return_type': 'void',
  'versions': [{'name': 'glFramebufferTextureMultiviewOVR',
                'extensions': ['GL_OVR_multiview2']}],
  'arguments': 'GLenum target, GLenum attachment, GLuint texture, GLint level, '
               'GLint baseViewIndex, GLsizei numViews', },
{ 'return_type': 'void',
  'names': ['glFrontFace'],
  'arguments': 'GLenum mode', },
{ 'return_type': 'void',
  'names': ['glGenBuffers'],
  'known_as': 'glGenBuffersARB',
  'arguments': 'GLsizei n, GLuint* buffers', },
{ 'return_type': 'void',
  'names': ['glGenerateMipmapEXT', 'glGenerateMipmap'],
  'arguments': 'GLenum target', },
{ 'return_type': 'void',
  'known_as': 'glGenFencesAPPLE',
  'versions': [{ 'name': 'glGenFencesAPPLE',
                 'extensions': ['GL_APPLE_fence'] }],
  'arguments': 'GLsizei n, GLuint* fences', },
{ 'return_type': 'void',
  'names': ['glGenFencesNV'],
  'arguments': 'GLsizei n, GLuint* fences', },
{ 'return_type': 'void',
  'names': ['glGenFramebuffersEXT', 'glGenFramebuffers'],
  'arguments': 'GLsizei n, GLuint* framebuffers', },
{ 'return_type': 'GLuint',
  'names': ['glGenPathsNV'],
  'versions': [{ 'name': 'glGenPathsNV',
                 'extensions': ['GL_NV_path_rendering'] },
               { 'name': 'glGenPathsCHROMIUM',
                 'extensions': ['GL_CHROMIUM_path_rendering'] }],
  'arguments': 'GLsizei range' },
{ 'return_type': 'GLuint',
  'names': ['glGenProgramPipelines'],
  'arguments': 'GLsizei n, GLuint* pipelines' },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glGenQueries' },
               { 'name': 'glGenQueriesARB', },
               { 'name' : 'glGenQueriesEXT',
                 'extensions': ['GL_EXT_occlusion_query_boolean'] }],
  'arguments': 'GLsizei n, GLuint* ids', },
{ 'return_type': 'void',
  'names': ['glGenRenderbuffersEXT', 'glGenRenderbuffers'],
  'arguments': 'GLsizei n, GLuint* renderbuffers', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glGenSamplers',
                 'extensions': ['GL_ARB_sampler_objects'] }],
  'arguments': 'GLsizei n, GLuint* samplers', },
{ 'return_type': 'void',
  'names': ['glGenSemaphoresEXT'],
  'arguments': 'GLsizei n, GLuint* semaphores', },
{ 'return_type': 'void',
  'names': ['glGenTextures'],
  'arguments': 'GLsizei n, GLuint* textures', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glGenTransformFeedbacks',
                 'extensions': ['GL_ARB_transform_feedback2'] }],
  'arguments': 'GLsizei n, GLuint* ids', },
{ 'return_type': 'void',
  'known_as': 'glGenVertexArraysOES',
  'versions': [{ 'name': 'glGenVertexArrays',
                 'extensions': ['GL_ARB_vertex_array_object'], },
               { 'name': 'glGenVertexArraysOES' },
               { 'name': 'glGenVertexArraysAPPLE',
                 'extensions': ['GL_APPLE_vertex_array_object'] }],
  'arguments': 'GLsizei n, GLuint* arrays', },
{ 'return_type': 'void',
  'names': ['glGetActiveAttrib'],
  'arguments':
      'GLuint program, GLuint index, GLsizei bufsize, GLsizei* length, '
      'GLint* size, GLenum* type, char* name', },
{ 'return_type': 'void',
  'names': ['glGetActiveUniform'],
  'arguments':
      'GLuint program, GLuint index, GLsizei bufsize, GLsizei* length, '
      'GLint* size, GLenum* type, char* name', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glGetActiveUniformBlockiv' }],
  'arguments': 'GLuint program, GLuint uniformBlockIndex, GLenum pname, '
               'GLint* params', },
{ 'return_type': 'void',
  'versions': [{'name': 'glGetActiveUniformBlockivRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLuint program, GLuint uniformBlockIndex, GLenum pname, '
      'GLsizei bufSize, GLsizei* length, GLint* params', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glGetActiveUniformBlockName' }],
  'arguments': 'GLuint program, GLuint uniformBlockIndex, GLsizei bufSize, '
               'GLsizei* length, char* uniformBlockName', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glGetActiveUniformsiv' }],
  'arguments': 'GLuint program, GLsizei uniformCount, '
               'const GLuint* uniformIndices, GLenum pname, GLint* params', },
{ 'return_type': 'void',
  'names': ['glGetAttachedShaders'],
  'arguments':
      'GLuint program, GLsizei maxcount, GLsizei* count, GLuint* shaders', },
{ 'return_type': 'GLint',
  'names': ['glGetAttribLocation'],
  'arguments': 'GLuint program, const char* name', },
{ 'return_type': 'void',
  'names': ['glGetBooleani_v'],
  'arguments': 'GLenum target, GLuint index, GLboolean* data', },
{ 'return_type': 'void',
  'versions': [{'name': 'glGetBooleani_vRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLenum target, GLuint index, GLsizei bufSize, GLsizei* length, '
      'GLboolean* data', },
{ 'return_type': 'void',
  'names': ['glGetBooleanv'],
  'arguments': 'GLenum pname, GLboolean* params', },
{ 'return_type': 'void',
  'versions': [{'name': 'glGetBooleanvRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLenum pname, GLsizei bufSize, GLsizei* length, GLboolean* data', },
{ 'return_type': 'void',
  'versions': [{'name': 'glGetBufferParameteri64vRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLenum target, GLenum pname, GLsizei bufSize, GLsizei* length, '
      'GLint64* params', },
{ 'return_type': 'void',
  'names': ['glGetBufferParameteriv'],
  'arguments': 'GLenum target, GLenum pname, GLint* params', },
{ 'return_type': 'void',
  'versions': [{'name': 'glGetBufferParameterivRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLenum target, GLenum pname, GLsizei bufSize, GLsizei* length, '
      'GLint* params', },
{ 'return_type': 'void',
  'versions': [{'name': 'glGetBufferPointervRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLenum target, GLenum pname, GLsizei bufSize, GLsizei* length, '
      'void** params', },
{ 'return_type': 'GLuint',
  'versions': [{ 'name': 'glGetDebugMessageLog' },
               { 'name': 'glGetDebugMessageLogKHR',
                 'extensions': ['GL_KHR_debug'] }],
  'arguments':
    'GLuint count, GLsizei bufSize, GLenum* sources, GLenum* types, '
    'GLuint* ids, GLenum* severities, GLsizei* lengths, char* messageLog', },
{ 'return_type': 'GLenum',
  'names': ['glGetError'],
  'arguments': 'void',
  'logging_code': """
  GL_SERVICE_LOG("GL_RESULT: " << GLEnums::GetStringError(result));
""", },
{ 'return_type': 'void',
  'names': ['glGetFenceivNV'],
  'arguments': 'GLuint fence, GLenum pname, GLint* params', },
{ 'return_type': 'void',
  'names': ['glGetFloatv'],
  'arguments': 'GLenum pname, GLfloat* params', },
{ 'return_type': 'void',
  'versions': [{'name': 'glGetFloatvRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLenum pname, GLsizei bufSize, GLsizei* length, GLfloat* data', },
{ 'return_type': 'GLint',
  'versions': [{'name': 'glGetFragDataIndex',
                'extensions': ['GL_ARB_blend_func_extended']},
               {'name': 'glGetFragDataIndexEXT',
                'extensions': ['GL_EXT_blend_func_extended']}],
  'arguments': 'GLuint program, const char* name', },
{ 'return_type': 'GLint',
  'versions': [{ 'name': 'glGetFragDataLocation' }],
  'arguments': 'GLuint program, const char* name', },
{ 'return_type': 'void',
  'names': ['glGetFramebufferAttachmentParameterivEXT',
            'glGetFramebufferAttachmentParameteriv'],
  'arguments': 'GLenum target, '
               'GLenum attachment, GLenum pname, GLint* params', },
{ 'return_type': 'void',
  'versions': [{'name': 'glGetFramebufferAttachmentParameterivRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLenum target, GLenum attachment, GLenum pname, GLsizei bufSize, '
      'GLsizei* length, GLint* params', },
{ 'return_type': 'void',
  'names': ['glGetFramebufferParameteriv'],
  'arguments': 'GLenum target, GLenum pname, GLint* params', },
{ 'return_type': 'void',
  'versions': [{'name': 'glGetFramebufferParameterivRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLenum target, GLenum pname, GLsizei bufSize, GLsizei* length, '
      'GLint* params', },
{ 'return_type': 'GLenum',
  'names': ['glGetGraphicsResetStatusARB',
            'glGetGraphicsResetStatusKHR',
            'glGetGraphicsResetStatusEXT',
            'glGetGraphicsResetStatus'],
  'arguments': 'void', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glGetInteger64i_v' }],
  'arguments': 'GLenum target, GLuint index, GLint64* data', },
{ 'return_type': 'void',
  'versions': [{'name': 'glGetInteger64i_vRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLenum target, GLuint index, GLsizei bufSize, GLsizei* length, '
      'GLint64* data', },
{ 'return_type': 'void',
  'names': ['glGetInteger64v'],
  'arguments': 'GLenum pname, GLint64* params', },
{ 'return_type': 'void',
  'versions': [{'name': 'glGetInteger64vRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLenum pname, GLsizei bufSize, GLsizei* length, GLint64* data', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glGetIntegeri_v' }],
  'arguments': 'GLenum target, GLuint index, GLint* data', },
{ 'return_type': 'void',
  'versions': [{'name': 'glGetIntegeri_vRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLenum target, GLuint index, GLsizei bufSize, GLsizei* length, '
      'GLint* data', },
{ 'return_type': 'void',
  'names': ['glGetIntegerv'],
  'arguments': 'GLenum pname, GLint* params', },
{ 'return_type': 'void',
  'versions': [{'name': 'glGetIntegervRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLenum pname, GLsizei bufSize, GLsizei* length, GLint* data', },
{ 'return_type': 'void',
  'versions': [{'name': 'glGetInternalformativ',
                'extensions': ['GL_ARB_internalformat_query']}],
  'arguments': 'GLenum target, GLenum internalformat, GLenum pname, '
               'GLsizei bufSize, GLint* params', },
{ 'return_type': 'void',
  'versions': [{'name': 'glGetInternalformativRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLenum target, GLenum internalformat, GLenum pname, GLsizei bufSize, '
      'GLsizei* length, GLint* params', },
{ 'return_type': 'void',
  'versions': [{'name': 'glGetInternalformatSampleivNV',
                'extensions': ['GL_NV_internalformat_sample_query']}],
  'arguments': 'GLenum target, GLenum internalformat, GLsizei samples, '
               'GLenum pname, GLsizei bufSize, GLint* params', },
{ 'return_type': 'void',
  'versions': [{'name': 'glGetMultisamplefv',
                'extensions': ['GL_ARB_texture_multisample']}],
  'arguments': 'GLenum pname, GLuint index, GLfloat* val', },
{ 'return_type': 'void',
  'versions': [{'name': 'glGetMultisamplefvRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLenum pname, GLuint index, GLsizei bufSize, GLsizei* length, '
      'GLfloat* val', },
{ 'return_type': 'void',
  'versions': [{'name': 'glGetnUniformfvRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLuint program, GLint location, GLsizei bufSize, GLsizei* length, '
      'GLfloat* params', },
{ 'return_type': 'void',
  'versions': [{'name': 'glGetnUniformivRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLuint program, GLint location, GLsizei bufSize, GLsizei* length, '
      'GLint* params', },
{ 'return_type': 'void',
  'versions': [{'name': 'glGetnUniformuivRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLuint program, GLint location, GLsizei bufSize, GLsizei* length, '
      'GLuint* params', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glGetObjectLabel' },
               { 'name': 'glGetObjectLabelKHR',
                 'extensions': ['GL_KHR_debug'] }],
  'arguments':
    'GLenum identifier, GLuint name, GLsizei bufSize, GLsizei* length, '
    'char* label', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glGetObjectPtrLabel' },
               { 'name': 'glGetObjectPtrLabelKHR',
                 'extensions': ['GL_KHR_debug'] }],
  'arguments': 'void* ptr, GLsizei bufSize, GLsizei* length, char* label', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glGetPointerv' },
               { 'name': 'glGetPointervKHR',
                 'extensions': ['GL_KHR_debug'] }],
  'arguments': 'GLenum pname, void** params', },
{ 'return_type': 'void',
  'versions': [{'name': 'glGetPointervRobustANGLERobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLenum pname, GLsizei bufSize, GLsizei* length, void** params', },
{ 'return_type': 'void',
  'known_as': 'glGetProgramBinary',
  'versions': [{ 'name': 'glGetProgramBinaryOES' },
               { 'name': 'glGetProgramBinary',
                 'extensions': ['GL_ARB_get_program_binary'] }],
  'arguments': 'GLuint program, GLsizei bufSize, GLsizei* length, '
               'GLenum* binaryFormat, GLvoid* binary' },
{ 'return_type': 'void',
  'names': ['glGetProgramInfoLog'],
  'arguments':
      'GLuint program, GLsizei bufsize, GLsizei* length, char* infolog', },
{ 'return_type': 'void',
  'versions': [{'name': 'glGetProgramInterfaceiv',
                'extensions': ['GL_ARB_program_interface_query']}],
  'arguments': 'GLuint program, GLenum programInterface, GLenum pname, '
  'GLint* params'},
{ 'return_type': 'void',
  'versions': [{'name': 'glGetProgramInterfaceivRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLuint program, GLenum programInterface, GLenum pname, '
      'GLsizei bufSize, GLsizei* length, GLint* params', },
{ 'return_type': 'void',
  'names': ['glGetProgramiv'],
  'arguments': 'GLuint program, GLenum pname, GLint* params', },
{ 'return_type': 'void',
  'versions': [{'name': 'glGetProgramivRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLuint program, GLenum pname, GLsizei bufSize, GLsizei* length, '
      'GLint* params', },
{ 'return_type': 'void',
  'names': ['glGetProgramPipelineInfoLog'],
  'arguments':
      'GLuint pipeline, GLsizei bufSize, GLsizei* length, GLchar* infoLog', },
{ 'return_type': 'void',
  'names': ['glGetProgramPipelineiv'],
  'arguments':
      'GLuint pipeline, GLenum pname, GLint* params', },
{ 'return_type': 'GLuint',
  'versions': [{'name': 'glGetProgramResourceIndex',
                'extensions': ['GL_ARB_program_interface_query']}],
  'arguments':
      'GLuint program, GLenum programInterface, const GLchar* name', },
{ 'return_type': 'void',
  'versions': [{'name': 'glGetProgramResourceiv',
                'extensions': ['GL_ARB_program_interface_query']}],
  'arguments': 'GLuint program, GLenum programInterface, GLuint index, '
  'GLsizei propCount, const GLenum* props, GLsizei bufSize, '
  'GLsizei* length, GLint* params'},
{ 'return_type': 'GLint',
  'versions': [{'name': 'glGetProgramResourceLocation',
                'extensions': ['GL_ARB_program_interface_query']}],
  'arguments': 'GLuint program, GLenum programInterface, const char* name', },
{ 'return_type': 'void',
  'versions': [{'name': 'glGetProgramResourceName',
                'extensions': ['GL_ARB_program_interface_query']}],
  'arguments': 'GLuint program, GLenum programInterface, GLuint index, '
  'GLsizei bufSize, GLsizei* length, GLchar* name'},
{ 'return_type': 'void',
  'versions': [{ 'name': 'glGetQueryiv' },
               { 'name': 'glGetQueryivARB' },
               { 'name': 'glGetQueryivEXT',
                 'extensions': ['GL_EXT_occlusion_query_boolean'] }],
  'arguments': 'GLenum target, GLenum pname, GLint* params', },
{ 'return_type': 'void',
  'versions': [{'name': 'glGetQueryivRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLenum target, GLenum pname, GLsizei bufSize, GLsizei* length, '
      'GLint* params', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glGetQueryObjecti64v',
                 'extensions': ['GL_ARB_timer_query'] },
               { 'name': 'glGetQueryObjecti64vEXT' }],
  'arguments': 'GLuint id, GLenum pname, GLint64* params', },
{ 'return_type': 'void',
  'versions': [{'name': 'glGetQueryObjecti64vRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLuint id, GLenum pname, GLsizei bufSize, GLsizei* length, '
      'GLint64* params', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glGetQueryObjectiv' },
               { 'name': 'glGetQueryObjectivARB' },
               { 'name': 'glGetQueryObjectivEXT' }],
  'arguments': 'GLuint id, GLenum pname, GLint* params', },
{ 'return_type': 'void',
  'versions': [{'name': 'glGetQueryObjectivRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLuint id, GLenum pname, GLsizei bufSize, GLsizei* length, '
      'GLint* params', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glGetQueryObjectui64v',
                 'extensions': ['GL_ARB_timer_query'] },
               { 'name': 'glGetQueryObjectui64vEXT' }],
  'arguments': 'GLuint id, GLenum pname, GLuint64* params', },
{ 'return_type': 'void',
  'versions': [{'name': 'glGetQueryObjectui64vRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLuint id, GLenum pname, GLsizei bufSize, GLsizei* length, '
      'GLuint64* params', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glGetQueryObjectuiv' },
               { 'name': 'glGetQueryObjectuivARB' },
               { 'name': 'glGetQueryObjectuivEXT',
                 'extensions': ['GL_EXT_occlusion_query_boolean'] }],
  'arguments': 'GLuint id, GLenum pname, GLuint* params', },
{ 'return_type': 'void',
  'versions': [{'name': 'glGetQueryObjectuivRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLuint id, GLenum pname, GLsizei bufSize, GLsizei* length, '
      'GLuint* params', },
{ 'return_type': 'void',
  'names': ['glGetRenderbufferParameterivEXT', 'glGetRenderbufferParameteriv'],
  'arguments': 'GLenum target, GLenum pname, GLint* params', },
{ 'return_type': 'void',
  'versions': [{'name': 'glGetRenderbufferParameterivRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLenum target, GLenum pname, GLsizei bufSize, GLsizei* length, '
      'GLint* params', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glGetSamplerParameterfv',
                 'extensions': ['GL_ARB_sampler_objects'] }],
  'arguments': 'GLuint sampler, GLenum pname, GLfloat* params', },
{ 'return_type': 'void',
  'versions': [{'name': 'glGetSamplerParameterfvRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLuint sampler, GLenum pname, GLsizei bufSize, GLsizei* length, '
      'GLfloat* params', },
{ 'return_type': 'void',
  'versions': [{'name': 'glGetSamplerParameterIivRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLuint sampler, GLenum pname, GLsizei bufSize, GLsizei* length, '
      'GLint* params', },
{ 'return_type': 'void',
  'versions': [{'name': 'glGetSamplerParameterIuivRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLuint sampler, GLenum pname, GLsizei bufSize, GLsizei* length, '
      'GLuint* params', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glGetSamplerParameteriv',
                 'extensions': ['GL_ARB_sampler_objects'] }],
  'arguments': 'GLuint sampler, GLenum pname, GLint* params', },
{ 'return_type': 'void',
  'versions': [{'name': 'glGetSamplerParameterivRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLuint sampler, GLenum pname, GLsizei bufSize, GLsizei* length, '
      'GLint* params', },
{ 'return_type': 'void',
  'names': ['glGetShaderInfoLog'],
  'arguments':
      'GLuint shader, GLsizei bufsize, GLsizei* length, char* infolog', },
{ 'return_type': 'void',
  'names': ['glGetShaderiv'],
  'arguments': 'GLuint shader, GLenum pname, GLint* params', },
{ 'return_type': 'void',
  'versions': [{'name': 'glGetShaderivRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLuint shader, GLenum pname, GLsizei bufSize, GLsizei* length, '
      'GLint* params', },
{ 'return_type': 'void',
  'versions': [{'name': 'glGetShaderPrecisionFormat',
                'extensions': ['GL_ARB_ES2_compatibility']}],
  'arguments': 'GLenum shadertype, GLenum precisiontype, '
               'GLint* range, GLint* precision', },
{ 'return_type': 'void',
  'names': ['glGetShaderSource'],
  'arguments':
      'GLuint shader, GLsizei bufsize, GLsizei* length, char* source', },
{ 'return_type': 'const GLubyte*',
  'names': ['glGetString'],
  'arguments': 'GLenum name', },
{ 'return_type': 'const GLubyte*',
  # This is needed for bootstrapping on the desktop GL core profile.
  # It won't be called unless the expected GL version is used.
  'versions': [{ 'name': 'glGetStringi',
                 'extensions': ['GL_CHROMIUM_glgetstringi_hack'] }],
  'arguments': 'GLenum name, GLuint index', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glGetSynciv',
                 'extensions': ['GL_ARB_sync'] }],
  'arguments':
    'GLsync sync, GLenum pname, GLsizei bufSize, GLsizei* length,'
    'GLint* values', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glGetTexLevelParameterfv' },
               {'name': 'glGetTexLevelParameterfvANGLE',
                'extensions': ['GL_ANGLE_get_tex_level_parameter']}],
  'arguments': 'GLenum target, GLint level, GLenum pname, GLfloat* params', },
{ 'return_type': 'void',
  'versions': [{'name': 'glGetTexLevelParameterfvRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLenum target, GLint level, GLenum pname, GLsizei bufSize, '
      'GLsizei* length, GLfloat* params', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glGetTexLevelParameteriv' },
               {'name': 'glGetTexLevelParameterivANGLE',
                'extensions': ['GL_ANGLE_get_tex_level_parameter']}],
  'arguments': 'GLenum target, GLint level, GLenum pname, GLint* params', },
{ 'return_type': 'void',
  'versions': [{'name': 'glGetTexLevelParameterivRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLenum target, GLint level, GLenum pname, GLsizei bufSize, '
      'GLsizei* length, GLint* params', },
{ 'return_type': 'void',
  'names': ['glGetTexParameterfv'],
  'arguments': 'GLenum target, GLenum pname, GLfloat* params', },
{ 'return_type': 'void',
  'versions': [{'name': 'glGetTexParameterfvRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLenum target, GLenum pname, GLsizei bufSize, GLsizei* length, '
      'GLfloat* params', },
{ 'return_type': 'void',
  'versions': [{'name': 'glGetTexParameterIivRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLenum target, GLenum pname, GLsizei bufSize, GLsizei* length, '
      'GLint* params', },
{ 'return_type': 'void',
  'versions': [{'name': 'glGetTexParameterIuivRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLenum target, GLenum pname, GLsizei bufSize, GLsizei* length, '
      'GLuint* params', },
{ 'return_type': 'void',
  'names': ['glGetTexParameteriv'],
  'arguments': 'GLenum target, GLenum pname, GLint* params', },
{ 'return_type': 'void',
  'versions': [{'name': 'glGetTexParameterivRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLenum target, GLenum pname, GLsizei bufSize, GLsizei* length, '
      'GLint* params', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glGetTransformFeedbackVarying' },
               { 'name': 'glGetTransformFeedbackVaryingEXT',
                 'extension': ['GL_EXT_transform_feedback'] }],
  'arguments': 'GLuint program, GLuint index, GLsizei bufSize, '
               'GLsizei* length, GLsizei* size, GLenum* type, char* name', },
{ 'return_type': 'void',
  'names': ['glGetTranslatedShaderSourceANGLE'],
  'arguments':
      'GLuint shader, GLsizei bufsize, GLsizei* length, char* source', },
{ 'return_type': 'GLuint',
  'versions': [{ 'name': 'glGetUniformBlockIndex' }],
  'arguments': 'GLuint program, const char* uniformBlockName', },
{ 'return_type': 'void',
  'names': ['glGetUniformfv'],
  'arguments': 'GLuint program, GLint location, GLfloat* params', },
{ 'return_type': 'void',
  'versions': [{'name': 'glGetUniformfvRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLuint program, GLint location, GLsizei bufSize, GLsizei* length, '
      'GLfloat* params', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glGetUniformIndices' }],
  'arguments': 'GLuint program, GLsizei uniformCount, '
               'const char* const* uniformNames, GLuint* uniformIndices', },
{ 'return_type': 'void',
  'names': ['glGetUniformiv'],
  'arguments': 'GLuint program, GLint location, GLint* params', },
{ 'return_type': 'void',
  'versions': [{'name': 'glGetUniformivRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLuint program, GLint location, GLsizei bufSize, GLsizei* length, '
      'GLint* params', },
{ 'return_type': 'GLint',
  'names': ['glGetUniformLocation'],
  'arguments': 'GLuint program, const char* name', },
{ 'return_type': 'void',
  'names': ['glGetUniformuiv'],
  'arguments': 'GLuint program, GLint location, GLuint* params', },
{ 'return_type': 'void',
  'versions': [{'name': 'glGetUniformuivRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLuint program, GLint location, GLsizei bufSize, GLsizei* length, '
      'GLuint* params', },
{ 'return_type': 'void',
  'names': ['glGetVertexAttribfv'],
  'arguments': 'GLuint index, GLenum pname, GLfloat* params', },
{ 'return_type': 'void',
  'versions': [{'name': 'glGetVertexAttribfvRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLuint index, GLenum pname, GLsizei bufSize, GLsizei* length, '
      'GLfloat* params', },
{ 'return_type': 'void',
  'versions': [{'name': 'glGetVertexAttribIivRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLuint index, GLenum pname, GLsizei bufSize, GLsizei* length, '
      'GLint* params', },
{ 'return_type': 'void',
  'versions': [{'name': 'glGetVertexAttribIuivRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLuint index, GLenum pname, GLsizei bufSize, GLsizei* length, '
      'GLuint* params', },
{ 'return_type': 'void',
  'names': ['glGetVertexAttribiv'],
  'arguments': 'GLuint index, GLenum pname, GLint* params', },
{ 'return_type': 'void',
  'versions': [{'name': 'glGetVertexAttribivRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLuint index, GLenum pname, GLsizei bufSize, GLsizei* length, '
      'GLint* params', },
{ 'return_type': 'void',
  'names': ['glGetVertexAttribPointerv'],
  'arguments': 'GLuint index, GLenum pname, void** pointer', },
{ 'return_type': 'void',
  'versions': [{'name': 'glGetVertexAttribPointervRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLuint index, GLenum pname, GLsizei bufSize, GLsizei* length, '
      'void** pointer', },
{ 'return_type': 'void',
  'names': ['glHint'],
  'arguments': 'GLenum target, GLenum mode', },
{ 'return_type': 'void',
  'names': ['glImportMemoryFdEXT'],
  'arguments': 'GLuint memory, GLuint64 size, GLenum handleType, GLint fd', },
{ 'return_type': 'void',
  'names': ['glImportMemoryWin32HandleEXT'],
  'arguments': 'GLuint memory, GLuint64 size, GLenum handleType, void* handle',
  },
{ 'return_type': 'void',
  'arguments': 'GLuint memory, GLuint64 size, GLenum handleType, GLuint handle',
  'versions': [{'name': 'glImportMemoryZirconHandleANGLE',
                'extensions': ['GL_ANGLE_memory_object_fuchsia']}]},
{ 'return_type': 'void',
  'names': ['glImportSemaphoreFdEXT'],
  'arguments': 'GLuint semaphore, GLenum handleType, GLint fd', },
{ 'return_type': 'void',
  'names': ['glImportSemaphoreWin32HandleEXT'],
  'arguments': 'GLuint semaphore, GLenum handleType, void* handle', },
{ 'return_type': 'void',
  'arguments': 'GLuint semaphore, GLenum handleType, GLuint handle',
  'versions': [{'name': 'glImportSemaphoreZirconHandleANGLE',
                'extensions': ['GL_ANGLE_semaphore_fuchsia']}]},
{ 'return_type': 'void',
  'names': ['glInsertEventMarkerEXT'],
  'arguments': 'GLsizei length, const char* marker', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glInvalidateFramebuffer' }],
  'arguments': 'GLenum target, GLsizei numAttachments, '
      'const GLenum* attachments' },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glInvalidateSubFramebuffer' }],
  'arguments':
      'GLenum target, GLsizei numAttachments, const GLenum* attachments, '
      'GLint x, GLint y, GLint width, GLint height', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glInvalidateTextureANGLE',
                 'extensions': ['GL_ANGLE_texture_external_update'] } ],
  'arguments': 'GLenum target', },
{ 'return_type': 'GLboolean',
  'names': ['glIsBuffer'],
  'arguments': 'GLuint buffer', },
{ 'return_type': 'GLboolean',
  'names': ['glIsEnabled'],
  'arguments': 'GLenum cap', },
{ 'return_type': 'GLboolean',
  'known_as': 'glIsEnablediOES',
  'versions': [
    { 'name': 'glIsEnabledi' },
    { 'name': 'glIsEnablediOES', 'extensions': ['GL_OES_draw_buffers_indexed'] }
  ],
  'arguments': 'GLenum target, GLuint index', },
{ 'return_type': 'GLboolean',
  'known_as': 'glIsFenceAPPLE',
  'versions': [{ 'name': 'glIsFenceAPPLE',
                 'extensions': ['GL_APPLE_fence'] }],
  'arguments': 'GLuint fence', },
{ 'return_type': 'GLboolean',
  'names': ['glIsFenceNV'],
  'arguments': 'GLuint fence', },
{ 'return_type': 'GLboolean',
  'names': ['glIsFramebufferEXT', 'glIsFramebuffer'],
  'arguments': 'GLuint framebuffer', },
{ 'return_type': 'GLboolean',
  'names': ['glIsPathNV'],
  'versions': [{ 'name': 'glIsPathNV',
                 'extensions': ['GL_NV_path_rendering'] },
               { 'name': 'glIsPathCHROMIUM',
                 'extensions': ['GL_CHROMIUM_path_rendering'] }],
  'arguments': 'GLuint path' },
{ 'return_type': 'GLboolean',
  'names': ['glIsProgram'],
  'arguments': 'GLuint program', },
{ 'return_type': 'GLboolean',
  'names': ['glIsProgramPipeline'],
  'arguments': 'GLuint pipeline', },
{ 'return_type': 'GLboolean',
  'versions': [{ 'name': 'glIsQuery' },
               { 'name': 'glIsQueryARB' },
               { 'name': 'glIsQueryEXT',
                 'extensions': ['GL_EXT_occlusion_query_boolean'] }],
  'arguments': 'GLuint query', },
{ 'return_type': 'GLboolean',
  'names': ['glIsRenderbufferEXT', 'glIsRenderbuffer'],
  'arguments': 'GLuint renderbuffer', },
{ 'return_type': 'GLboolean',
  'versions': [{ 'name': 'glIsSampler',
                 'extensions': ['GL_ARB_sampler_objects'] }],
  'arguments': 'GLuint sampler', },
{ 'return_type': 'GLboolean',
  'names': ['glIsShader'],
  'arguments': 'GLuint shader', },
{ 'return_type': 'GLboolean',
  'versions': [{ 'name': 'glIsSync',
                 'extensions': ['GL_ARB_sync'] }],
  'arguments': 'GLsync sync', },
{ 'return_type': 'GLboolean',
  'versions': [{ 'name': 'glIsSyncAPPLE',
                 'extensions': ['GL_APPLE_sync'] }],
  'arguments': 'GLsync sync', },
{ 'return_type': 'GLboolean',
  'names': ['glIsTexture'],
  'arguments': 'GLuint texture', },
{ 'return_type': 'GLboolean',
  'versions': [{ 'name': 'glIsTransformFeedback',
                 'extensions': ['GL_ARB_transform_feedback2'] }],
  'arguments': 'GLuint id', },
{ 'return_type': 'GLboolean',
  'known_as': 'glIsVertexArrayOES',
  'versions': [{ 'name': 'glIsVertexArray',
                 'extensions': ['GL_ARB_vertex_array_object'], },
               { 'name': 'glIsVertexArrayOES' },
               { 'name': 'glIsVertexArrayAPPLE',
                 'extensions': ['GL_APPLE_vertex_array_object'] }],
  'arguments': 'GLuint array' },
{ 'return_type': 'void',
  'names': ['glLineWidth'],
  'arguments': 'GLfloat width', },
{ 'return_type': 'void',
  'names': ['glLinkProgram'],
  'arguments': 'GLuint program', },
{ 'return_type': 'void*',
  'known_as': 'glMapBuffer',
  'names': ['glMapBufferOES', 'glMapBuffer'],
  'arguments': 'GLenum target, GLenum access', },
{ 'return_type': 'void*',
  'known_as': 'glMapBufferRange',
  'versions': [{ 'name': 'glMapBufferRange',
                 'extensions': ['GL_ARB_map_buffer_range'] },
               { 'name': 'glMapBufferRangeEXT',
                 'extensions': ['GL_EXT_map_buffer_range'] }],
  'arguments':
      'GLenum target, GLintptr offset, GLsizeiptr length, GLbitfield access', },
{ 'return_type': 'void',
  'known_as': 'glMatrixLoadfEXT',
  'versions': [{ 'name': 'glMatrixLoadfEXT',
                 'extensions': ['GL_EXT_direct_state_access',
                                'GL_NV_path_rendering'] },
               { 'name': 'glMatrixLoadfCHROMIUM',
                 'extensions': ['GL_CHROMIUM_path_rendering'] }],
  'arguments': 'GLenum matrixMode, const GLfloat* m' },
{ 'return_type': 'void',
  'known_as': 'glMatrixLoadIdentityEXT',
  'versions': [{ 'name': 'glMatrixLoadIdentityEXT',
                 'extensions': ['GL_EXT_direct_state_access',
                                'GL_NV_path_rendering'] },
               { 'name': 'glMatrixLoadIdentityCHROMIUM',
                 'extensions': ['GL_CHROMIUM_path_rendering'] }],
  'arguments': 'GLenum matrixMode' },
{'return_type': 'void',
 'known_as': 'glMaxShaderCompilerThreadsKHR',
  'versions': [{ 'name': 'glMaxShaderCompilerThreadsKHR',
                 'extensions': ['GL_KHR_parallel_shader_compile'] }],
  'arguments': 'GLuint count', },
{ 'return_type': 'void',
  'names': ['glMemoryBarrierByRegion'],
  'arguments': 'GLbitfield barriers', },
{'return_type': 'void',
  'known_as': 'glMemoryBarrierEXT',
  'versions': [{ 'name': 'glMemoryBarrier',
                 'extensions': ['GL_ARB_shader_image_load_store'] },
               { 'name': 'glMemoryBarrierEXT',
                 'extensions': ['GL_EXT_shader_image_load_store'] }],
  'arguments': 'GLbitfield barriers', },
{ 'return_type': 'void',
  'names': ['glMemoryObjectParameterivEXT'],
  'arguments': 'GLuint memoryObject, GLenum pname, const GLint* param'},
{ 'return_type': 'void',
  'names': ['glMinSampleShading'],
  'arguments': 'GLfloat value', },
{ 'return_type': 'void',
  'versions' : [{'name': 'glMultiDrawArraysANGLE',
                 'extensions': ['GL_ANGLE_multi_draw'] }],
  'arguments': 'GLenum mode, const GLint* firsts, '
               'const GLsizei* counts, GLsizei drawcount', },
{ 'return_type': 'void',
  'versions' : [{'name': 'glMultiDrawArraysInstancedANGLE',
                 'extensions': ['GL_ANGLE_multi_draw'] }],
  'arguments': 'GLenum mode, const GLint* firsts, '
               'const GLsizei* counts, const GLsizei* instanceCounts, '
               'GLsizei drawcount', },
{ 'return_type': 'void',
  'versions' : [{'name': 'glMultiDrawArraysInstancedBaseInstanceANGLE',
                 'extensions': ['GL_ANGLE_base_vertex_base_instance'] }],
  'arguments': 'GLenum mode, const GLint* firsts, '
               'const GLsizei* counts, const GLsizei* instanceCounts, '
               'const GLuint* baseInstances, GLsizei drawcount', },
{ 'return_type': 'void',
  'versions' : [{'name': 'glMultiDrawElementsANGLE',
                 'extensions': ['GL_ANGLE_multi_draw'] }],
  'arguments': 'GLenum mode, const GLsizei* counts, '
               'GLenum type, const GLvoid* const* indices, '
               'GLsizei drawcount', },
{ 'return_type': 'void',
  'versions' : [{'name': 'glMultiDrawElementsInstancedANGLE',
                 'extensions': ['GL_ANGLE_multi_draw'] }],
  'arguments': 'GLenum mode, const GLsizei* counts, '
               'GLenum type, const GLvoid* const* indices, '
               'const GLsizei* instanceCounts, GLsizei drawcount', },
{ 'return_type': 'void',
  'versions' : [{'name': 'glMultiDrawElementsInstancedBaseVertexBaseInstanceANGLE',
                 'extensions': ['GL_ANGLE_base_vertex_base_instance'] }],
  'arguments': 'GLenum mode, '
               'const GLsizei* counts, GLenum type, '
               'const GLvoid* const* indices, const GLsizei* instanceCounts, '
               'const GLint* baseVertices, const GLuint* baseInstances, '
               'GLsizei drawcount', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glObjectLabel' },
               { 'name': 'glObjectLabelKHR',
                 'extensions': ['GL_KHR_debug'] }],
  'arguments':
    'GLenum identifier, GLuint name, GLsizei length, const char* label', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glObjectPtrLabel' },
               { 'name': 'glObjectPtrLabelKHR',
                 'extensions': ['GL_KHR_debug'] }],
  'arguments': 'void* ptr, GLsizei length, const char* label', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glPatchParameteri',
                 'extensions': ['GL_ARB_tessellation_shader'] },
               { 'name': 'glPatchParameteriOES',
                 'extensions': ['GL_OES_tessellation_shader'] }],
  'arguments': 'GLenum pname, GLint value', },
{ 'return_type': 'void',
  'names': ['glPathCommandsNV'],
  'versions': [{ 'name': 'glPathCommandsNV',
                 'extensions': ['GL_NV_path_rendering'] },
               { 'name': 'glPathCommandsCHROMIUM',
                 'extensions': ['GL_CHROMIUM_path_rendering'] }],
  'arguments': 'GLuint path, GLsizei numCommands, const GLubyte* commands, '
  'GLsizei numCoords, GLenum coordType, const GLvoid* coords' },
{ 'return_type': 'void',
  'names': ['glPathParameterfNV'],
  'versions': [{ 'name': 'glPathParameterfNV',
                 'extensions': ['GL_NV_path_rendering'] },
               { 'name': 'glPathParameterfCHROMIUM',
                 'extensions': ['GL_CHROMIUM_path_rendering'] }],
  'arguments': 'GLuint path, GLenum pname, GLfloat value' },
{ 'return_type': 'void',
  'names': ['glPathParameteriNV'],
  'versions': [{ 'name': 'glPathParameteriNV',
                 'extensions': ['GL_NV_path_rendering'] },
               { 'name': 'glPathParameteriCHROMIUM',
                 'extensions': ['GL_CHROMIUM_path_rendering'] }],
  'arguments': 'GLuint path, GLenum pname, GLint value' },
{ 'return_type': 'void',
  'names': ['glPathStencilFuncNV'],
  'versions': [{ 'name': 'glPathStencilFuncNV',
                 'extensions': ['GL_NV_path_rendering'] },
               { 'name': 'glPathStencilFuncCHROMIUM',
                 'extensions': ['GL_CHROMIUM_path_rendering'] }],
  'arguments': 'GLenum func, GLint ref, GLuint mask' },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glPauseTransformFeedback',
                 'extensions': ['GL_ARB_transform_feedback2'] }],
  'arguments': 'void', },
{ 'return_type': 'void',
  'names': ['glPixelStorei'],
  'arguments': 'GLenum pname, GLint param', },
{ 'return_type': 'void',
  'names': ['glPointParameteri'],
  'arguments': 'GLenum pname, GLint param', },
{ 'return_type': 'void',
  'names': ['glPolygonMode'],
  'arguments': 'GLenum face, GLenum mode', },
{ 'return_type': 'void',
  'names': ['glPolygonOffset'],
  'arguments': 'GLfloat factor, GLfloat units', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glPopDebugGroup' },
               { 'name': 'glPopDebugGroupKHR',
                 'extensions': ['GL_KHR_debug'] }],
  'arguments': '', },
{ 'return_type': 'void',
  'names': ['glPopGroupMarkerEXT'],
  'arguments': 'void', },
{ 'return_type': 'void',
  'names': ['glPrimitiveRestartIndex'],
  'arguments': 'GLuint index', },
{ 'return_type': 'void',
  'known_as': 'glProgramBinary',
  'versions': [{ 'name': 'glProgramBinaryOES' },
               { 'name': 'glProgramBinary',
                 'extensions': ['GL_ARB_get_program_binary'] }],
  'arguments': 'GLuint program, GLenum binaryFormat, '
               'const GLvoid* binary, GLsizei length' },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glProgramParameteri',
                 'extensions': ['GL_ARB_get_program_binary'] }],
  'arguments': 'GLuint program, GLenum pname, GLint value' },
{ 'return_type': 'void',
  'names': ['glProgramPathFragmentInputGenNV'],
  'versions': [{ 'name': 'glProgramPathFragmentInputGenNV',
                 'extensions': ['GL_NV_path_rendering'] },
               { 'name': 'glProgramPathFragmentInputGenCHROMIUM',
                 'extensions': ['GL_CHROMIUM_path_rendering'] }],
  'arguments': 'GLuint program, GLint location, GLenum genMode, '
  'GLint components, const GLfloat* coeffs',
  'is_optional': True, },
{ 'return_type': 'void',
  'names': ['glProgramUniform1f'],
  'arguments': 'GLuint program, GLint location, GLfloat v0' },
{ 'return_type': 'void',
  'names': ['glProgramUniform1fv'],
  'arguments': 'GLuint program, GLint location, GLsizei count, '
               'const GLfloat* value' },
{ 'return_type': 'void',
  'names': ['glProgramUniform1i'],
  'arguments': 'GLuint program, GLint location, GLint v0' },
{ 'return_type': 'void',
  'names': ['glProgramUniform1iv'],
  'arguments': 'GLuint program, GLint location, GLsizei count, '
               'const GLint* value' },
{ 'return_type': 'void',
  'names': ['glProgramUniform1ui'],
  'arguments': 'GLuint program, GLint location, GLuint v0' },
{ 'return_type': 'void',
  'names': ['glProgramUniform1uiv'],
  'arguments': 'GLuint program, GLint location, GLsizei count, '
               'const GLuint* value' },
{ 'return_type': 'void',
  'names': ['glProgramUniform2f'],
  'arguments': 'GLuint program, GLint location, GLfloat v0, GLfloat v1' },
{ 'return_type': 'void',
  'names': ['glProgramUniform2fv'],
  'arguments': 'GLuint program, GLint location, GLsizei count, '
               'const GLfloat* value' },
{ 'return_type': 'void',
  'names': ['glProgramUniform2i'],
  'arguments': 'GLuint program, GLint location, GLint v0, GLint v1' },
{ 'return_type': 'void',
  'names': ['glProgramUniform2iv'],
  'arguments': 'GLuint program, GLint location, GLsizei count, '
               'const GLint* value' },
{ 'return_type': 'void',
  'names': ['glProgramUniform2ui'],
  'arguments': 'GLuint program, GLint location, GLuint v0, GLuint v1' },
{ 'return_type': 'void',
  'names': ['glProgramUniform2uiv'],
  'arguments': 'GLuint program, GLint location, GLsizei count, '
               'const GLuint* value' },
{ 'return_type': 'void',
  'names': ['glProgramUniform3f'],
  'arguments': 'GLuint program, GLint location, GLfloat v0, GLfloat v1, '
               'GLfloat v2' },
{ 'return_type': 'void',
  'names': ['glProgramUniform3fv'],
  'arguments': 'GLuint program, GLint location, GLsizei count, '
               'const GLfloat* value' },
{ 'return_type': 'void',
  'names': ['glProgramUniform3i'],
  'arguments': 'GLuint program, GLint location, GLint v0, GLint v1, GLint v2' },
{ 'return_type': 'void',
  'names': ['glProgramUniform3iv'],
  'arguments': 'GLuint program, GLint location, GLsizei count, '
               'const GLint* value' },
{ 'return_type': 'void',
  'names': ['glProgramUniform3ui'],
  'arguments': 'GLuint program, GLint location, GLuint v0, GLuint v1, '
               'GLuint v2' },
{ 'return_type': 'void',
  'names': ['glProgramUniform3uiv'],
  'arguments': 'GLuint program, GLint location, GLsizei count, '
               'const GLuint* value' },
{ 'return_type': 'void',
  'names': ['glProgramUniform4f'],
  'arguments': 'GLuint program, GLint location, GLfloat v0, GLfloat v1, '
               'GLfloat v2, GLfloat v3' },
{ 'return_type': 'void',
  'names': ['glProgramUniform4fv'],
  'arguments': 'GLuint program, GLint location, GLsizei count, '
               'const GLfloat* value' },
{ 'return_type': 'void',
  'names': ['glProgramUniform4i'],
  'arguments': 'GLuint program, GLint location, GLint v0, GLint v1, GLint v2, '
               'GLint v3' },
{ 'return_type': 'void',
  'names': ['glProgramUniform4iv'],
  'arguments': 'GLuint program, GLint location, GLsizei count, '
               'const GLint* value' },
{ 'return_type': 'void',
  'names': ['glProgramUniform4ui'],
  'arguments': 'GLuint program, GLint location, GLuint v0, GLuint v1, '
               'GLuint v2, GLuint v3' },
{ 'return_type': 'void',
  'names': ['glProgramUniform4uiv'],
  'arguments': 'GLuint program, GLint location, GLsizei count, '
               'const GLuint* value' },
{ 'return_type': 'void',
  'names': ['glProgramUniformMatrix2fv'],
  'arguments': 'GLuint program, GLint location, GLsizei count, '
               'GLboolean transpose, const GLfloat* value' },
{ 'return_type': 'void',
  'names': ['glProgramUniformMatrix2x3fv'],
  'arguments': 'GLuint program, GLint location, GLsizei count, '
               'GLboolean transpose, const GLfloat* value' },
{ 'return_type': 'void',
  'names': ['glProgramUniformMatrix2x4fv'],
  'arguments': 'GLuint program, GLint location, GLsizei count, '
               'GLboolean transpose, const GLfloat* value' },
{ 'return_type': 'void',
  'names': ['glProgramUniformMatrix3fv'],
  'arguments': 'GLuint program, GLint location, GLsizei count, '
               'GLboolean transpose, const GLfloat* value' },
{ 'return_type': 'void',
  'names': ['glProgramUniformMatrix3x2fv'],
  'arguments': 'GLuint program, GLint location, GLsizei count, '
               'GLboolean transpose, const GLfloat* value' },
{ 'return_type': 'void',
  'names': ['glProgramUniformMatrix3x4fv'],
  'arguments': 'GLuint program, GLint location, GLsizei count, '
               'GLboolean transpose, const GLfloat* value' },
{ 'return_type': 'void',
  'names': ['glProgramUniformMatrix4fv'],
  'arguments': 'GLuint program, GLint location, GLsizei count, '
               'GLboolean transpose, const GLfloat* value' },
{ 'return_type': 'void',
  'names': ['glProgramUniformMatrix4x2fv'],
  'arguments': 'GLuint program, GLint location, GLsizei count, '
               'GLboolean transpose, const GLfloat* value' },
{ 'return_type': 'void',
  'names': ['glProgramUniformMatrix4x3fv'],
  'arguments': 'GLuint program, GLint location, GLsizei count, '
               'GLboolean transpose, const GLfloat* value' },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glPushDebugGroup' },
               { 'name': 'glPushDebugGroupKHR',
                 'extensions': ['GL_KHR_debug'] }],
  'arguments':
    'GLenum source, GLuint id, GLsizei length, const char* message', },
{ 'return_type': 'void',
  'names': ['glPushGroupMarkerEXT'],
  'arguments': 'GLsizei length, const char* marker', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glQueryCounter',
                 'extensions': ['GL_ARB_timer_query'] },
               { 'name': 'glQueryCounterEXT' }],
  'arguments': 'GLuint id, GLenum target', },
{ 'return_type': 'void',
  'names': ['glReadBuffer'],
  'arguments': 'GLenum src', },
{ 'return_type': 'void',
  'versions': [{'name': 'glReadnPixelsRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLint x, GLint y, GLsizei width, GLsizei height, GLenum format, '
      'GLenum type, GLsizei bufSize, GLsizei* length, GLsizei* columns, '
      'GLsizei* rows, void* data', },
{ 'return_type': 'void',
  'names': ['glReadPixels'],
  'arguments':
    'GLint x, GLint y, GLsizei width, GLsizei height, GLenum format, '
    'GLenum type, void* pixels', },
{ 'return_type': 'void',
  'versions': [{'name': 'glReadPixelsRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLint x, GLint y, GLsizei width, GLsizei height, GLenum format, '
      'GLenum type, GLsizei bufSize, GLsizei* length, GLsizei* columns, '
      'GLsizei* rows, void* pixels', },
{ 'return_type': 'void',
  'names': ['glReleaseShaderCompiler'],
  'arguments': 'void', },
{ 'return_type': 'void',
  'names': ['glRenderbufferStorageEXT', 'glRenderbufferStorage'],
  'arguments':
      'GLenum target, GLenum internalformat, GLsizei width, GLsizei height', },
{ 'return_type': 'void',
 'versions' : [{'name': 'glRenderbufferStorageMultisample',
                'extensions': ['GL_ARB_framebuffer_object']},
               {'name': 'glRenderbufferStorageMultisampleANGLE'},
               {'name': 'glRenderbufferStorageMultisampleEXT',
                'extensions': ['GL_EXT_framebuffer_multisample'],
                'explicit_only': True}],
  'arguments': 'GLenum target, GLsizei samples, GLenum internalformat, '
               'GLsizei width, GLsizei height', },
{ 'return_type': 'void',
 'versions' : [{'name': 'glRenderbufferStorageMultisampleAdvancedAMD',
                'extensions': ['GL_AMD_framebuffer_multisample_advanced'] ,
                'explicit_only': True}],
  'arguments': 'GLenum target, GLsizei samples, GLsizei storageSamples, '
               'GLenum internalformat,GLsizei width, GLsizei height', },
{ 'return_type': 'void',
 'versions' : [{'name': 'glRenderbufferStorageMultisampleEXT',
                'extensions': ['GL_EXT_multisampled_render_to_texture'],
                'explicit_only': True},
               {'name': 'glRenderbufferStorageMultisampleIMG'}],
  'arguments': 'GLenum target, GLsizei samples, GLenum internalformat, '
               'GLsizei width, GLsizei height', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glRequestExtensionANGLE',
                 'extensions': ['GL_ANGLE_request_extension'] }],
  'arguments': 'const char* name', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glResumeTransformFeedback',
                 'extensions': ['GL_ARB_transform_feedback2'] }],
  'arguments': 'void', },
{ 'return_type': 'void',
  'names': ['glSampleCoverage'],
  'arguments': 'GLclampf value, GLboolean invert', },
{ 'return_type': 'void',
  'names': ['glSampleMaski'],
  'arguments': 'GLuint maskNumber, GLbitfield mask', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glSamplerParameterf',
                 'extensions': ['GL_ARB_sampler_objects'] }],
  'arguments': 'GLuint sampler, GLenum pname, GLfloat param', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glSamplerParameterfv',
                 'extensions': ['GL_ARB_sampler_objects'] }],
  'arguments': 'GLuint sampler, GLenum pname, const GLfloat* params', },
{ 'return_type': 'void',
  'versions': [{'name': 'glSamplerParameterfvRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLuint sampler, GLenum pname, GLsizei bufSize, const GLfloat* param', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glSamplerParameteri',
                 'extensions': ['GL_ARB_sampler_objects'] }],
  'arguments': 'GLuint sampler, GLenum pname, GLint param', },
{ 'return_type': 'void',
  'versions': [{'name': 'glSamplerParameterIivRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLuint sampler, GLenum pname, GLsizei bufSize, const GLint* param', },
{ 'return_type': 'void',
  'versions': [{'name': 'glSamplerParameterIuivRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLuint sampler, GLenum pname, GLsizei bufSize, const GLuint* param', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glSamplerParameteriv',
                 'extensions': ['GL_ARB_sampler_objects'] }],
  'arguments': 'GLuint sampler, GLenum pname, const GLint* params', },
{ 'return_type': 'void',
  'versions': [{'name': 'glSamplerParameterivRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLuint sampler, GLenum pname, GLsizei bufSize, const GLint* param', },
{ 'return_type': 'void',
  'names': ['glScissor'],
  'arguments': 'GLint x, GLint y, GLsizei width, GLsizei height', },
{ 'return_type': 'void',
  'known_as': 'glSetFenceAPPLE',
  'versions': [{ 'name': 'glSetFenceAPPLE',
                 'extensions': ['GL_APPLE_fence'] }],
  'arguments': 'GLuint fence', },
{ 'return_type': 'void',
  'names': ['glSetFenceNV'],
  'arguments': 'GLuint fence, GLenum condition', },
{ 'return_type': 'void',
  'names': ['glShaderBinary'],
  'arguments': 'GLsizei n, const GLuint* shaders, GLenum binaryformat, '
               'const void* binary, GLsizei length', },
{ 'return_type': 'void',
  'names': ['glShaderSource'],
  'arguments': 'GLuint shader, GLsizei count, const char* const* str, '
               'const GLint* length',
  'logging_code': """
  GL_SERVICE_LOG_CODE_BLOCK({
    for (GLsizei ii = 0; ii < count; ++ii) {
      if (str[ii]) {
        if (length && length[ii] >= 0) {
          std::string source(str[ii], length[ii]);
          GL_SERVICE_LOG("  " << ii << ": ---\\n" << source << "\\n---");
        } else {
          GL_SERVICE_LOG("  " << ii << ": ---\\n" << str[ii] << "\\n---");
        }
      } else {
        GL_SERVICE_LOG("  " << ii << ": NULL");
      }
    }
  });
""", },
{ 'return_type': 'void',
  'names': ['glSignalSemaphoreEXT'],
 'arguments': 'GLuint semaphore, GLuint numBufferBarriers, '
 'const GLuint* buffers, GLuint numTextureBarriers, '
 'const GLuint* textures, const GLenum* dstLayouts', },
{ 'return_type': 'void',
  'names': ['glStartTilingQCOM'],
  'versions': [{ 'name': 'glStartTilingQCOM',
                 'extension': ['GL_QCOM_tiled_rendering'] }],
  'arguments':
      'GLuint x, GLuint y, GLuint width, GLuint height, '
      'GLbitfield preserveMask', },
{ 'return_type': 'void',
  'names': ['glStencilFillPathInstancedNV'],
  'versions': [{ 'name': 'glStencilFillPathInstancedNV',
                 'extensions': ['GL_NV_path_rendering'] },
               { 'name': 'glStencilFillPathInstancedCHROMIUM',
                 'extensions': ['GL_CHROMIUM_path_rendering'] }],
  'arguments': 'GLsizei numPaths, GLenum pathNameType, const void* paths, '
  'GLuint pathBase, GLenum fillMode, GLuint mask, GLenum transformType, '
  'const GLfloat* transformValues' },
{ 'return_type': 'void',
  'names': ['glStencilFillPathNV'],
  'versions': [{ 'name': 'glStencilFillPathNV',
                 'extensions': ['GL_NV_path_rendering'] },
               { 'name': 'glStencilFillPathCHROMIUM',
                 'extensions': ['GL_CHROMIUM_path_rendering'] }],
  'arguments': 'GLuint path, GLenum fillMode, GLuint mask' },
{ 'return_type': 'void',
  'names': ['glStencilFunc'],
  'arguments': 'GLenum func, GLint ref, GLuint mask', },
{ 'return_type': 'void',
  'names': ['glStencilFuncSeparate'],
  'arguments': 'GLenum face, GLenum func, GLint ref, GLuint mask', },
{ 'return_type': 'void',
  'names': ['glStencilMask'],
  'arguments': 'GLuint mask', },
{ 'return_type': 'void',
  'names': ['glStencilMaskSeparate'],
  'arguments': 'GLenum face, GLuint mask', },
{ 'return_type': 'void',
  'names': ['glStencilOp'],
  'arguments': 'GLenum fail, GLenum zfail, GLenum zpass', },
{ 'return_type': 'void',
  'names': ['glStencilOpSeparate'],
  'arguments': 'GLenum face, GLenum fail, GLenum zfail, GLenum zpass', },
{ 'return_type': 'void',
  'names': ['glStencilStrokePathInstancedNV'],
  'versions': [{ 'name': 'glStencilStrokePathInstancedNV',
                 'extensions': ['GL_NV_path_rendering'] },
               { 'name': 'glStencilStrokePathInstancedCHROMIUM',
                 'extensions': ['GL_CHROMIUM_path_rendering'] }],
  'arguments': 'GLsizei numPaths, GLenum pathNameType, const void* paths, '
  'GLuint pathBase, GLint ref, GLuint mask, GLenum transformType, '
  'const GLfloat* transformValues' },
{ 'return_type': 'void',
  'names': ['glStencilStrokePathNV'],
  'versions': [{ 'name': 'glStencilStrokePathNV',
                 'extensions': ['GL_NV_path_rendering'] },
               { 'name': 'glStencilStrokePathCHROMIUM',
                 'extensions': ['GL_CHROMIUM_path_rendering'] }],
  'arguments': 'GLuint path, GLint reference, GLuint mask' },
{ 'return_type': 'void',
  'names': ['glStencilThenCoverFillPathInstancedNV'],
  'versions': [{ 'name': 'glStencilThenCoverFillPathInstancedNV',
                 'extensions': ['GL_NV_path_rendering'] },
               { 'name': 'glStencilThenCoverFillPathInstancedCHROMIUM',
                 'extensions': ['GL_CHROMIUM_path_rendering'] }],
  'arguments': 'GLsizei numPaths, GLenum pathNameType, const void* paths, '
  'GLuint pathBase, GLenum fillMode, GLuint mask, GLenum coverMode, '
  'GLenum transformType, const GLfloat* transformValues',
  'is_optional': True, },
{ 'return_type': 'void',
  'names': ['glStencilThenCoverFillPathNV'],
  'versions': [{ 'name': 'glStencilThenCoverFillPathNV',
                 'extensions': ['GL_NV_path_rendering'] },
               { 'name': 'glStencilThenCoverFillPathCHROMIUM',
                 'extensions': ['GL_CHROMIUM_path_rendering'] }],
  'arguments': 'GLuint path, GLenum fillMode, GLuint mask, GLenum coverMode',
  'is_optional': True, },
{ 'return_type': 'void',
  'names': ['glStencilThenCoverStrokePathInstancedNV'],
  'versions': [{ 'name': 'glStencilThenCoverStrokePathInstancedNV',
                 'extensions': ['GL_NV_path_rendering'] },
               { 'name': 'glStencilThenCoverStrokePathInstancedCHROMIUM',
                 'extensions': ['GL_CHROMIUM_path_rendering'] }],
  'arguments': 'GLsizei numPaths, GLenum pathNameType, const void* paths, '
  'GLuint pathBase, GLint ref, GLuint mask, GLenum coverMode, '
  'GLenum transformType, const GLfloat* transformValues',
  'is_optional': True, },
{ 'return_type': 'void',
  'names': ['glStencilThenCoverStrokePathNV'],
  'versions': [{ 'name': 'glStencilThenCoverStrokePathNV',
                 'extensions': ['GL_NV_path_rendering'] },
               { 'name': 'glStencilThenCoverStrokePathCHROMIUM',
                 'extensions': ['GL_CHROMIUM_path_rendering'] }],
  'arguments': 'GLuint path, GLint reference, GLuint mask, GLenum coverMode',
  'is_optional': True, },
{ 'return_type': 'GLboolean',
  'known_as': 'glTestFenceAPPLE',
  'versions': [{ 'name': 'glTestFenceAPPLE',
                 'extensions': ['GL_APPLE_fence'] }],
  'arguments': 'GLuint fence', },
{ 'return_type': 'GLboolean',
  'names': ['glTestFenceNV'],
  'arguments': 'GLuint fence', },
{ 'return_type': 'void',
  'names': ['glTexBuffer', 'glTexBufferOES', 'glTexBufferEXT'],
  'arguments': 'GLenum target, GLenum internalformat, GLuint buffer', } ,
{ 'return_type': 'void',
  'names': ['glTexBufferRange', 'glTexBufferRangeOES', 'glTexBufferRangeEXT'],
  'arguments':
      'GLenum target, GLenum internalformat, GLuint buffer, '
      'GLintptr offset, GLsizeiptr size', },
{ 'return_type': 'void',
  'names': ['glTexImage2D'],
  'arguments':
      'GLenum target, GLint level, GLint internalformat, GLsizei width, '
      'GLsizei height, GLint border, GLenum format, GLenum type, '
      'const void* pixels', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glTexImage2DExternalANGLE',
                 'extensions': ['GL_ANGLE_texture_external_update'] } ],
  'arguments': 'GLenum target, GLint level, GLint internalformat, '
               'GLsizei width, GLsizei height, GLint border, GLenum format, '
               'GLenum type', },
{ 'return_type': 'void',
  'versions': [{'name': 'glTexImage2DRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLenum target, GLint level, GLint internalformat, GLsizei width, '
      'GLsizei height, GLint border, GLenum format, GLenum type, '
      'GLsizei bufSize, const void* pixels', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glTexImage3D' }],
  'arguments':
      'GLenum target, GLint level, GLint internalformat, GLsizei width, '
      'GLsizei height, GLsizei depth, GLint border, GLenum format, '
      'GLenum type, const void* pixels', },
{ 'return_type': 'void',
  'versions': [{'name': 'glTexImage3DRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLenum target, GLint level, GLint internalformat, GLsizei width, '
      'GLsizei height, GLsizei depth, GLint border, GLenum format, '
      'GLenum type, GLsizei bufSize, const void* pixels', },
{ 'return_type': 'void',
  'names': ['glTexParameterf'],
  'arguments': 'GLenum target, GLenum pname, GLfloat param', },
{ 'return_type': 'void',
  'names': ['glTexParameterfv'],
  'arguments': 'GLenum target, GLenum pname, const GLfloat* params', },
{ 'return_type': 'void',
  'versions': [{'name': 'glTexParameterfvRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLenum target, GLenum pname, GLsizei bufSize, const GLfloat* params', },
{ 'return_type': 'void',
  'names': ['glTexParameteri'],
  'arguments': 'GLenum target, GLenum pname, GLint param', },
{ 'return_type': 'void',
  'versions': [{'name': 'glTexParameterIivRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLenum target, GLenum pname, GLsizei bufSize, const GLint* params', },
{ 'return_type': 'void',
  'versions': [{'name': 'glTexParameterIuivRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLenum target, GLenum pname, GLsizei bufSize, const GLuint* params', },
{ 'return_type': 'void',
  'names': ['glTexParameteriv'],
  'arguments': 'GLenum target, GLenum pname, const GLint* params', },
{ 'return_type': 'void',
  'versions': [{'name': 'glTexParameterivRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLenum target, GLenum pname, GLsizei bufSize, const GLint* params', },
{ 'return_type': 'void',
  'known_as': 'glTexStorage2DEXT',
  'versions': [{ 'name': 'glTexStorage2D',
                 'extensions': ['GL_ARB_texture_storage'] },
               { 'name': 'glTexStorage2DEXT',
                 'extensions': ['GL_EXT_texture_storage'] }],
  'arguments': 'GLenum target, GLsizei levels, GLenum internalformat, '
               'GLsizei width, GLsizei height', },
{ 'return_type': 'void',
  'names': ['glTexStorage2DMultisample'],
  'arguments':
      'GLenum target, GLsizei samples, GLenum internalformat, '
      'GLsizei width, GLsizei height, GLboolean fixedsamplelocations', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glTexStorage3D',
                 'extensions': ['GL_ARB_texture_storage'] }],
  'arguments': 'GLenum target, GLsizei levels, GLenum internalformat, '
               'GLsizei width, GLsizei height, GLsizei depth', },
{ 'return_type': 'void',
 'names': [ 'glTexStorageMem2DEXT'] ,
  'arguments': 'GLenum target, GLsizei levels, GLenum internalFormat, '
  'GLsizei width, GLsizei height, GLuint memory, GLuint64 offset', },
{ 'return_type': 'void',
 'names': [ 'glTexStorageMemFlags2DANGLE'] ,
  'versions': [{ 'name': 'glTexStorageMemFlags2DANGLE',
                 'extensions': ['GL_ANGLE_memory_object_flags'] }],
  'arguments': 'GLenum target, GLsizei levels, GLenum internalFormat, '
  'GLsizei width, GLsizei height, GLuint memory, GLuint64 offset, '
  'GLbitfield createFlags, GLbitfield usageFlags', },
{ 'return_type': 'void',
  'names': ['glTexSubImage2D'],
  'arguments':
     'GLenum target, GLint level, GLint xoffset, GLint yoffset, '
     'GLsizei width, GLsizei height, GLenum format, GLenum type, '
     'const void* pixels', },
{ 'return_type': 'void',
  'versions': [{'name': 'glTexSubImage2DRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLenum target, GLint level, GLint xoffset, GLint yoffset, '
      'GLsizei width, GLsizei height, GLenum format, GLenum type, '
      'GLsizei bufSize, const void* pixels', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glTexSubImage3D' }],
  'arguments':
      'GLenum target, GLint level, GLint xoffset, GLint yoffset, '
      'GLint zoffset, GLsizei width, GLsizei height, GLsizei depth, '
      'GLenum format, GLenum type, const void* pixels', },
{ 'return_type': 'void',
  'versions': [{'name': 'glTexSubImage3DRobustANGLE',
                'extensions': ['GL_ANGLE_robust_client_memory']}],
  'arguments':
      'GLenum target, GLint level, GLint xoffset, GLint yoffset, '
      'GLint zoffset, GLsizei width, GLsizei height, GLsizei depth, '
      'GLenum format, GLenum type, GLsizei bufSize, const void* pixels', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glTransformFeedbackVaryings' },
               { 'name': 'glTransformFeedbackVaryingsEXT',
                 'extension': ['GL_EXT_transform_feedback'] }],
  'arguments': 'GLuint program, GLsizei count, const char* const* varyings, '
               'GLenum bufferMode', },
{ 'return_type': 'void',
  'names': ['glUniform1f'],
  'arguments': 'GLint location, GLfloat x', },
{ 'return_type': 'void',
  'names': ['glUniform1fv'],
  'arguments': 'GLint location, GLsizei count, const GLfloat* v', },
{ 'return_type': 'void',
  'names': ['glUniform1i'],
  'arguments': 'GLint location, GLint x', },
{ 'return_type': 'void',
  'names': ['glUniform1iv'],
  'arguments': 'GLint location, GLsizei count, const GLint* v', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glUniform1ui' }],
  'arguments': 'GLint location, GLuint v0', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glUniform1uiv' }],
  'arguments': 'GLint location, GLsizei count, const GLuint* v', },
{ 'return_type': 'void',
  'names': ['glUniform2f'],
  'arguments': 'GLint location, GLfloat x, GLfloat y', },
{ 'return_type': 'void',
  'names': ['glUniform2fv'],
  'arguments': 'GLint location, GLsizei count, const GLfloat* v', },
{ 'return_type': 'void',
  'names': ['glUniform2i'],
  'arguments': 'GLint location, GLint x, GLint y', },
{ 'return_type': 'void',
  'names': ['glUniform2iv'],
  'arguments': 'GLint location, GLsizei count, const GLint* v', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glUniform2ui' }],
  'arguments': 'GLint location, GLuint v0, GLuint v1', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glUniform2uiv' }],
  'arguments': 'GLint location, GLsizei count, const GLuint* v', },
{ 'return_type': 'void',
  'names': ['glUniform3f'],
  'arguments': 'GLint location, GLfloat x, GLfloat y, GLfloat z', },
{ 'return_type': 'void',
  'names': ['glUniform3fv'],
  'arguments': 'GLint location, GLsizei count, const GLfloat* v', },
{ 'return_type': 'void',
  'names': ['glUniform3i'],
  'arguments': 'GLint location, GLint x, GLint y, GLint z', },
{ 'return_type': 'void',
  'names': ['glUniform3iv'],
  'arguments': 'GLint location, GLsizei count, const GLint* v', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glUniform3ui' }],
  'arguments': 'GLint location, GLuint v0, GLuint v1, GLuint v2', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glUniform3uiv' }],
  'arguments': 'GLint location, GLsizei count, const GLuint* v', },
{ 'return_type': 'void',
  'names': ['glUniform4f'],
  'arguments': 'GLint location, GLfloat x, GLfloat y, GLfloat z, GLfloat w', },
{ 'return_type': 'void',
  'names': ['glUniform4fv'],
  'arguments': 'GLint location, GLsizei count, const GLfloat* v', },
{ 'return_type': 'void',
  'names': ['glUniform4i'],
  'arguments': 'GLint location, GLint x, GLint y, GLint z, GLint w', },
{ 'return_type': 'void',
  'names': ['glUniform4iv'],
  'arguments': 'GLint location, GLsizei count, const GLint* v', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glUniform4ui' }],
  'arguments': 'GLint location, GLuint v0, GLuint v1, GLuint v2, GLuint v3', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glUniform4uiv' }],
  'arguments': 'GLint location, GLsizei count, const GLuint* v', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glUniformBlockBinding' }],
  'arguments': 'GLuint program, GLuint uniformBlockIndex, '
               'GLuint uniformBlockBinding', },
{ 'return_type': 'void',
  'names': ['glUniformMatrix2fv'],
  'arguments': 'GLint location, GLsizei count, '
               'GLboolean transpose, const GLfloat* value', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glUniformMatrix2x3fv' }],
  'arguments': 'GLint location, GLsizei count, '
               'GLboolean transpose, const GLfloat* value', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glUniformMatrix2x4fv' }],
  'arguments': 'GLint location, GLsizei count, '
               'GLboolean transpose, const GLfloat* value', },
{ 'return_type': 'void',
  'names': ['glUniformMatrix3fv'],
  'arguments': 'GLint location, GLsizei count, '
               'GLboolean transpose, const GLfloat* value', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glUniformMatrix3x2fv' }],
  'arguments': 'GLint location, GLsizei count, '
               'GLboolean transpose, const GLfloat* value', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glUniformMatrix3x4fv' }],
  'arguments': 'GLint location, GLsizei count, '
               'GLboolean transpose, const GLfloat* value', },
{ 'return_type': 'void',
  'names': ['glUniformMatrix4fv'],
  'arguments': 'GLint location, GLsizei count, '
               'GLboolean transpose, const GLfloat* value', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glUniformMatrix4x2fv' }],
  'arguments': 'GLint location, GLsizei count, '
               'GLboolean transpose, const GLfloat* value', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glUniformMatrix4x3fv' }],
  'arguments': 'GLint location, GLsizei count, '
               'GLboolean transpose, const GLfloat* value', },
{ 'return_type': 'GLboolean',
  'known_as': 'glUnmapBuffer',
  'names': ['glUnmapBufferOES', 'glUnmapBuffer'],
  'arguments': 'GLenum target', },
{ 'return_type': 'void',
  'names': ['glUseProgram'],
  'arguments': 'GLuint program', },
{ 'return_type': 'void',
  'names': ['glUseProgramStages'],
  'arguments': 'GLuint pipeline, GLbitfield stages, GLuint program', },
{ 'return_type': 'void',
  'names': ['glValidateProgram'],
  'arguments': 'GLuint program', },
{ 'return_type': 'void',
  'names': ['glValidateProgramPipeline'],
  'arguments': 'GLuint pipeline', },
{ 'return_type': 'void',
  'names': ['glVertexAttrib1f'],
  'arguments': 'GLuint indx, GLfloat x', },
{ 'return_type': 'void',
  'names': ['glVertexAttrib1fv'],
  'arguments': 'GLuint indx, const GLfloat* values', },
{ 'return_type': 'void',
  'names': ['glVertexAttrib2f'],
  'arguments': 'GLuint indx, GLfloat x, GLfloat y', },
{ 'return_type': 'void',
  'names': ['glVertexAttrib2fv'],
  'arguments': 'GLuint indx, const GLfloat* values', },
{ 'return_type': 'void',
  'names': ['glVertexAttrib3f'],
  'arguments': 'GLuint indx, GLfloat x, GLfloat y, GLfloat z', },
{ 'return_type': 'void',
  'names': ['glVertexAttrib3fv'],
  'arguments': 'GLuint indx, const GLfloat* values', },
{ 'return_type': 'void',
  'names': ['glVertexAttrib4f'],
  'arguments': 'GLuint indx, GLfloat x, GLfloat y, GLfloat z, GLfloat w', },
{ 'return_type': 'void',
  'names': ['glVertexAttrib4fv'],
  'arguments': 'GLuint indx, const GLfloat* values', },
{ 'return_type': 'void',
  'names': ['glVertexAttribBinding'],
  'arguments': 'GLuint attribindex, GLuint bindingindex', },
{ 'return_type': 'void',
  'known_as': 'glVertexAttribDivisorANGLE',
  'names': ['glVertexAttribDivisorARB', 'glVertexAttribDivisorANGLE',
            'glVertexAttribDivisorEXT', 'glVertexAttribDivisor'],
  'arguments':
      'GLuint index, GLuint divisor', },
{ 'return_type': 'void',
  'names': ['glVertexAttribFormat'],
  'arguments': 'GLuint attribindex, GLint size, GLenum type, '
               'GLboolean normalized, GLuint relativeoffset', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glVertexAttribI4i' }],
  'arguments': 'GLuint indx, GLint x, GLint y, GLint z, GLint w', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glVertexAttribI4iv' }],
  'arguments': 'GLuint indx, const GLint* values', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glVertexAttribI4ui' }],
  'arguments': 'GLuint indx, GLuint x, GLuint y, GLuint z, GLuint w', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glVertexAttribI4uiv' }],
  'arguments': 'GLuint indx, const GLuint* values', },
{ 'return_type': 'void',
  'names': ['glVertexAttribIFormat'],
  'arguments': 'GLuint attribindex, GLint size, GLenum type, '
               'GLuint relativeoffset', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glVertexAttribIPointer' }],
  'arguments': 'GLuint indx, GLint size, GLenum type, GLsizei stride, '
               'const void* ptr', },
{ 'return_type': 'void',
  'names': ['glVertexAttribPointer'],
  'arguments': 'GLuint indx, GLint size, GLenum type, GLboolean normalized, '
               'GLsizei stride, const void* ptr', },
{ 'return_type': 'void',
  'names': ['glVertexBindingDivisor'],
  'arguments': 'GLuint bindingindex, GLuint divisor', },
{ 'return_type': 'void',
  'names': ['glViewport'],
  'arguments': 'GLint x, GLint y, GLsizei width, GLsizei height', },
{ 'return_type': 'void',
  'names': ['glWaitSemaphoreEXT'],
  'arguments': 'GLuint semaphore, GLuint numBufferBarriers, '
  'const GLuint* buffers, GLuint numTextureBarriers, const GLuint* textures, '
  'const GLenum* srcLayouts', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glWaitSync',
                 'extensions': ['GL_ARB_sync'] }],
  'arguments':
    'GLsync sync, GLbitfield flags, GLuint64 timeout', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'glWaitSyncAPPLE',
                 'extensions': ['GL_APPLE_sync'] }],
  'arguments':
    'GLsync sync, GLbitfield flags, GLuint64 timeout', },
{ 'return_type': 'void',
  'names': ['glWindowRectanglesEXT'],
  'arguments': 'GLenum mode, GLsizei n, const GLint* box', },
]

EGL_FUNCTIONS = [
{ 'return_type': 'EGLBoolean',
  'names': ['eglBindAPI'],
  'arguments': 'EGLenum api', },
{ 'return_type': 'EGLBoolean',
  'names': ['eglBindTexImage'],
  'arguments': 'EGLDisplay dpy, EGLSurface surface, EGLint buffer', },
{ 'return_type': 'EGLBoolean',
  'names': ['eglChooseConfig'],
  'arguments': 'EGLDisplay dpy, const EGLint* attrib_list, EGLConfig* configs, '
               'EGLint config_size, EGLint* num_config', },
{ 'return_type': 'EGLint',
  'versions': [{ 'name': 'eglClientWaitSyncKHR',
                 'extensions': [
                   'EGL_KHR_fence_sync',
                   'GL_CHROMIUM_egl_khr_fence_sync_hack'
                 ] }],
  'arguments': 'EGLDisplay dpy, EGLSyncKHR sync, EGLint flags, '
      'EGLTimeKHR timeout' },
{ 'return_type': 'EGLBoolean',
  'names': ['eglCopyBuffers'],
  'arguments':
      'EGLDisplay dpy, EGLSurface surface, EGLNativePixmapType target', },
{ 'return_type': 'EGLContext',
  'names': ['eglCreateContext'],
  'arguments': 'EGLDisplay dpy, EGLConfig config, EGLContext share_context, '
              'const EGLint* attrib_list', },
{ 'return_type': 'EGLImageKHR',
  'versions': [{ 'name': 'eglCreateImageKHR',
                 'extensions':
                     ['EGL_KHR_image_base', 'EGL_KHR_gl_texture_2D_image'] }],
  'arguments':
      'EGLDisplay dpy, EGLContext ctx, EGLenum target, EGLClientBuffer buffer, '
      'const EGLint* attrib_list' },
{ 'return_type': 'EGLSurface',
  'names': ['eglCreatePbufferFromClientBuffer'],
  'arguments':
      'EGLDisplay dpy, EGLenum buftype, void* buffer, EGLConfig config, '
      'const EGLint* attrib_list', },
{ 'return_type': 'EGLSurface',
  'names': ['eglCreatePbufferSurface'],
  'arguments': 'EGLDisplay dpy, EGLConfig config, const EGLint* attrib_list', },
{ 'return_type': 'EGLSurface',
  'names': ['eglCreatePixmapSurface'],
  'arguments': 'EGLDisplay dpy, EGLConfig config, EGLNativePixmapType pixmap, '
               'const EGLint* attrib_list', },
{ 'return_type': 'EGLStreamKHR',
  'versions': [{ 'name': 'eglCreateStreamKHR',
                 'extensions': ['EGL_KHR_stream'] }],
  'arguments': 'EGLDisplay dpy, const EGLint* attrib_list' },
{ 'return_type': 'EGLBoolean',
    'versions': [{'name': 'eglCreateStreamProducerD3DTextureANGLE',
                  'extensions':
                      ['EGL_ANGLE_stream_producer_d3d_texture']}],
  'arguments':
      'EGLDisplay dpy, EGLStreamKHR stream, EGLAttrib* attrib_list', },
{ 'return_type': 'EGLSyncKHR',
  'versions': [{ 'name': 'eglCreateSyncKHR',
                 'extensions': [
                   'EGL_KHR_fence_sync',
                   'GL_CHROMIUM_egl_khr_fence_sync_hack'
                 ] }],
  'arguments': 'EGLDisplay dpy, EGLenum type, const EGLint* attrib_list' },
{ 'return_type': 'EGLSurface',
  'names': ['eglCreateWindowSurface'],
  'arguments': 'EGLDisplay dpy, EGLConfig config, EGLNativeWindowType win, '
               'const EGLint* attrib_list', },
{ 'return_type': 'EGLint',
  'known_as': 'eglDebugMessageControlKHR',
  'versions': [{ 'name': 'eglDebugMessageControlKHR',
                 'client_extensions': ['EGL_KHR_debug'], }],
  'arguments': 'EGLDEBUGPROCKHR callback, const EGLAttrib* attrib_list', },
{ 'return_type': 'EGLBoolean',
  'names': ['eglDestroyContext'],
  'arguments': 'EGLDisplay dpy, EGLContext ctx', },
{ 'return_type': 'EGLBoolean',
  'versions': [{ 'name' : 'eglDestroyImageKHR',
                 'extensions': ['EGL_KHR_image_base'] }],
  'arguments': 'EGLDisplay dpy, EGLImageKHR image' },
{ 'return_type': 'EGLBoolean',
  'versions': [{ 'name': 'eglDestroyStreamKHR',
                 'extensions': ['EGL_KHR_stream'] }],
  'arguments': 'EGLDisplay dpy, EGLStreamKHR stream' },
{ 'return_type': 'EGLBoolean',
  'names': ['eglDestroySurface'],
  'arguments': 'EGLDisplay dpy, EGLSurface surface', },
{ 'return_type': 'EGLBoolean',
  'versions': [{ 'name': 'eglDestroySyncKHR',
                 'extensions': [
                   'EGL_KHR_fence_sync',
                   'GL_CHROMIUM_egl_khr_fence_sync_hack'
                 ] }],
  'arguments': 'EGLDisplay dpy, EGLSyncKHR sync' },
{ 'return_type': 'EGLint',
  # At least some Android O devices such as Pixel implement this
  # but don't export the EGL_ANDROID_native_fence_sync extension.
  'versions': [{ 'name': 'eglDupNativeFenceFDANDROID',
                 'extensions': [
                     'EGL_ANDROID_native_fence_sync',
                     'GL_CHROMIUM_egl_android_native_fence_sync_hack']}],
  'arguments':
      'EGLDisplay dpy, EGLSyncKHR sync' },
{ 'return_type': 'EGLBoolean',
  'versions': [{ 'name': 'eglExportDMABUFImageMESA',
                 'extensions': ['EGL_MESA_image_dma_buf_export'] }],
  'arguments': 'EGLDisplay dpy, EGLImageKHR image, int* fds, EGLint* strides, '
               'EGLint* offsets', },
{ 'return_type': 'EGLBoolean',
  'versions': [{ 'name': 'eglExportDMABUFImageQueryMESA',
                 'extensions': ['EGL_MESA_image_dma_buf_export'] }],
  'arguments': 'EGLDisplay dpy, EGLImageKHR image, int* fourcc, '
               'int* num_planes, EGLuint64KHR* modifiers', },
{ 'return_type': 'EGLBoolean',
  'versions': [{ 'name': 'eglGetCompositorTimingANDROID',
                 'extensions': [
                   'EGL_ANDROID_get_frame_timestamps'
                 ] }],
  'arguments': 'EGLDisplay dpy, EGLSurface surface, EGLint numTimestamps, '
               'EGLint* names, EGLnsecsANDROID* values', },
{ 'return_type': 'EGLBoolean',
  'versions': [{ 'name': 'eglGetCompositorTimingSupportedANDROID',
                 'extensions': [
                   'EGL_ANDROID_get_frame_timestamps'
                 ] }],
  'arguments': 'EGLDisplay dpy, EGLSurface surface, EGLint timestamp', },
{ 'return_type': 'EGLBoolean',
  'names': ['eglGetConfigAttrib'],
  'arguments':
      'EGLDisplay dpy, EGLConfig config, EGLint attribute, EGLint* value', },
{ 'return_type': 'EGLBoolean',
  'names': ['eglGetConfigs'],
  'arguments': 'EGLDisplay dpy, EGLConfig* configs, EGLint config_size, '
               'EGLint* num_config', },
{ 'return_type': 'EGLContext',
  'names': ['eglGetCurrentContext'],
  'arguments': 'void', },
{ 'return_type': 'EGLDisplay',
  'names': ['eglGetCurrentDisplay'],
  'arguments': 'void', },
{ 'return_type': 'EGLSurface',
  'names': ['eglGetCurrentSurface'],
  'arguments': 'EGLint readdraw', },
{ 'return_type': 'EGLDisplay',
  'names': ['eglGetDisplay'],
  'arguments': 'EGLNativeDisplayType display_id', },
{ 'return_type': 'EGLint',
  'names': ['eglGetError'],
  'arguments': 'void', },
 { 'return_type': 'EGLBoolean',
  'versions': [{ 'name': 'eglGetFrameTimestampsANDROID',
                 'extensions': [
                   'EGL_ANDROID_get_frame_timestamps'
                 ] }],
  'arguments': 'EGLDisplay dpy, EGLSurface surface, EGLuint64KHR frameId, '
               'EGLint numTimestamps, EGLint* timestamps, '
               'EGLnsecsANDROID* values', },
{ 'return_type': 'EGLBoolean',
  'versions': [{ 'name': 'eglGetFrameTimestampSupportedANDROID',
                 'extensions': [
                   'EGL_ANDROID_get_frame_timestamps'
                 ] }],
  'arguments': 'EGLDisplay dpy, EGLSurface surface, EGLint timestamp', },
{ 'return_type': 'EGLBoolean',
  'versions': [{ 'name': 'eglGetMscRateANGLE',
                 'extensions': [
                   'EGL_ANGLE_sync_control_rate'
                 ] }],
  'arguments':
      'EGLDisplay dpy, EGLSurface surface, '
      'EGLint* numerator, EGLint* denominator', },
{ 'return_type': 'EGLClientBuffer',
  'versions': [{ 'name': 'eglGetNativeClientBufferANDROID',
                 'extensions': ['EGL_ANDROID_get_native_client_buffer'], }],
  'arguments': 'const struct AHardwareBuffer* ahardwarebuffer', },
{ 'return_type': 'EGLBoolean',
  'versions': [{ 'name': 'eglGetNextFrameIdANDROID',
                 'extensions': [
                   'EGL_ANDROID_get_frame_timestamps'
                 ] }],
  'arguments': 'EGLDisplay dpy, EGLSurface surface, EGLuint64KHR* frameId', },
{ 'return_type': 'EGLDisplay',
  'names': ['eglGetPlatformDisplay'],
  'arguments': 'EGLenum platform, void* native_display, '
               'const EGLAttrib* attrib_list', },
{ 'return_type': '__eglMustCastToProperFunctionPointerType',
  'names': ['eglGetProcAddress'],
  'arguments': 'const char* procname',
  'logging_code': """
  GL_SERVICE_LOG("GL_RESULT: " << reinterpret_cast<void*>(result));
""", },
{ 'return_type': 'EGLBoolean',
  'versions': [{ 'name': 'eglGetSyncAttribKHR',
                 'extensions': [
                   'EGL_KHR_fence_sync',
                   'GL_CHROMIUM_egl_khr_fence_sync_hack'
                 ] }],
  'arguments': 'EGLDisplay dpy, EGLSyncKHR sync, EGLint attribute, '
      'EGLint* value' },
{ 'return_type': 'EGLBoolean',
  'versions': [{ 'name': 'eglGetSyncValuesCHROMIUM',
                 'extensions': [
                   'EGL_CHROMIUM_sync_control'
                 ] }],
  'arguments':
      'EGLDisplay dpy, EGLSurface surface, '
      'EGLuint64CHROMIUM* ust, EGLuint64CHROMIUM* msc, '
      'EGLuint64CHROMIUM* sbc', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'eglHandleGPUSwitchANGLE',
                 'extensions': ['EGL_ANGLE_power_preference'] }],
  'arguments': 'EGLDisplay dpy' },
{ 'return_type': 'EGLBoolean',
  'versions': [{ 'name': 'eglImageFlushExternalEXT',
                 'extensions': ['EGL_EXT_image_flush_external'] }],
  'arguments':
      'EGLDisplay dpy, EGLImageKHR image, const EGLAttrib* attrib_list' },
{ 'return_type': 'EGLBoolean',
  'names': ['eglInitialize'],
  'arguments': 'EGLDisplay dpy, EGLint* major, EGLint* minor', },
{ 'return_type': 'EGLint',
  'known_as': 'eglLabelObjectKHR',
  'versions': [{ 'name': 'eglLabelObjectKHR',
                 'client_extensions': ['EGL_KHR_debug'], }],
  'arguments': 'EGLDisplay display, EGLenum objectType, EGLObjectKHR object, '
    'EGLLabelKHR label', },
{ 'return_type': 'EGLBoolean',
  'names': ['eglMakeCurrent'],
  'arguments':
      'EGLDisplay dpy, EGLSurface draw, EGLSurface read, EGLContext ctx', },
{ 'return_type': 'EGLBoolean',
  'names': ['eglPostSubBufferNV'],
  'arguments': 'EGLDisplay dpy, EGLSurface surface, '
    'EGLint x, EGLint y, EGLint width, EGLint height', },
{ 'return_type': 'EGLenum',
  'names': ['eglQueryAPI'],
  'arguments': 'void', },
{ 'return_type': 'EGLBoolean',
  'names': ['eglQueryContext'],
  'arguments':
      'EGLDisplay dpy, EGLContext ctx, EGLint attribute, EGLint* value', },
{ 'return_type': 'EGLBoolean',
  'known_as': 'eglQueryDebugKHR',
  'versions': [{ 'name': 'eglQueryDebugKHR',
                 'client_extensions': ['EGL_KHR_debug'], }],
  'arguments': 'EGLint attribute, EGLAttrib* value', },
{ 'return_type': 'EGLBoolean',
  'known_as': 'eglQueryDevicesEXT',
  'versions': [{ 'name': 'eglQueryDevicesEXT',
                 'client_extensions': ['EGL_EXT_device_enumeration'], }],
  'arguments':
      'EGLint max_devices, EGLDeviceEXT* devices, EGLint* num_devices', },
{ 'return_type': 'const char *',
  'known_as': 'eglQueryDeviceStringEXT',
  'versions': [{ 'name': 'eglQueryDeviceStringEXT',
                 'client_extensions': ['EGL_EXT_device_query'], }],
  'arguments': 'EGLDeviceEXT device, EGLint name', },
{ 'return_type': 'EGLBoolean',
  'versions': [{ 'name': 'eglQueryDisplayAttribANGLE',
                 'client_extensions': ['EGL_ANGLE_feature_control'] }],
  'arguments': 'EGLDisplay dpy, EGLint attribute, EGLAttrib* value' },
{ 'return_type': 'EGLBoolean',
  'versions': [{ 'name': 'eglQueryStreamKHR',
                 'extensions': ['EGL_KHR_stream'] }],
  'arguments':
      'EGLDisplay dpy, EGLStreamKHR stream, EGLenum attribute, '
      'EGLint* value' },
{ 'return_type': 'EGLBoolean',
  'versions': [{ 'name': 'eglQueryStreamu64KHR',
                 'extensions': ['EGL_KHR_stream'] }],
  'arguments':
      'EGLDisplay dpy, EGLStreamKHR stream, EGLenum attribute, '
      'EGLuint64KHR* value' },
{ 'return_type': 'const char*',
  'names': ['eglQueryString'],
  'arguments': 'EGLDisplay dpy, EGLint name', },
{ 'return_type': 'const char *',
  'versions': [{ 'name': 'eglQueryStringiANGLE',
                 'client_extensions': ['EGL_ANGLE_feature_control'] }],
  'arguments': 'EGLDisplay dpy, EGLint name, EGLint index' },
{ 'return_type': 'EGLBoolean',
  'names': ['eglQuerySurface'],
  'arguments':
      'EGLDisplay dpy, EGLSurface surface, EGLint attribute, EGLint* value', },
{ 'return_type': 'EGLBoolean',
  'names': ['eglQuerySurfacePointerANGLE'],
  'arguments':
      'EGLDisplay dpy, EGLSurface surface, EGLint attribute, void** value', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'eglReacquireHighPowerGPUANGLE',
                 'extensions': ['EGL_ANGLE_power_preference'] }],
  'arguments': 'EGLDisplay dpy, EGLContext ctx' },
{ 'return_type': 'void',
  'versions': [{ 'name': 'eglReleaseHighPowerGPUANGLE',
                 'extensions': ['EGL_ANGLE_power_preference'] }],
  'arguments': 'EGLDisplay dpy, EGLContext ctx' },
{ 'return_type': 'EGLBoolean',
  'names': ['eglReleaseTexImage'],
  'arguments': 'EGLDisplay dpy, EGLSurface surface, EGLint buffer', },
{ 'return_type': 'EGLBoolean',
  'names': ['eglReleaseThread'],
  'arguments': 'void', },
{ 'return_type': 'void',
  'versions': [{ 'name': 'eglSetBlobCacheFuncsANDROID',
                 'extensions': ['EGL_ANDROID_blob_cache'] }],
  'arguments':
      'EGLDisplay dpy, EGLSetBlobFuncANDROID set, EGLGetBlobFuncANDROID get' },
{ 'return_type': 'EGLBoolean',
  'versions': [{ 'name': 'eglStreamAttribKHR',
                 'extensions': ['EGL_KHR_stream'] }],
  'arguments':
      'EGLDisplay dpy, EGLStreamKHR stream, EGLenum attribute, EGLint value' },
{ 'return_type': 'EGLBoolean',
    'versions': [{ 'name': 'eglStreamConsumerAcquireKHR',
                   'extensions': ['EGL_KHR_stream_consumer_gltexture']}],
  'arguments':
      'EGLDisplay dpy, EGLStreamKHR stream', },
{ 'return_type': 'EGLBoolean',
    'versions': [{ 'name': 'eglStreamConsumerGLTextureExternalAttribsNV',
                   'extensions': ['EGL_NV_stream_consumer_gltexture_yuv']}],
  'arguments':
      'EGLDisplay dpy, EGLStreamKHR stream, EGLAttrib* attrib_list', },
{ 'return_type': 'EGLBoolean',
    'versions': [{ 'name': 'eglStreamConsumerGLTextureExternalKHR',
                   'extensions': ['EGL_KHR_stream_consumer_gltexture']}],
  'arguments':
      'EGLDisplay dpy, EGLStreamKHR stream', },
{ 'return_type': 'EGLBoolean',
    'versions': [{ 'name': 'eglStreamConsumerReleaseKHR',
                   'extensions': ['EGL_KHR_stream_consumer_gltexture']}],
  'arguments':
      'EGLDisplay dpy, EGLStreamKHR stream', },
{ 'return_type': 'EGLBoolean',
    'versions': [{ 'name': 'eglStreamPostD3DTextureANGLE',
                   'extensions':
                       ['EGL_ANGLE_stream_producer_d3d_texture']}],
  'arguments':
      'EGLDisplay dpy, EGLStreamKHR stream, void* texture, '
      'const EGLAttrib* attrib_list', },
{ 'return_type': 'EGLBoolean',
  'names': ['eglSurfaceAttrib'],
  'arguments':
      'EGLDisplay dpy, EGLSurface surface, EGLint attribute, EGLint value', },
{ 'return_type': 'EGLBoolean',
  'names': ['eglSwapBuffers'],
  'arguments': 'EGLDisplay dpy, EGLSurface surface', },
{ 'return_type': 'EGLBoolean',
  'names': ['eglSwapBuffersWithDamageKHR'],
  'arguments':
      'EGLDisplay dpy, EGLSurface surface, EGLint* rects, EGLint n_rects', },
{ 'return_type': 'EGLBoolean',
  'names': ['eglSwapInterval'],
  'arguments': 'EGLDisplay dpy, EGLint interval', },
{ 'return_type': 'EGLBoolean',
  'names': ['eglTerminate'],
  'arguments': 'EGLDisplay dpy', },
{ 'return_type': 'EGLBoolean',
  'names': ['eglWaitClient'],
  'arguments': 'void', },
{ 'return_type': 'EGLBoolean',
  'names': ['eglWaitGL'],
  'arguments': 'void', },
{ 'return_type': 'EGLBoolean',
  'names': ['eglWaitNative'],
  'arguments': 'EGLint engine', },
{ 'return_type': 'EGLint',
  'versions': [{ 'name': 'eglWaitSyncKHR',
                 'extensions': ['EGL_KHR_wait_sync'] }],
  'arguments': 'EGLDisplay dpy, EGLSyncKHR sync, EGLint flags' },
]

WGL_FUNCTIONS = [
{ 'return_type': 'BOOL',
  'names': ['wglChoosePixelFormatARB'],
  'arguments':
      'HDC dc, const int* int_attrib_list, const float* float_attrib_list, '
      'UINT max_formats, int* formats, UINT* num_formats', },
{ 'return_type': 'BOOL',
  'names': ['wglCopyContext'],
  'arguments': 'HGLRC hglrcSrc, HGLRC hglrcDst, UINT mask', },
{ 'return_type': 'HGLRC',
  'names': ['wglCreateContext'],
  'arguments': 'HDC hdc', },
{ 'return_type': 'HGLRC',
  'names': ['wglCreateContextAttribsARB'],
  'arguments': 'HDC hDC, HGLRC hShareContext, const int* attribList', },
{ 'return_type': 'HGLRC',
  'names': ['wglCreateLayerContext'],
  'arguments': 'HDC hdc, int iLayerPlane', },
{ 'return_type': 'HPBUFFERARB',
  'names': ['wglCreatePbufferARB'],
  'arguments': 'HDC hDC, int iPixelFormat, int iWidth, int iHeight, '
               'const int* piAttribList', },
{ 'return_type': 'BOOL',
  'names': ['wglDeleteContext'],
  'arguments': 'HGLRC hglrc', },
{ 'return_type': 'BOOL',
  'names': ['wglDestroyPbufferARB'],
  'arguments': 'HPBUFFERARB hPbuffer', },
{ 'return_type': 'HGLRC',
  'names': ['wglGetCurrentContext'],
  'arguments': '', },
{ 'return_type': 'HDC',
  'names': ['wglGetCurrentDC'],
  'arguments': '', },
{ 'return_type': 'const char*',
  'names': ['wglGetExtensionsStringARB'],
  'arguments': 'HDC hDC', },
{ 'return_type': 'const char*',
  'names': ['wglGetExtensionsStringEXT'],
  'arguments': '', },
{ 'return_type': 'HDC',
  'names': ['wglGetPbufferDCARB'],
  'arguments': 'HPBUFFERARB hPbuffer', },
{ 'return_type': 'BOOL',
  'names': ['wglMakeCurrent'],
  'arguments': 'HDC hdc, HGLRC hglrc', },
{ 'return_type': 'BOOL',
  'names': ['wglQueryPbufferARB'],
  'arguments': 'HPBUFFERARB hPbuffer, int iAttribute, int* piValue', },
{ 'return_type': 'int',
  'names': ['wglReleasePbufferDCARB'],
  'arguments': 'HPBUFFERARB hPbuffer, HDC hDC', },
{ 'return_type': 'BOOL',
  'names': ['wglShareLists'],
  'arguments': 'HGLRC hglrc1, HGLRC hglrc2', },
{ 'return_type': 'BOOL',
  'names': ['wglSwapIntervalEXT'],
  'arguments': 'int interval', },
{ 'return_type': 'BOOL',
  'names': ['wglSwapLayerBuffers'],
  'arguments': 'HDC hdc, UINT fuPlanes', },
]

GLX_FUNCTIONS = [
{ 'return_type': 'void',
  'names': ['glXBindTexImageEXT'],
  'arguments':
      'Display* dpy, GLXDrawable drawable, int buffer, int* attribList', },
{ 'return_type': 'GLXFBConfig*',
  'names': ['glXChooseFBConfig'],
  'arguments':
      'Display* dpy, int screen, const int* attribList, int* nitems', },
{ 'return_type': 'XVisualInfo*',
  'names': ['glXChooseVisual'],
  'arguments': 'Display* dpy, int screen, int* attribList', },
{ 'return_type': 'void',
  'names': ['glXCopyContext'],
  'arguments':
      'Display* dpy, GLXContext src, GLXContext dst, unsigned long mask', },
{ 'return_type': 'void',
  'names': ['glXCopySubBufferMESA'],
  'arguments': 'Display* dpy, GLXDrawable drawable, '
               'int x, int y, int width, int height', },
{ 'return_type': 'GLXContext',
  'names': ['glXCreateContext'],
  'arguments':
      'Display* dpy, XVisualInfo* vis, GLXContext shareList, int direct', },
{ 'return_type': 'GLXContext',
  'names': ['glXCreateContextAttribsARB'],
  'arguments':
      'Display* dpy, GLXFBConfig config, GLXContext share_context, int direct, '
      'const int* attrib_list', },
{ 'return_type': 'GLXPixmap',
  'names': ['glXCreateGLXPixmap'],
  'arguments': 'Display* dpy, XVisualInfo* visual, Pixmap pixmap', },
{ 'return_type': 'GLXContext',
  'names': ['glXCreateNewContext'],
  'arguments': 'Display* dpy, GLXFBConfig config, int renderType, '
               'GLXContext shareList, int direct', },
{ 'return_type': 'GLXPbuffer',
  'names': ['glXCreatePbuffer'],
  'arguments': 'Display* dpy, GLXFBConfig config, const int* attribList', },
{ 'return_type': 'GLXPixmap',
  'names': ['glXCreatePixmap'],
  'arguments': 'Display* dpy, GLXFBConfig config, '
               'Pixmap pixmap, const int* attribList', },
{ 'return_type': 'GLXWindow',
  'names': ['glXCreateWindow'],
  'arguments':
      'Display* dpy, GLXFBConfig config, Window win, const int* attribList', },
{ 'return_type': 'void',
  'names': ['glXDestroyContext'],
  'arguments': 'Display* dpy, GLXContext ctx', },
{ 'return_type': 'void',
  'names': ['glXDestroyGLXPixmap'],
  'arguments': 'Display* dpy, GLXPixmap pixmap', },
{ 'return_type': 'void',
  'names': ['glXDestroyPbuffer'],
  'arguments': 'Display* dpy, GLXPbuffer pbuf', },
{ 'return_type': 'void',
  'names': ['glXDestroyPixmap'],
  'arguments': 'Display* dpy, GLXPixmap pixmap', },
{ 'return_type': 'void',
  'names': ['glXDestroyWindow'],
  'arguments': 'Display* dpy, GLXWindow window', },
{ 'return_type': 'const char*',
  'names': ['glXGetClientString'],
  'arguments': 'Display* dpy, int name', },
{ 'return_type': 'int',
  'names': ['glXGetConfig'],
  'arguments': 'Display* dpy, XVisualInfo* visual, int attrib, int* value', },
{ 'return_type': 'GLXContext',
  'names': ['glXGetCurrentContext'],
  'arguments': 'void', },
{ 'return_type': 'Display*',
  'names': ['glXGetCurrentDisplay'],
  'arguments': 'void', },
{ 'return_type': 'GLXDrawable',
  'names': ['glXGetCurrentDrawable'],
  'arguments': 'void', },
{ 'return_type': 'GLXDrawable',
  'names': ['glXGetCurrentReadDrawable'],
  'arguments': 'void', },
{ 'return_type': 'int',
  'names': ['glXGetFBConfigAttrib'],
  'arguments': 'Display* dpy, GLXFBConfig config, int attribute, int* value', },
{ 'return_type': 'GLXFBConfig',
  'names': ['glXGetFBConfigFromVisualSGIX'],
  'arguments': 'Display* dpy, XVisualInfo* visualInfo', },
{ 'return_type': 'GLXFBConfig*',
  'names': ['glXGetFBConfigs'],
  'arguments': 'Display* dpy, int screen, int* nelements', },
{ 'return_type': 'bool',
  'names': ['glXGetMscRateOML'],
  'arguments':
      'Display* dpy, GLXDrawable drawable, int32_t* numerator, '
      'int32_t* denominator' },
{ 'return_type': 'void',
  'names': ['glXGetSelectedEvent'],
  'arguments': 'Display* dpy, GLXDrawable drawable, unsigned long* mask', },
{ 'return_type': 'bool',
  'names': ['glXGetSyncValuesOML'],
  'arguments':
      'Display* dpy, GLXDrawable drawable, int64_t* ust, int64_t* msc, '
      'int64_t* sbc' },
{ 'return_type': 'XVisualInfo*',
  'names': ['glXGetVisualFromFBConfig'],
  'arguments': 'Display* dpy, GLXFBConfig config', },
{ 'return_type': 'int',
  'names': ['glXIsDirect'],
  'arguments': 'Display* dpy, GLXContext ctx', },
{ 'return_type': 'int',
  'names': ['glXMakeContextCurrent'],
  'arguments':
      'Display* dpy, GLXDrawable draw, GLXDrawable read, GLXContext ctx', },
{ 'return_type': 'int',
  'names': ['glXMakeCurrent'],
  'arguments': 'Display* dpy, GLXDrawable drawable, GLXContext ctx', },
{ 'return_type': 'int',
  'names': ['glXQueryContext'],
  'arguments': 'Display* dpy, GLXContext ctx, int attribute, int* value', },
{ 'return_type': 'void',
  'names': ['glXQueryDrawable'],
  'arguments':
      'Display* dpy, GLXDrawable draw, int attribute, unsigned int* value', },
{ 'return_type': 'int',
  'names': ['glXQueryExtension'],
  'arguments': 'Display* dpy, int* errorb, int* event', },
{ 'return_type': 'const char*',
  'names': ['glXQueryExtensionsString'],
  'arguments': 'Display* dpy, int screen', },
{ 'return_type': 'const char*',
  'names': ['glXQueryServerString'],
  'arguments': 'Display* dpy, int screen, int name', },
{ 'return_type': 'int',
  'names': ['glXQueryVersion'],
  'arguments': 'Display* dpy, int* maj, int* min', },
{ 'return_type': 'void',
  'names': ['glXReleaseTexImageEXT'],
  'arguments': 'Display* dpy, GLXDrawable drawable, int buffer', },
{ 'return_type': 'void',
  'names': ['glXSelectEvent'],
  'arguments': 'Display* dpy, GLXDrawable drawable, unsigned long mask', },
{ 'return_type': 'void',
  'names': ['glXSwapBuffers'],
  'arguments': 'Display* dpy, GLXDrawable drawable', },
{ 'return_type': 'void',
  'names': ['glXSwapIntervalEXT'],
  'arguments': 'Display* dpy, GLXDrawable drawable, int interval', },
{ 'return_type': 'void',
  'names': ['glXSwapIntervalMESA'],
  'arguments': 'unsigned int interval', },
{ 'return_type': 'void',
  'names': ['glXUseXFont'],
  'arguments': 'Font font, int first, int count, int list', },
{ 'return_type': 'void',
  'names': ['glXWaitGL'],
  'arguments': 'void', },
{ 'return_type': 'int',
  'names': ['glXWaitVideoSyncSGI'],
  'arguments': 'int divisor, int remainder, unsigned int* count', },
{ 'return_type': 'void',
  'names': ['glXWaitX'],
  'arguments': 'void', },
]

FUNCTION_SETS = [
  [GL_FUNCTIONS, 'gl', [
      'GL/gl.h',
      'noninclude/GL/glext.h',
      'GLES2/gl2ext.h',
      'GLES3/gl3.h',
      'GLES3/gl31.h',
      'GLES3/gl32.h',
      # Files below are Chromium-specific and shipped with Chromium sources.
      'GL/glextchromium.h',
      'GLES2/gl2chromium.h',
      'GLES2/gl2extchromium.h'
    ], [
      "GL_ANGLE_robust_resource_initialization",
      "GL_ARB_texture_swizzle",
      "GL_EXT_texture_swizzle",
      "GL_EXT_texture_format_BGRA8888",
      "GL_EXT_unpack_subimage",
    ]
  ],
  [EGL_FUNCTIONS, 'egl', [
      'EGL/eglext.h',
      # Files below are Chromium-specific and shipped with Chromium sources.
      'EGL/eglextchromium.h',
    ],
    [
      'EGL_ANGLE_d3d_share_handle_client_buffer',
      'EGL_ANGLE_surface_d3d_texture_2d_share_handle',
    ],
  ],
  [WGL_FUNCTIONS, 'wgl', ['noninclude/GL/wglext.h'], []],
  [GLX_FUNCTIONS, 'glx', ['GL/glx.h', 'noninclude/GL/glxext.h'], []],
]

GLES2_HEADERS_WITH_ENUMS = [
  'GLES2/gl2.h',
  'GLES2/gl2ext.h',
  'GLES2/gl2chromium.h',
  'GLES2/gl2extchromium.h',
  'GLES3/gl3.h',
  'GLES3/gl31.h',
]

SELF_LOCATION = os.path.dirname(os.path.abspath(__file__))

LICENSE_AND_HEADER = """\
// Copyright 2016 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
//
// This file is auto-generated from
// ui/gl/generate_bindings.py
// It's formatted by clang-format using chromium coding style:
//    clang-format -i -style=chromium filename
// DO NOT EDIT!

"""

GLVersion = namedtuple('GLVersion', 'is_es major_version minor_version')

def GLVersionBindAlways(version):
  return version.major_version <= 2


def GetStaticBinding(func):
  """If this function has a name assigned to it that should  be bound always,
  then return this name.

  This will be the case if either a function name is specified
  that depends on an extension from UNCONDITIONALLY_BOUND_EXTENSIONS,
  or if the GL version it depends on is assumed to be available (e.g. <=2.1).
  There can only be one name that satisfies this condition (or the bindings
  would be ambiguous)."""

  static_bindings = set([])

  for version in func['versions']:
    if 'extensions' in version:
      extensions = version['extensions']
      num_unconditional_extensions = len(
          extensions & UNCONDITIONALLY_BOUND_EXTENSIONS)
      if num_unconditional_extensions:
        static_bindings.add(version['name'])
    elif 'gl_versions' in version:
      versions = [v for v in version['gl_versions'] if GLVersionBindAlways(v)]
      # It's only unconditional if it exists in GL and GLES
      if len(versions) == 2:
        assert versions[0].is_es != versions[1].is_es
        static_bindings.add(version['name'])
    else:
        static_bindings.add(version['name'])

  # Avoid ambiguous bindings (static binding with different names)
  assert len(static_bindings) <= 1
  if len(static_bindings):
    static_name = static_bindings.pop()
    # Avoid ambiguous bindings (static and dynamic bindings with
    # different names)
    assert len([v['name'] for v in func['versions']
               if v['name'] != static_name]) == 0, func
    return static_name
  else:
    return None


def GenerateHeader(file, functions, set_name,
                   used_extensions, used_client_extensions):
  """Generates gl_bindings_autogen_x.h"""

  # Write file header.
  file.write(LICENSE_AND_HEADER +
"""

#ifndef UI_GL_GL_BINDINGS_AUTOGEN_%(name)s_H_
#define UI_GL_GL_BINDINGS_AUTOGEN_%(name)s_H_

#include <string>

namespace gl {

class GLContext;

""" % {'name': set_name.upper()})

  # Write typedefs for function pointer types. Always use the GL name for the
  # typedef.
  file.write('\n')
  for func in functions:
    file.write('typedef %s (GL_BINDING_CALL *%sProc)(%s);\n' %
        (func['return_type'], func['known_as'], func['arguments']))

  # Write declarations for booleans indicating which extensions are available.
  file.write('\n')
  file.write("struct Extensions%s {\n" % set_name.upper())
  for extension in sorted(used_client_extensions):
    file.write('  bool b_%s;\n' % extension)
  for extension in sorted(used_extensions):
    file.write('  bool b_%s;\n' % extension)
  file.write('};\n')
  file.write('\n')

  # Write Procs struct.
  file.write("struct Procs%s {\n" % set_name.upper())
  for func in functions:
    file.write('  %sProc %sFn;\n' % (func['known_as'], func['known_as']))
  file.write('};\n')
  file.write('\n')

  # Write Api class.
  file.write(
"""class GL_EXPORT %(name)sApi {
 public:
  %(name)sApi();
  virtual ~%(name)sApi();

  virtual void SetDisabledExtensions(
      const std::string& disabled_extensions) {}

""" % {'name': set_name.upper()})
  for func in functions:
    file.write('  virtual %s %sFn(%s) = 0;\n' %
      (func['return_type'], func['known_as'], func['arguments']))
  file.write('};\n')
  file.write('\n')

  file.write( '}  // namespace gl\n')

  # Write macros to invoke function pointers. Always use the GL name for the
  # macro.
  file.write('\n')
  for func in functions:
    file.write('#define %s ::gl::g_current_%s_context->%sFn\n' %
        (func['known_as'], set_name.lower(), func['known_as']))

  file.write('\n')
  file.write('#endif  //  UI_GL_GL_BINDINGS_AUTOGEN_%s_H_\n' %
      set_name.upper())


def GenerateAPIHeader(file, functions, set_name):
  """Generates gl_bindings_api_autogen_x.h"""

  # Write file header.
  file.write(LICENSE_AND_HEADER +
"""

// Silence presubmit and Tricium warnings about include guards
// no-include-guard-because-multiply-included
// NOLINT(build/header_guard)

""")

  # Write API declaration.
  for func in functions:
    file.write('  %s %sFn(%s) override;\n' %
      (func['return_type'], func['known_as'], func['arguments']))

  file.write('\n')


def GenerateMockHeader(file, functions, set_name):
  """Generates gl_mock_autogen_x.h"""

  # Write file header.
  file.write(LICENSE_AND_HEADER +
"""

// Silence presubmit and Tricium warnings about include guards
// no-include-guard-because-multiply-included
// NOLINT(build/header_guard)

""")

  # Write API declaration.
  for func in functions:
    args = func['arguments']
    if args == 'void':
      args = ''
    arg_count = 0
    if len(args):
      arg_count = func['arguments'].count(',') + 1
    # TODO(zmo): crbug.com/456340
    # For now gmock supports at most 10 args.
    if arg_count <= 10:
      file.write('  MOCK_METHOD%d(%s, %s(%s));\n' %
          (arg_count, func['known_as'][len(set_name):], func['return_type'],
           args))
    else:
      file.write('  // TODO(zmo): crbug.com/456340\n')
      file.write('  // %s cannot be mocked because it has %d args.\n' %
          (func['known_as'], arg_count))

  file.write('\n')

def GenerateStubHeader(file, functions):
  """Generates gl_stub_autogen_gl.h"""

  # Write file header.
  file.write(LICENSE_AND_HEADER +
"""

#ifndef UI_GL_GL_STUB_AUTOGEN_H_
#define UI_GL_GL_STUB_AUTOGEN_H_

""")


  # Write API declaration.
  for func in functions:
    args = func['arguments']
    if args == 'void':
      args = ''
    return_type = func['return_type'];
    file.write('  %s gl%sFn(%s) override' % (return_type, func['known_as'][2:],
                                             args))
    if return_type == 'void':
      file.write(' {}\n');
    else:
      file.write(';\n');

  file.write('\n')
  file.write('#endif  //  UI_GL_GL_STUB_AUTOGEN_H_')

def GenerateStubSource(file, functions):
  """Generates gl_stub_autogen_gl.cc"""

  # Write file header.
  file.write(LICENSE_AND_HEADER)
  file.write('\n#include "ui/gl/gl_stub_api_base.h"\n\n')
  file.write('namespace gl {\n\n')

  # Write API declaration.
  for func in functions:
    return_type = func['return_type'];
    if return_type == 'void':
      continue
    args = func['arguments']
    if args == 'void':
      args = ''
    file.write('%s GLStubApiBase::gl%sFn(%s) {\n' % (return_type,
                                                     func['known_as'][2:],
                                                     args))
    file.write('  return 0;\n');
    file.write('}\n\n');

  file.write('\n}  // namespace gl\n')


def GenerateSource(file, functions, set_name, used_extensions,
                   used_client_extensions, options):
  """Generates gl_bindings_autogen_x.cc"""

  set_header_name = "ui/gl/gl_" + set_name.lower() + "_api_implementation.h"
  include_list = [ 'base/trace_event/trace_event.h',
                   'ui/gl/gl_enums.h',
                   'ui/gl/gl_bindings.h',
                   'ui/gl/gl_context.h',
                   'ui/gl/gl_implementation.h',
                   'ui/gl/gl_version_info.h',
                   set_header_name ]

  includes_string = "\n".join(["#include \"{0}\"".format(h)
                               for h in sorted(include_list)])

  # Write file header.
  file.write(LICENSE_AND_HEADER +
"""

#include <string>

%s

namespace gl {
""" % includes_string)

  file.write('\n')
  if set_name != 'gl':
    file.write('Driver%s g_driver_%s;  // Exists in .bss\n' % (
        set_name.upper(), set_name.lower()))
  file.write('\n')

  # Write stub functions that take the place of some functions before a context
  # is initialized. This is done to provide clear asserts on debug build and to
  # avoid crashing in case of a bug on release build.
  file.write('\n')
  num_dynamic = 0
  for func in functions:
    static_binding = GetStaticBinding(func)
    if static_binding:
      func['static_binding'] = static_binding
    else:
      num_dynamic = num_dynamic + 1

  print "[%s] %d static bindings, %d dynamic bindings" % (
      set_name, len(functions) - num_dynamic, num_dynamic)

  # Write function to initialize the function pointers that are always the same
  # and to initialize bindings where choice of the function depends on the
  # extension string or the GL version to point to stub functions.
  file.write('\n')
  file.write('void Driver%s::InitializeStaticBindings() {\n' %
             set_name.upper())
  file.write('  // Ensure struct has been zero-initialized.\n')
  file.write('  char* this_bytes = reinterpret_cast<char*>(this);\n')
  file.write('  DCHECK(this_bytes[0] == 0);\n');
  file.write('  DCHECK('
             'memcmp(this_bytes, this_bytes + 1, sizeof(*this) - 1) == 0);\n');
  file.write('\n')

  def WriteFuncBinding(file, known_as, version_name):
    file.write(
        '  fn.%sFn = reinterpret_cast<%sProc>(GetGLProcAddress("%s"));\n' %
        (known_as, known_as, version_name))

  for func in functions:
    if 'static_binding' in func:
      WriteFuncBinding(file, func['known_as'], func['static_binding'])

  def GetGLVersionCondition(gl_version):
    if GLVersionBindAlways(gl_version):
      if gl_version.is_es:
        return 'ver->is_es'
      else:
        return '!ver->is_es'
    elif gl_version.is_es:
      return 'ver->IsAtLeastGLES(%du, %du)' % (
          gl_version.major_version, gl_version.minor_version)
    else:
      return 'ver->IsAtLeastGL(%du, %du)' % (
          gl_version.major_version, gl_version.minor_version)

  def GetBindingCondition(version):
    conditions = []
    if 'gl_versions' in version:
      conditions.extend(
          [GetGLVersionCondition(v) for v in version['gl_versions']])
    if 'extensions' in version and version['extensions']:
      conditions.extend(
          ['ext.b_%s' % e for e in version['extensions']])
    return ' || '.join(conditions)

  def WriteConditionalFuncBinding(file, func):
    assert len(func['versions']) > 0
    known_as = func['known_as']
    i = 0
    first_version = True
    while i < len(func['versions']):
      version = func['versions'][i]
      cond = GetBindingCondition(version)
      if first_version:
        file.write('  if (%s) {\n  ' % cond)
      else:
        file.write('  else if (%s) {\n  ' % (cond))

      WriteFuncBinding(file, known_as, version['name'])
      if options.validate_bindings:
        if not 'is_optional' in func or not func['is_optional']:
          file.write('DCHECK(fn.%sFn);\n' % known_as)
      file.write('}\n')
      i += 1
      first_version = False

  # TODO(jmadill): make more robust
  def IsClientExtensionFunc(func):
    assert len(func['versions']) > 0
    if 'client_extensions' in func['versions'][0]:
      assert len(func['versions']) == 1
      return True
    return False

  file.write("}\n\n");

  if set_name == 'gl':
    file.write("""\
void DriverGL::InitializeDynamicBindings(const GLVersionInfo* ver,
                                         const gfx::ExtensionSet& extensions) {
""")
  elif set_name == 'egl':
    file.write("""\
void DriverEGL::InitializeClientExtensionBindings() {
  std::string client_extensions(GetClientExtensions());
  gfx::ExtensionSet extensions(gfx::MakeExtensionSet(client_extensions));
  ALLOW_UNUSED_LOCAL(extensions);

""")
  else:
    file.write("""\
void Driver%s::InitializeExtensionBindings() {
  std::string platform_extensions(GetPlatformExtensions());
  gfx::ExtensionSet extensions(gfx::MakeExtensionSet(platform_extensions));
  ALLOW_UNUSED_LOCAL(extensions);

""" % (set_name.upper(),))

  def OutputExtensionBindings(extension_var, extensions, extension_funcs):
    # Extra space at the end of the extension name is intentional,
    # it is used as a separator
    for extension in extensions:
      file.write('  ext.b_%s = gfx::HasExtension(%s, "%s");\n' %
                 (extension, extension_var, extension))

    for func in extension_funcs:
      if not 'static_binding' in func:
        file.write('\n')
        WriteConditionalFuncBinding(file, func)

  OutputExtensionBindings(
    'extensions',
    sorted(used_client_extensions),
    [ f for f in functions if IsClientExtensionFunc(f) ])

  if set_name == 'egl':
    file.write("""\
}

void DriverEGL::InitializeExtensionBindings() {
  std::string platform_extensions(GetPlatformExtensions());
  gfx::ExtensionSet extensions(gfx::MakeExtensionSet(platform_extensions));
  ALLOW_UNUSED_LOCAL(extensions);

""")

  OutputExtensionBindings(
    'extensions',
    sorted(used_extensions),
    [ f for f in functions if not IsClientExtensionFunc(f) ])
  file.write('}\n')

  # Write function to clear all function pointers.
  file.write('\n')
  file.write("""void Driver%s::ClearBindings() {
  memset(this, 0, sizeof(*this));
}
""" % set_name.upper())

  def MakeArgNames(arguments):
    argument_names = re.sub(
        r'(const )?[a-zA-Z0-9_]+\** ([a-zA-Z0-9_]+)', r'\2', arguments)
    argument_names = re.sub(
        r'(const )?[a-zA-Z0-9_]+\** ([a-zA-Z0-9_]+)', r'\2', argument_names)
    if argument_names == 'void' or argument_names == '':
      argument_names = ''
    return argument_names

  # Write GLApiBase functions
  for func in functions:
    function_name = func['known_as']
    return_type = func['return_type']
    arguments = func['arguments']
    file.write('\n')
    file.write('%s %sApiBase::%sFn(%s) {\n' %
        (return_type, set_name.upper(), function_name, arguments))
    argument_names = MakeArgNames(arguments)
    if return_type == 'void':
      file.write('  driver_->fn.%sFn(%s);\n' %
          (function_name, argument_names))
    else:
      file.write('  return driver_->fn.%sFn(%s);\n' %
          (function_name, argument_names))
    file.write('}\n')

  # Write TraceGLApi functions
  for func in functions:
    function_name = func['known_as']
    return_type = func['return_type']
    arguments = func['arguments']
    file.write('\n')
    file.write('%s Trace%sApi::%sFn(%s) {\n' %
        (return_type, set_name.upper(), function_name, arguments))
    argument_names = MakeArgNames(arguments)
    file.write('  TRACE_EVENT_BINARY_EFFICIENT0("gpu", "Trace%sAPI::%s");\n' %
               (set_name.upper(), function_name))
    if return_type == 'void':
      file.write('  %s_api_->%sFn(%s);\n' %
          (set_name.lower(), function_name, argument_names))
    else:
      file.write('  return %s_api_->%sFn(%s);\n' %
          (set_name.lower(), function_name, argument_names))
    file.write('}\n')

  # Write LogGLApi functions
  for func in functions:
    return_type = func['return_type']
    arguments = func['arguments']
    file.write('\n')
    file.write('%s Log%sApi::%sFn(%s) {\n' %
        (return_type, set_name.upper(), func['known_as'], arguments))
    # Strip pointer types.
    argument_names = re.sub(
        r'(const )?[a-zA-Z0-9_]+\** ([a-zA-Z0-9_]+)', r'\2', arguments)
    argument_names = re.sub(
        r'(const )?[a-zA-Z0-9_]+\** ([a-zA-Z0-9_]+)', r'\2', argument_names)
    # Replace certain `Type varname` combinations with TYPE_varname.
    log_argument_names = re.sub(
        r'const char\* ([a-zA-Z0-9_]+)', r'CONSTCHAR_\1', arguments)
    log_argument_names = re.sub(
        r'(const )?[a-zA-Z0-9_]+\* ([a-zA-Z0-9_]+)',
        r'CONSTVOID_\2', log_argument_names)
    log_argument_names = re.sub(
        r'(const )?EGL[GS]etBlobFuncANDROID ([a-zA-Z0-9_]+)',
        r'FUNCPTR_\2', log_argument_names)
    log_argument_names = re.sub(
        r'(?<!E)GLboolean ([a-zA-Z0-9_]+)', r'GLboolean_\1', log_argument_names)
    log_argument_names = re.sub(
        r'GLDEBUGPROC ([a-zA-Z0-9_]+)',
        r'GLDEBUGPROC_\1', log_argument_names)
    log_argument_names = re.sub(
        r'EGLDEBUGPROCKHR ([a-zA-Z0-9_]+)',
        r'EGLDEBUGPROCKHR_\1', log_argument_names)
    log_argument_names = re.sub(
        r'(?<!E)GLenum ([a-zA-Z0-9_]+)', r'GLenum_\1', log_argument_names)
    # Strip remaining types.
    log_argument_names = re.sub(
        r'(const )?[a-zA-Z0-9_]+\** ([a-zA-Z0-9_]+)', r'\2',
        log_argument_names)
    # One more round of stripping to remove both type parts in `unsigned long`.
    log_argument_names = re.sub(
        r'(const )?[a-zA-Z0-9_]+\** ([a-zA-Z0-9_]+)', r'\2',
        log_argument_names)
    # Convert TYPE_varname log arguments to the corresponding log expression.
    log_argument_names = re.sub(
        r'CONSTVOID_([a-zA-Z0-9_]+)',
        r'static_cast<const void*>(\1)', log_argument_names)
    log_argument_names = re.sub(
        r'FUNCPTR_([a-zA-Z0-9_]+)',
        r'reinterpret_cast<const void*>(\1)', log_argument_names)
    log_argument_names = re.sub(
        r'CONSTCHAR_([a-zA-Z0-9_]+)', r'\1', log_argument_names)
    log_argument_names = re.sub(
        r'GLboolean_([a-zA-Z0-9_]+)', r'GLEnums::GetStringBool(\1)',
        log_argument_names)
    log_argument_names = re.sub(
        r'GLDEBUGPROC_([a-zA-Z0-9_]+)',
        r'reinterpret_cast<void*>(\1)', log_argument_names)
    log_argument_names = re.sub(
        r'EGLDEBUGPROCKHR_([a-zA-Z0-9_]+)',
        r'reinterpret_cast<void*>(\1)', log_argument_names)
    log_argument_names = re.sub(
        r'GLenum_([a-zA-Z0-9_]+)', r'GLEnums::GetStringEnum(\1)',
        log_argument_names)
    log_argument_names = log_argument_names.replace(',', ' << ", " <<')
    if argument_names == 'void' or argument_names == '':
      argument_names = ''
      log_argument_names = ''
    else:
      log_argument_names = " << " + log_argument_names
    function_name = func['known_as']
    if return_type == 'void':
      file.write('  GL_SERVICE_LOG("%s" << "(" %s << ")");\n' %
          (function_name, log_argument_names))
      file.write('  %s_api_->%sFn(%s);\n' %
          (set_name.lower(), function_name, argument_names))
      if 'logging_code' in func:
        file.write("%s\n" % func['logging_code'])
      if options.generate_dchecks and set_name == 'gl':
        file.write('  {\n')
        file.write('    GLenum error = %s_api_->glGetErrorFn();\n'
            % set_name.lower())
        file.write('    DCHECK(error == 0) << "OpenGL error 0x" << std::hex '
                   '<< error << std::dec;\n')
        file.write('  }\n')
    else:
      file.write('  GL_SERVICE_LOG("%s" << "(" %s << ")");\n' %
          (function_name, log_argument_names))
      file.write('  %s result = %s_api_->%sFn(%s);\n' %
          (return_type, set_name.lower(), function_name, argument_names))
      if 'logging_code' in func:
        file.write("%s\n" % func['logging_code'])
      else:
        file.write('  GL_SERVICE_LOG("GL_RESULT: " << result);\n')
      if options.generate_dchecks and set_name == 'gl':
        file.write('  {\n')
        file.write('    GLenum _error = %s_api_->glGetErrorFn();\n' %
            set_name.lower())
        file.write('    DCHECK(_error == 0) << "OpenGL error " << std::hex '
                   '<< _error << std::dec;\n')
        file.write('  }\n')
      file.write('  return result;\n')
    file.write('}\n')

  # Write NoContextGLApi functions
  if set_name.upper() == "GL":
    file.write('\n')
    file.write('namespace {\n')
    file.write('void NoContextHelper(const char* method_name) {\n')
    no_context_error = ('<< "Trying to call " << method_name << " without '
                        'current GL context"')
    file.write('  NOTREACHED() %s;\n' % no_context_error)
    file.write('  LOG(ERROR) %s;\n' % no_context_error)
    file.write('}\n')
    file.write('}  // namespace\n')
    for func in functions:
      function_name = func['known_as']
      return_type = func['return_type']
      arguments = func['arguments']
      file.write('\n')
      file.write('%s NoContextGLApi::%sFn(%s) {\n' %
          (return_type, function_name, arguments))
      argument_names = MakeArgNames(arguments)
      file.write('  NoContextHelper("%s");\n' % function_name)
      default_value = { 'GLenum': 'static_cast<GLenum>(0)',
                        'GLuint': '0U',
                        'GLint': '0',
                        'GLboolean': 'GL_FALSE',
                        'GLbyte': '0',
                        'GLubyte': '0',
                        'GLbutfield': '0',
                        'GLushort': '0',
                        'GLsizei': '0',
                        'GLfloat': '0.0f',
                        'GLdouble': '0.0',
                        'GLsync': 'nullptr',
                        'GLDEBUGPROC': 'NULL'}
      if return_type.endswith('*'):
        file.write('  return NULL;\n')
      elif return_type != 'void':
        file.write('  return %s;\n' % default_value[return_type])
      file.write('}\n')

  file.write('\n')
  file.write('}  // namespace gl\n')


def GetUniquelyNamedFunctions(functions):
  uniquely_named_functions = {}

  for func in functions:
    for version in func['versions']:
      uniquely_named_functions[version['name']] = ({
        'name': version['name'],
        'return_type': func['return_type'],
        'arguments': func['arguments'],
        'known_as': func['known_as']
      })
  return uniquely_named_functions


def GenerateMockBindingsHeader(file, functions):
  """Headers for functions that invoke MockGLInterface members"""

  file.write(LICENSE_AND_HEADER +
"""

// Silence presubmit and Tricium warnings about include guards
// no-include-guard-because-multiply-included
// NOLINT(build/header_guard)

""")
  uniquely_named_functions = GetUniquelyNamedFunctions(functions)

  for key in sorted(uniquely_named_functions.iterkeys()):
    func = uniquely_named_functions[key]
    file.write('static %s GL_BINDING_CALL Mock_%s(%s);\n' %
        (func['return_type'], func['name'], func['arguments']))


def GenerateMockBindingsSource(file, functions, set_name):
  """Generates functions that invoke MockGLInterface members and a
  GetGLProcAddress function that returns addresses to those functions."""

  file.write(LICENSE_AND_HEADER +
"""

#include <string.h>

#include "base/notreached.h"
#include "ui/gl/%s_mock.h"

namespace {
// This is called mainly to prevent the compiler combining the code of mock
// functions with identical contents, so that their function pointers will be
// different.
void Make%sMockFunctionUnique(const char *func_name) {
    VLOG(2) << "Calling mock " << func_name;
}
}  // namespace

namespace gl {
""" % (set_name, set_name.capitalize()))

  # Write functions that trampoline into the set MockGLInterface instance.
  uniquely_named_functions = GetUniquelyNamedFunctions(functions)
  sorted_function_names = sorted(uniquely_named_functions.iterkeys())

  for key in sorted_function_names:
    func = uniquely_named_functions[key]
    file.write('\n')
    file.write('%s GL_BINDING_CALL Mock%sInterface::Mock_%s(%s) {\n' %
        (func['return_type'], set_name.upper(), func['name'],
         func['arguments']))
    file.write('  Make%sMockFunctionUnique("%s");\n' % (set_name.capitalize(),
                                                        func['name']))
    arg_re = r'(const |struct )*[a-zA-Z0-9]+((\s*const\s*)?\*)* ([a-zA-Z0-9]+)'
    argument_names = re.sub(arg_re, r'\4', func['arguments'])
    if argument_names == 'void':
      argument_names = ''
    function_name = func['known_as'][len(set_name):]
    if func['return_type'] == 'void':
      file.write('  interface_->%s(%s);\n' %
          (function_name, argument_names))
    else:
      file.write('  return interface_->%s(%s);\n' %
          (function_name, argument_names))
    file.write('}\n')

  # Write an 'invalid' function to catch code calling through uninitialized
  # function pointers or trying to interpret the return value of
  # GLProcAddress().
  file.write('\n')
  file.write('static void Mock%sInvalidFunction() {\n' % set_name.capitalize())
  file.write('  NOTREACHED();\n')
  file.write('}\n')

  # Write a function to lookup a mock GL function based on its name.
  file.write('\n')
  file.write('GLFunctionPointerType GL_BINDING_CALL ' +
             'Mock%sInterface::GetGLProcAddress(const char* name) {\n' % (
                 set_name.upper(),))
  for key in sorted_function_names:
    name = uniquely_named_functions[key]['name']
    file.write('  if (strcmp(name, "%s") == 0)\n' % name)
    file.write(
        '    return reinterpret_cast<GLFunctionPointerType>(Mock_%s);\n' %
            name)
  # Always return a non-NULL pointer like some EGL implementations do.
  file.write('  return reinterpret_cast<GLFunctionPointerType>('
             '&Mock%sInvalidFunction);\n' % set_name.capitalize())
  file.write('}\n')

  file.write('\n')
  file.write('}  // namespace gl\n')

def SamePrefix(a, b):
  return a[:a.rfind("_")] == b[:b.rfind("_")]

def GenerateEnumUtils(out_file, input_filenames):
  enum_re = re.compile(r'\#define\s+(GL_[a-zA-Z0-9_]+)\s+([0-9A-Fa-fx]+)')
  dict = {}
  for fname in input_filenames:
    lines = open(fname).readlines()
    for line in lines:
      m = enum_re.match(line)
      if m:
        name = m.group(1)
        value = m.group(2)
        if len(value) <= 10 and value.startswith('0x'):
          if not value in dict:
            dict[value] = name
          # check our own _CHROMIUM macro conflicts with khronos GL headers.
          # we allow for name duplication if they have the same prefix.
          elif dict[value] != name and ((name.endswith('_CHROMIUM') or
              dict[value].endswith('_CHROMIUM')) and
              not SamePrefix(name, dict[value])):
            raise RuntimeError("code collision: %s and %s have the same code %s"
                               %  (dict[value], name, value))

  out_file.write(LICENSE_AND_HEADER +
"""

#ifndef UI_GL_GL_ENUMS_IMPLEMENTATION_AUTOGEN_H_
#define UI_GL_GL_ENUMS_IMPLEMENTATION_AUTOGEN_H_

""")
  out_file.write("static const GLEnums::EnumToString "
                 "enum_to_string_table[] = {\n")
  for value in sorted(dict):
    out_file.write('  { %s, "%s", },\n' % (value, dict[value]))
  out_file.write("""};

const GLEnums::EnumToString* const GLEnums::enum_to_string_table_ =
  enum_to_string_table;
const size_t GLEnums::enum_to_string_table_len_ =
  sizeof(enum_to_string_table) / sizeof(enum_to_string_table[0]);

""")
  out_file.write('#endif  //  UI_GL_GL_ENUMS_IMPLEMENTATION_AUTOGEN_H_')


def ParseFunctionsFromHeader(header_file, extensions, versions):
  """Parse a C extension header file and return a map from extension names to
  a list of functions.

  Args:
    header_file: Line-iterable C header file.
  Returns:
    Map of extension name => functions, Map of gl version => functions.
    Functions will only be in either one of the two maps.
  """
  version_start = re.compile(
      r'#ifndef GL_(ES_|)VERSION((?:_[0-9])+)$')
  extension_start = re.compile(
      r'#ifndef ((?:GL|EGL|WGL|GLX)_[A-Z]+_[a-zA-Z]\w+)')
  extension_function = re.compile(r'.+\s+([a-z]+\w+)\s*\(')
  typedef = re.compile(r'typedef .*')
  macro_start = re.compile(r'^#(if|ifdef|ifndef).*')
  macro_end = re.compile(r'^#endif.*')
  macro_depth = 0
  current_version = None
  current_version_depth = 0
  current_extension = None
  current_extension_depth = 0

  # Pick up all core functions here, since some of them are missing in the
  # Khronos headers.
  hdr = os.path.basename(header_file.name)
  if hdr == "gl.h":
    current_version = GLVersion(False, 1, 0)

  line_num = 1
  for line in header_file:
    version_match = version_start.match(line)
    if macro_start.match(line):
      macro_depth += 1
      if version_match:
        if current_version:
          raise RuntimeError('Nested GL version macro in %s at line %d' % (
              header_file.name, line_num))
        current_version_depth = macro_depth
        es = version_match.group(1)
        major_version, minor_version =\
            version_match.group(2).lstrip('_').split('_')
        is_es = len(es) > 0
        if (not is_es) and (major_version == '1'):
          minor_version = 0
        current_version = GLVersion(
            is_es, int(major_version), int(minor_version))
    elif macro_end.match(line):
      macro_depth -= 1
      if macro_depth < current_extension_depth:
        current_extension = None
      if macro_depth < current_version_depth:
        current_version = None

    match = extension_start.match(line)
    if match and not version_match:
      if current_version and hdr != "gl.h":
        raise RuntimeError('Nested GL version macro in %s at line %d' % (
            header_file.name, line_num))
      current_extension = match.group(1)
      current_extension_depth = macro_depth

    match = extension_function.match(line)
    if match and not typedef.match(line):
      if current_extension:
        extensions[current_extension].add(match.group(1))
      elif current_version:
        versions[current_version].add(match.group(1))
    line_num = line_num + 1


def GetDynamicFunctions(extension_headers):
  """Parse all optional functions from a list of header files.

  Args:
    extension_headers: List of header file names.
  Returns:
    Map of extension name => list of functions,
    Map of gl version => list of functions.
  """
  extensions = collections.defaultdict(lambda: set([]))
  gl_versions = collections.defaultdict(lambda: set([]))
  for header in extension_headers:
    ParseFunctionsFromHeader(open(header), extensions, gl_versions)
  return extensions, gl_versions


def GetFunctionToExtensionsMap(extensions):
  """Construct map from a function names to extensions which define the
  function.

  Args:
    extensions: Map of extension name => functions.
  Returns:
    Map of function name => extension names.
  """
  function_to_extensions = {}
  for extension, functions in extensions.items():
    for function in functions:
      if not function in function_to_extensions:
        function_to_extensions[function] = set([])
      function_to_extensions[function].add(extension)
  return function_to_extensions

def GetFunctionToGLVersionsMap(gl_versions):
  """Construct map from a function names to GL versions which define the
  function.

  Args:
    extensions: Map of gl versions => functions.
  Returns:
    Map of function name => gl versions.
  """
  function_to_gl_versions = {}
  for gl_version, functions in gl_versions.items():
    for function in functions:
      if not function in function_to_gl_versions:
        function_to_gl_versions[function] = set([])
      function_to_gl_versions[function].add(gl_version)
  return function_to_gl_versions


def LooksLikeExtensionFunction(function):
  """Heuristic to see if a function name is consistent with extension function
  naming."""
  vendor = re.match(r'\w+?([A-Z][A-Z]+)$', function)
  return vendor is not None and not vendor.group(1) in ['GL', 'API', 'DC']


def SortVersions(key):
   # Prefer functions from the core for binding
  if 'gl_versions' in key:
    return 0
  else:
    return 1

def FillExtensionsFromHeaders(functions, extension_headers, extra_extensions):
  """Determine which functions belong to extensions based on extension headers,
  and fill in this information to the functions table for functions that don't
  already have the information.

  Args:
    functions: List of (return type, function versions, arguments).
    extension_headers: List of header file names.
    extra_extensions: Extensions to add to the list.
  Returns:
    Set of used extensions.
  """
  # Parse known extensions.
  extensions, gl_versions = GetDynamicFunctions(extension_headers)
  functions_to_extensions = GetFunctionToExtensionsMap(extensions)
  functions_to_gl_versions = GetFunctionToGLVersionsMap(gl_versions)

  # Fill in the extension information.
  used_extensions = set()
  used_client_extensions = set()
  used_functions_by_version = collections.defaultdict(lambda: set([]))
  for func in functions:
    for version in func['versions']:
      name = version['name']

      # There should only be one version entry per name string.
      if len([v for v in func['versions'] if v['name'] == name]) > 1:
        raise RuntimeError(
            'Duplicate version entries with same name for %s' % name)

      # Make sure we know about all extensions and extension functions.
      extensions_from_headers = set([])
      if name in functions_to_extensions:
        extensions_from_headers = set(functions_to_extensions[name])

      explicit_extensions = set([])

      if 'client_extensions' in version:
        assert not 'extensions' in version
        version['extensions'] = version['client_extensions']

      if 'extensions' in version:
        explicit_extensions = set(version['extensions'])

      in_both = explicit_extensions.intersection(extensions_from_headers)
      if len(in_both):
        print "[%s] Specified redundant extensions for binding: %s" % (
            name, ', '.join(in_both))
      diff = explicit_extensions - extensions_from_headers
      if len(diff):
        print "[%s] Specified extra extensions for binding: %s" % (
            name, ', '.join(diff))

      if version.get('explicit_only', False):
        all_extensions = explicit_extensions
      else:
        all_extensions = extensions_from_headers.union(explicit_extensions)
      if len(all_extensions):
        version['extensions'] = all_extensions

      if 'extensions' in version:
        assert len(version['extensions'])
        if 'client_extensions' in version:
          used_client_extensions.update(version['extensions'])
        else:
          used_extensions.update(version['extensions'])

      if not 'extensions' in version and LooksLikeExtensionFunction(name):
        raise RuntimeError('%s looks like an extension function but does not '
            'belong to any of the known extensions.' % name)

      if name in functions_to_gl_versions:
        assert not 'gl_versions' in version
        version['gl_versions'] = functions_to_gl_versions[name]
        for v in version['gl_versions']:
          used_functions_by_version[v].add(name)

    func['versions'] = sorted(func['versions'], key=SortVersions)

  # Add extensions that do not have any functions.
  used_extensions.update(extra_extensions)

  # Print out used function count by GL(ES) version.
  for v in sorted([v for v in used_functions_by_version if v.is_es]):
    print "OpenGL ES %d.%d: %d used functions" % (
        v.major_version, v.minor_version, len(used_functions_by_version[v]))
  for v in sorted([v for v in used_functions_by_version if not v.is_es]):
    print "OpenGL %d.%d: %d used functions" % (
        v.major_version, v.minor_version, len(used_functions_by_version[v]))

  return used_extensions, used_client_extensions


def ResolveHeader(header, header_paths):
  for path in header_paths:
    result = os.path.join(path, header)
    if not os.path.isabs(path):
      result = os.path.abspath(os.path.join(SELF_LOCATION, result))
    if os.path.exists(result):
      # Always use forward slashes as path separators. Otherwise backslashes
      # may be incorrectly interpreted as escape characters.
      return result.replace(os.path.sep, '/')

  raise Exception('Header %s not found.' % header)


def main(argv):
  """This is the main function."""

  parser = optparse.OptionParser()
  parser.add_option('--inputs', action='store_true')
  parser.add_option('--verify-order', action='store_true')
  parser.add_option('--generate-dchecks', action='store_true',
                    help='Generates DCHECKs into the logging functions '
                         'asserting no GL errors (useful for debugging with '
                         '--enable-gpu-service-logging)')
  parser.add_option('--validate-bindings', action='store_true',
                    help='Generate DCHECKs to validate function bindings '
                         ' were correctly supplied (useful for debugging)')

  options, args = parser.parse_args(argv)

  if options.inputs:
    for [_, _, headers, _] in FUNCTION_SETS:
      for header in headers:
        print ResolveHeader(header, HEADER_PATHS)
    return 0

  directory = SELF_LOCATION
  if len(args) >= 1:
    directory = args[0]

  def ClangFormat(filename):
    formatter = "clang-format"
    if platform.system() == "Windows":
      formatter += ".bat"
    call([formatter, "-i", "-style=chromium", filename])

  for [functions, set_name, extension_headers, extensions] in FUNCTION_SETS:
    # Function names can be specified in two ways (list of unique names or list
    # of versions with different binding conditions). Fill in the data to the
    # versions list in case it is missing, so that can be used from here on:
    for func in functions:
      assert 'versions' in func or 'names' in func, 'Function with no names'
      if 'versions' not in func:
        func['versions'] = [{'name': n} for n in func['names']]
      # Use the first version's name unless otherwise specified
      if 'known_as' not in func:
        func['known_as'] = func['versions'][0]['name']
      # Make sure that 'names' is not accidentally used instead of 'versions'
      if 'names' in func:
        del func['names']

    # Check function names in each set is sorted in alphabetical order.
    for index in range(len(functions) - 1):
      func_name = functions[index]['known_as']
      next_func_name = functions[index + 1]['known_as']
      if func_name.lower() > next_func_name.lower():
        raise Exception(
            'function %s is not in alphabetical order' % next_func_name)
    if options.verify_order:
      continue

    extension_headers = [ResolveHeader(h, HEADER_PATHS)
                         for h in extension_headers]
    used_extensions, used_client_extensions = FillExtensionsFromHeaders(
        functions, extension_headers, extensions)

    header_file = open(
        os.path.join(directory, 'gl_bindings_autogen_%s.h' % set_name), 'wb')
    GenerateHeader(header_file, functions, set_name,
                   used_extensions, used_client_extensions)
    header_file.close()
    ClangFormat(header_file.name)

    header_file = open(
        os.path.join(directory, 'gl_bindings_api_autogen_%s.h' % set_name),
        'wb')
    GenerateAPIHeader(header_file, functions, set_name)
    header_file.close()
    ClangFormat(header_file.name)

    source_file = open(
        os.path.join(directory, 'gl_bindings_autogen_%s.cc' % set_name), 'wb')
    GenerateSource(source_file, functions, set_name,
                   used_extensions, used_client_extensions, options)
    source_file.close()
    ClangFormat(source_file.name)

  if not options.verify_order:
    header_file = open(
        os.path.join(directory, 'gl_mock_autogen_gl.h'), 'wb')
    GenerateMockHeader(header_file, GL_FUNCTIONS, 'gl')
    header_file.close()
    ClangFormat(header_file.name)

    header_file = open(os.path.join(directory, 'gl_bindings_autogen_mock.h'),
                       'wb')
    GenerateMockBindingsHeader(header_file, GL_FUNCTIONS)
    header_file.close()
    ClangFormat(header_file.name)

    source_file = open(os.path.join(directory, 'gl_bindings_autogen_mock.cc'),
                       'wb')
    GenerateMockBindingsSource(source_file, GL_FUNCTIONS, 'gl')
    source_file.close()
    ClangFormat(source_file.name)

    header_file = open(
        os.path.join(directory, 'gl_mock_autogen_egl.h'), 'wb')
    GenerateMockHeader(header_file, EGL_FUNCTIONS, 'egl')
    header_file.close()
    ClangFormat(header_file.name)

    header_file = open(os.path.join(directory, 'egl_bindings_autogen_mock.h'),
                       'wb')
    GenerateMockBindingsHeader(header_file, EGL_FUNCTIONS)
    header_file.close()
    ClangFormat(header_file.name)

    source_file = open(os.path.join(directory, 'egl_bindings_autogen_mock.cc'),
                       'wb')
    GenerateMockBindingsSource(source_file, EGL_FUNCTIONS, 'egl')
    source_file.close()
    ClangFormat(source_file.name)

    enum_header_filenames = [ResolveHeader(h, HEADER_PATHS)
                             for h in GLES2_HEADERS_WITH_ENUMS]
    header_file = open(os.path.join(directory,
                                    'gl_enums_implementation_autogen.h'),
                       'wb')
    GenerateEnumUtils(header_file, enum_header_filenames)
    header_file.close()
    ClangFormat(header_file.name)

    header_file = open(
        os.path.join(directory, 'gl_stub_autogen_gl.h'), 'wb')
    GenerateStubHeader(header_file, GL_FUNCTIONS)
    header_file.close()
    ClangFormat(header_file.name)

    header_file = open(
        os.path.join(directory, 'gl_stub_autogen_gl.cc'), 'wb')
    GenerateStubSource(header_file, GL_FUNCTIONS)
    header_file.close()
    ClangFormat(header_file.name)
  return 0


if __name__ == '__main__':
  sys.exit(main(sys.argv[1:]))
