// Copyright 2022 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef UI_ACCESSIBILITY_PLATFORM_FUCHSIA_SEMANTIC_PROVIDER_H_
#define UI_ACCESSIBILITY_PLATFORM_FUCHSIA_SEMANTIC_PROVIDER_H_

#include <fuchsia/accessibility/semantics/cpp/fidl.h>

#include "ui/accessibility/ax_export.h"

namespace ui {

// Manages the connection with the Fuchsia Semantics API.
//
// Semantic nodes can be added or deleted. When a batch of nodes would leave the
// Fuchsia semantic tree in a valid state, they are committed. Please see
// |fuchsia.accessibility.semantics| API for more documentation on valid
// semantic trees.
class AX_EXPORT AXFuchsiaSemanticProvider {
 public:
  // Fuchsia root node id.
  static constexpr uint32_t kFuchsiaRootNodeId = 0u;

  // A delegate that can be registered by clients of this library to be notified
  // about Semantic changes.
  class Delegate {
   public:
    Delegate();
    virtual ~Delegate();

    using HitTestCallback =
        fuchsia::accessibility::semantics::SemanticListener::HitTestCallback;

    // Called when the FIDL channel to the Semantics Manager is closed. If this
    // callback returns true, an attempt to reconnect will be made.
    virtual bool OnSemanticsManagerConnectionClosed(zx_status_t status) = 0;

    // Processes an incoming accessibility action from Fuchsia. It
    // receives the Fuchsia node ID and the action requested. If this
    // method returns true, this means that the action will be handled.
    virtual bool OnAccessibilityAction(
        uint32_t node_id,
        fuchsia::accessibility::semantics::Action action) = 0;

    // Processes an incoming hit test request from Fuchsia. It
    // receives a point in Scenic View pixel coordinates and a callback to
    // return the result when the hit test is done. Please see
    // |fuchsia.accessibility.semantics.SemanticListener| for documentation on
    // hit tests.
    virtual void OnHitTest(fuchsia::math::PointF point,
                           HitTestCallback callback) = 0;

    // Called whenever Fuchsia enables / disables semantic updates.
    virtual void OnSemanticsEnabled(bool enabled) = 0;
  };

  AXFuchsiaSemanticProvider() = default;
  virtual ~AXFuchsiaSemanticProvider() = default;

  // Adds a semantic node to be updated. It is mandatory that the node has at
  // least an unique ID.
  virtual bool Update(fuchsia::accessibility::semantics::Node node) = 0;

  // Marks a semantic node to be deleted. Returns false if the node is not
  // present in the list of semantic nodes known by this provider.
  virtual bool Delete(uint32_t node_id) = 0;

  // Clears the semantic tree.
  virtual bool Clear() = 0;

  // Sends an accessibility event to Fuchsia. Please consult
  // https://cs.opensource.google/fuchsia/fuchsia/+/master:sdk/fidl/fuchsia.accessibility.semantics/semantics_manager.fidl
  // for documentation on events.
  virtual void SendEvent(
      fuchsia::accessibility::semantics::SemanticEvent event) = 0;

  // Returns true if there are pending updates or deletions to be made.
  virtual bool HasPendingUpdates() const = 0;

  // TODO(abrusher): Push updates to the semantic provider, rather than polling.
  // Returns the pixel scale.
  virtual float GetPixelScale() const = 0;
};

}  // namespace ui

#endif  // UI_ACCESSIBILITY_PLATFORM_FUCHSIA_SEMANTIC_PROVIDER_H_
